/**
	\file
	\brief		Header file for file and folder manipulation from a plugin
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_FILE_MANIP__
#define __HDI_CORE_FILE_MANIP__

#include <vector>

#include "hdicoreTypes.h"
	
namespace hdi
{
	namespace core
	{
		/**
			\brief	Organizes all file-/folder-related functionality into one namespace for convenience
		*/
		namespace files
		{
			typedef std::vector<std::string> StringVector;
		
			/**
				\brief		Checks for the existence of a file or folder
				\author		GW
				\date		08/2013

				\param		filePath_	Path for the file whose existence is in question, as UTF-8
				\returns	true for existent file, false otherwise
			*/
			bool exists(const std::string& filePath_);
		
			#if defined(HDI_CORE_AIP_MODE)
				/**
					\brief		Gets whether the given file is stored on a network drive
					\author		GW
					\date		07/2017

					\param		filePath_	Absolute path to the file in question
					\returns	true if the file is on a network drive, false otherwise
				*/
				bool onNetwork(const std::string& filePath_);
			#endif

			/**
				\brief		Copies a source file to a destination path
				\author		GW
				\date		08/2013

				\param		filePath_	Source file to copy, as UTF-8
				\param		destPath_	Destination file, as UTF-8, into which the source file will be written
				\returns	true for success, false for failure

				\note		The copy is performed in a binary fashion, and is safe for both text-based and binary files
				\note		The destination file will be overwritten if it already exists
			*/
			bool copy(const std::string& filePath_, const std::string& destPath_);

			/**
				\brief		Renames a file
				\author		GW
				\date		08/2013
				
				\param		filePath_	Absolute path to the file that should be renamed, as UTF-8
				\param		newName_	New name for the file, as UTF-8
				\returns	true for success, false for failure

				\note		This function cannot be used to move a file; use move() instead
			*/
			bool rename(const std::string& filePath_, const std::string& newName_);

			/**
				\brief		Moves a file from one location to another
				\author		GW
				\date		08/2013
				
				\param		fromFilePath_	Absolute path to the file that should be moved, as UTF-8
				\param		toFolderPath_	Absolute path to the destination folder, as UTF-8
				\returns	true for success, false for failure

				\note		This function cannot be used to rename a file; use rename() instead
			*/
			bool move(const std::string& fromFilePath_, const std::string& toFolderPath_);

			/**
				\brief		Returns the size of a file in bytes
				\author		GW
				\date		08/2013
				
				\param		filePath_	Path for the file whose size is in question, as UTF-8
				\returns	Size of the file in bytes, or -1 for error
			*/
			int64_t getSize(const std::string& filePath_);
			
			/**
				\brief		Gets the contents of a text-based file, assumed to be in UTF-8 encoding
				\author		GW
				\date		08/2013
				
				\param		filePath_	Path for the file whose contents are in question, as UTF-8
				\returns	String for the contents of the file, in UTF-8 encoding

				\warning	This should only be used when you expect the file to be non-binary.
			*/
			std::string getTextContents(const std::string& filePath_);

			/**
				\brief		Sets the contents of a text-based file in UTF-8 encoding
				\author		GW
				\date		08/2013
				
				\param		filePath_	Path for the file whose contents are to be set, as UTF-8
				\param		contents_	Contents with which to overwrite the file, as UTF-8
				\returns	true for success, false for failure

				\note		As of now, this function only overwrites the contents of a file and does not allow data
							appending.

				\warning	This should only be used when you expect the file to be non-binary.
			*/
			bool setTextContents(const std::string& filePath_, const std::string& contents_);

			/**
				\brief		Gets the permission bits of a file or folder
				\author		GW
				\date		08/2013
				
				\param		filePath_	Path for the file whose permissions are in question, as UTF-8
				\param		perms__		Return-by-reference for bitwise OR of permissions (0 by default)
				\returns	true if the permissions could be read, false otherwise

				\note		Due to the "unique way" permissions work on Windows, this function will only return the
							permission bits for the current user (not those of the owning group or everyone else).
			*/
			bool getPermissions(const std::string& filePath_, mode_t& perms__);

			/**
				\brief		Cross-platform function to create a directory at a given path with given permissions
				\author		GW
				\date		08/2013

				\param		path_	Location at which the directory will be created, as UTF-8
				\param		perms_	UNIX-style permissions for the file (e.g. octal 0755 means user rwx, group r-x,
									others r-x)
				\returns	true if the directory is created successfully, false otherwise

				\note		This function is safe to call if the directory already exists.
				\note		Due to the "unique way" permissions work on Windows, user/group/everyone permission bits
							will be mapped in the best way possible (typically, the "user" bits apply to the current
							user, the "group" bits apply to administrators, and the "everyone" bits apply to the local
							system).
			*/
			bool makeDirectory(const std::string& path_, const mode_t perms_);
			
			/**
				\brief		Cross-platform function to retrieve the names of every file in a given directory
				\author		GW
				\date		11/2013
				
				\param		path_	Path to directory whose file contents will be read, as UTF-8
				\param		files__	Return-by-reference for a vector of all file names in the given directory, as UTF-8
				\returns	true if the directory exists and its file contents could be read, false otherwise
			*/
			bool filesInDirectory(const std::string& path_, StringVector& files__);

			/**
				\brief		Moves a file to the Trash (Mac) or Recycle Bin (Windows)
				\author		GW
				\date		08/2013

				\param		filePath_	Absolute path to the file that should be deleted, as UTF-8
				\returns	true for success, false for failure
			*/
			bool moveToTrash(const std::string& filePath_);

			/**
				\brief		Un-gzips a source file to a destination path
				\author		GW
				\date		08/2013
				
				\param		sourcePath_		Absolute path to the file that should be un-gzipped, as UTF-8
				\param		destPath_		Absolute path to the file that should be created from the gzipped file, as
											UTF-8
				\returns	true if the file was un-gzipped successfully, false otherwise
				
				\note		This function does not understand archive files (e.g. gzipped tar files)
			*/
			bool ungzip(const std::string& sourcePath_, const std::string& destPath_);

			/**
				\brief		Gets the prefs/registration/etc. folder path (given a subfolder inside it)
				\author		GW
				\date		08/2013

				\param		subfolder_	Your containing prefs subfolder inside the platform prefs path, as UTF-8
				\param		create_		true to create the folder if it's missing, false to simply return the path
				\returns	A string for the path to your prefs/registration/etc. folder, as UTF-8

				\note		On OS X: `/Library/(user name)/Preferences/$subfolder_/`
				\note		On Win XP: `\Documents and Settings\(user name)\Local Settings\Application Data\$subfolder_\`
				\note		On Win Vista, 7, 8: `\Users\(user name)\AppData\Local\$subfolder_\`
			*/
			std::string platformPrefsPath(const std::string& subfolder_, const bool create_ = true);
		}
	}
}

#endif
// __HDI_CORE_FILE_MANIP__
