/**
	\file
	\brief		Header file for fill style map preference data storage
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_FILL_STYLE_MAP_PREF__
#define __HDI_CORE_FILL_STYLE_MAP_PREF__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicorePathStyle.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace aip
	{
		class PersistentDictionary;
	}

	namespace core
	{
		/**
			\brief	Allows for storage of fill style maps
		*/
		class FillStyleMapPref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty FillStyleMapPref object
					\author	GW
					\date	11/2015
					
					\note	To test if a FillStyleMapPref object is empty, call isEmpty() on it
					\note	Empty FillStyleMapPref objects do not relate to any actual preference data; they are designed
							to be "receivers" of some other FillStyleMapPref object via the overloaded assignment operator.
							Empty FillStyleMapPref objects are useless until such time (though it is safe to call any of
							their methods)
				*/
				FillStyleMapPref();
			
				/**
					\brief	Constructs a new FillStyleMapPref object from an existing FillStyleMapPref object (copy constructor)
					\author	GW
					\date	11/2015

					\param	fs_		Existing FillStyleMapPref object
				*/
				FillStyleMapPref(const FillStyleMapPref& fs_);

				/**
					\brief	Constructs a FillStyleMapPref object, with new data, to be added to a container at a later time
					\author	GW
					\date	11/2015

					\param	value_	Initial fill style map value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				FillStyleMapPref(const FillStyle::Map& value_, const std::string& name_ = "");

				/**
					\brief	Destructs a FillStyleMapPref object
					\author	GW
					\date	11/2015
				*/
				virtual ~FillStyleMapPref();

				/**
					\brief		Allows one FillStyleMapPref object to be assigned from another
					\author		GW
					\date		11/2015

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target FillStyleMapPref object, but with its value updated to match that of rhs_
				*/
				virtual FillStyleMapPref& operator=(const FillStyleMapPref& rhs_);
				
				/**
					\brief		Convenience method to clone an FillStyleMapPref object on the heap
					\author		GW
					\date		11/2015

					\returns	A pointer to the new FillStyleMapPref object
					
					\note		If you subclass FillStyleMapPref, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your FillStyleMapPref subclass, you will
								experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned FillStyleMapPref object.
				*/
				virtual FillStyleMapPref* clone() const;

				/**
					\brief		Gets the type of persistent data
					\author		GW
					\date		11/2015

					\returns	FillStyleMapType

					\note		This is useful for knowing how to cast a PrefData object to its proper subclass
				*/
				virtual Type dataType() const;

				/**
					\brief		Gets the fill style map value for this object
					\author		GW
					\date		11/2015

					\returns	The fill style map value being described by this wrapper
				*/
				virtual FillStyle::Map fillStyleMap() const;

				/**
					\brief		Sets the fill style map value for this object
					\author		GW
					\date		11/2015

					\param		value_	New fill style map value
					\returns	true if the fill style map is successfully updated, or false otherwise
				*/
				virtual bool setFillStyleMap(const FillStyle::Map& value_);


			private:
				friend class PathStyleMapPref;

				/**
					\brief	Internal use only
					\author	GW
					\date	11/2015
				*/
				FillStyleMapPref(aip::PersistentDictionary*&);
		};
		
		typedef std::unique_ptr<FillStyleMapPref> FillStyleMapPrefUP;
		typedef std::shared_ptr<FillStyleMapPref> FillStyleMapPrefSP;
		typedef std::weak_ptr<FillStyleMapPref> FillStyleMapPrefWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_FILL_STYLE_MAP_PREF__
