/**
	\file
	\brief		Header file for fill style preference data storage
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_FILL_STYLE_PREF__
#define __HDI_CORE_FILL_STYLE_PREF__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicorePathStyle.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace aip
	{
		class PersistentDictionary;
	}

	namespace core
	{
		/**
			\brief	Allows for storage of fill styles
		*/
		class FillStylePref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty FillStylePref object
					\author	GW
					\date	11/2015
					
					\note	To test if a FillStylePref object is empty, call isEmpty() on it
					\note	Empty FillStylePref objects do not relate to any actual preference data; they are designed to
							be "receivers" of some other FillStylePref object via the overloaded assignment operator.
							Empty FillStylePref objects are useless until such time (though it is safe to call any of
							their methods)
				*/
				FillStylePref();
			
				/**
					\brief	Constructs a new FillStylePref object from an existing FillStylePref object (copy constructor)
					\author	GW
					\date	11/2015

					\param	fs_		Existing FillStylePref object
				*/
				FillStylePref(const FillStylePref& fs_);

				/**
					\brief	Constructs a FillStylePref object, with new data, to be added to a container at a later time
					\author	GW
					\date	11/2015

					\param	value_	Initial fill style value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				FillStylePref(const FillStyle& value_, const std::string& name_ = "");

				/**
					\brief	Destructs a FillStylePref object
					\author	GW
					\date	11/2015
				*/
				virtual ~FillStylePref();

				/**
					\brief		Allows one FillStylePref object to be assigned from another
					\author		GW
					\date		11/2015

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target FillStylePref object, but with its value updated to match that of rhs_
				*/
				virtual FillStylePref& operator=(const FillStylePref& rhs_);
				
				/**
					\brief		Convenience method to clone an FillStylePref object on the heap
					\author		GW
					\date		11/2015

					\returns	A pointer to the new FillStylePref object
					
					\note		If you subclass FillStylePref, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your FillStylePref subclass, you will
								experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned FillStylePref object.
				*/
				virtual FillStylePref* clone() const;

				/**
					\brief		Gets the type of persistent data
					\author		GW
					\date		11/2015

					\returns	FillStyleType

					\note		This is useful for knowing how to cast a PrefData object to its proper subclass
				*/
				virtual Type dataType() const;

				/**
					\brief		Gets the fill style value for this object
					\author		GW
					\date		11/2015

					\returns	The fill style value being described by this wrapper
				*/
				virtual FillStyle fillStyle() const;

				/**
					\brief		Sets the fill style value for this object
					\author		GW
					\date		11/2015

					\param		value_	New fill style value
					\returns	true if the fill style is successfully updated, or false otherwise
				*/
				virtual bool setFillStyle(const FillStyle& value_);


			private:
				friend class PathStylePref;

				/**
					\brief	Internal use only
					\author	GW
					\date	11/2015
				*/
				FillStylePref(aip::PersistentDictionary*&);
		};
		
		typedef std::unique_ptr<FillStylePref> FillStylePrefUP;
		typedef std::shared_ptr<FillStylePref> FillStylePrefSP;
		typedef std::weak_ptr<FillStylePref> FillStylePrefWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_FILL_STYLE_PREF__
