/**
	\file
	\brief		Header file for a class to create and manipulate plugin floating dialogs
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_FLOATING_DIALOG__
#define __HDI_CORE_FLOATING_DIALOG__

#include "hdicoreSize.h"
#include "hdicoreWindow.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		class Widget;

		/**
			\brief	Used to create and display a floating dialog to the user, which is similar to a modal dialog but
					non-blocking
		*/
		class FloatingDialog : public Window
		{
			public:
				/**
					\brief	Constructs an empty FloatingDialog object
					\author	GW
					\date	07/2014
					
					\note	To test if a FloatingDialog object is empty, call isEmpty() on it
					\note	Empty FloatingDialog objects do not relate to any actual UI dialog; they are designed to
							be "receivers" of some other FloatingDialog object via the overloaded assignment operator.
							Empty FloatingDialog objects are useless until such time (though it is safe to call any of
							their methods).
				*/
				FloatingDialog();

				/**
					\brief	Constructs a new FloatingDialog object from an existing FloatingDialog object (copy constructor)
					\author	GW
					\date	07/2014

					\param	fd_		Existing FloatingDialog object
				*/
				FloatingDialog(const FloatingDialog& fd_);
			
				/**
					\brief	Constructs a FloatingDialog object with a given name and size
					\author	GW
					\date	07/2014

					\param	name_	Name (title) of the dialog, as UTF-8
					\param	size_	Width and height, in 1x resolution coordinates

					\note	All floating dialogs are presented in the center of the screen.
				*/
				FloatingDialog(const std::string& name_, const Size& size_);

				/**
					\brief	Destructs a FloatingDialog object
					\author	GW
					\date	07/2014
				*/
				virtual ~FloatingDialog();

				/**
					\brief		Allows one FloatingDialog object to be assigned from another
					\author		GW
					\date		07/2014

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target FloatingDialog object, but with its value updated to match that of rhs_
				*/
				virtual FloatingDialog& operator=(const FloatingDialog& rhs_);
				
				/**
					\brief		Convenience method to clone a FloatingDialog object on the heap
					\author		GW
					\date		07/2014

					\returns	A pointer to the new FloatingDialog object
					
					\note		If you subclass FloatingDialog, you MUST overload this method yourself! If you don't
								and/or your clone() method does not return an instance of your FloatingDialog subclass,
								you will experience "object slicing" when e.g. adding the modal dialog to a window
								collection.

					\warning	The caller must manage the memory for the returned FloatingDialog object.
				*/
				virtual FloatingDialog* clone() const;

				/**
					\brief	Destroys the UI floating dialog, converting the target object to an empty FloatingDialog
							object (see the default constructor for more info on empty FloatingDialog objects)
					\author	GW
					\date	07/2014

					\note	This method must be called for any existing floating dialogs by the time the plugin is
							shutting down. If you no longer retain any of your own references to a floating dialog then
							the lib will call this method for you as it is shutting down.
				*/
				virtual void destroy();

				/**
					\brief		Gets the platform-specific floating dialog, around which the target object is wrapped
					\author		GW
					\date		07/2014

					\returns	Pointer to platform-specific floating dialog. See PlatformFloatingDialogPtr typedef.

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of UI functionality is not handled by this class (or related
								classes), then it should probably be added to the hdi_core library.
				*/
				virtual PlatformFloatingDialogPtr platformFloatingDialog() const;

				/**
					\brief		Gets whether the dialog is visible
					\author		GW
					\date		07/2014

					\returns	true if the dialog is visible, false otherwise
				*/
				virtual bool visible() const;
				
				/**
					\brief	Shows the floating dialog
					\author	GW
					\date	07/2014
				*/
				virtual void show();

				/**
					\brief	Hides the floating dialog
					\author	GW
					\date	07/2014
				*/
				virtual void hide();

				/**
					\brief		Adds a widget (label, button, text field, etc.) to the floating dialog
					\author		GW
					\date		07/2014

					\param		widget_		Widget to place in the dialog
					\returns	true if the widget was added, false otherwise
				*/
				virtual bool addWidget(const Widget& widget_);

				/**
					\brief		Removes a widget (label, button, text field, etc.) from the floating dialog
					\author		GW
					\date		07/2014

					\param		widget_		Widget to remove from the dialog
					\returns	true if the widget was removed, false otherwise
				*/
				virtual bool removeWidget(const Widget& widget_);

				/**
					\brief		Gets the current floating dialog title
					\author		GW
					\date		07/2014
					
					\returns	String value for the floating dialog title, as UTF-8
				*/
				virtual std::string title() const;

				/**
					\brief	Sets the current floating dialog title
					\author	GW
					\date	07/2014
					
					\param	title_	New string value for the floating dialog title, as UTF-8
				*/
				virtual void setTitle(const std::string& title_);

				/**
					\brief		Gets the floating dialog's size
					\author		GW
					\date		07/2014

					\returns	A Size object representing the dialog's dimensions, in 1x resolution coordinates
				*/
				virtual Size size() const;

				/**
					\brief	Sets the floating dialog's size
					\author	GW
					\date	07/2014

					\param	size_	New dialog size, in 1x resolution coordinates
				*/
				virtual void setSize(const Size& size_);

				/**
					\brief	Forces the widgets in the floating dialog to update/redraw
					\author	GW
					\date	07/2014
				*/
				virtual void update() const;

				/**
					\brief		Gets the current close button callback
					\author		GW
					\date		07/2014

					\returns	A pointer to the currently registered close button callback, or NULL for none
				*/
				virtual Callback* const closeCallback() const;

				/**
					\brief	Sets the close button callback for the dialog
					\author	GW
					\date	07/2014

					\param	cb_	New callback (target and action) for when the close button is clicked
				*/
				virtual void setCloseCallback(const Callback& cb_);

				/**
					\brief		Gets the current dialog focus callback
					\author		GW
					\date		07/2014

					\returns	A pointer to the currently registered focus callback, or NULL for none
				*/
				virtual Callback* const focusCallback() const;

				/**
					\brief	Sets the dialog focus callback
					\author	GW
					\date	07/2014

					\param	cb_	New callback (target and action) for when the dialog becomes focused
				*/
				void setFocusCallback(const Callback& cb_);

				/**
					\brief		Gets the current dialog blur callback
					\author		GW
					\date		07/2014

					\returns	A pointer to the currently registered blur callback, or NULL for none
				*/
				virtual Callback* const blurCallback() const;

				/**
					\brief	Sets the dialog blur callback
					\author	GW
					\date	07/2014

					\param	cb_	New callback (target and action) for when the dialog becomes blurred
				*/
				void setBlurCallback(const Callback& cb_);
		};
		
		typedef std::unique_ptr<FloatingDialog> FloatingDialogUP;
		typedef std::shared_ptr<FloatingDialog> FloatingDialogSP;
		typedef std::weak_ptr<FloatingDialog> FloatingDialogWP;
	}
}

#endif
// __HDI_CORE_FLOATING_DIALOG__
