/**
	\file
	\brief		Header file for Illustrator user interface panel flyout menus
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_FLYOUT__
#define __HDI_CORE_FLYOUT__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace pui
	{
		class PopupMenu;
	}

	namespace core
	{
		class Callback;
		class PopupEntry;
		class Panel;

		/**
			\brief	Flyout menu, which can be added to a Panel for additional options
		*/
		class Flyout
		{
			public:
				typedef std::vector<PopupEntry> EntryVector;
			
				/**
					\brief	Constructs an empty Flyout menu, to which entries may be added later
					\author	GW
					\date	09/2013
				*/
				Flyout();
				
				/**
					\brief	Constructs a new Flyout object from an existing Flyout object (copy constructor)
					\author	GW
					\date	09/2013

					\param	fo_		Existing Flyout object
				*/
				Flyout(const Flyout& fo_);

				/**
					\brief	Constructs a Flyout menu with a list of entries
					\author	GW
					\date	09/2013

					\param	entries_		Vector of string entries, whose order in the flyout will match the vector
				*/
				Flyout(const EntryVector& entries_);
				
				/**
					\brief	Flyout destructor
					\author	GW
					\date	09/2013
				*/
				virtual ~Flyout();
				
				/**
					\brief		Overloaded assigment operator to copy one Flyout object's values to another
					\author		GW
					\date		12/2014
					
					\param		rhs_	Righthand side of the assignment operator
					\returns	The lefthand side of the assignment operator, but with values updated to match that of
								the rhs_ parameter
				*/
				virtual Flyout& operator=(const Flyout& rhs_);

				/**
					\brief		Gets the index of the currently selected entry
					\author		GW
					\date		09/2013

					\returns	The index number of the currently selected entry, or -1 if no entry is selected
				*/
				virtual int32_t currentIndex() const;
				
				/**
					\brief		Gets the ID of the currently selected entry
					\author		GW
					\date		09/2013

					\returns	UTF-8 ID string of the currently selected entry, or "" if no entry is currently selected
				*/
				virtual std::string currentID() const;
				
				/**
					\brief		Gets the string value of the currently selected entry
					\author		GW
					\date		09/2013

					\returns	Value string of the flyout menu, as UTF-8
				*/
				virtual std::string currentValue() const;

				/**
					\brief		Gets the current entry object
					\author		GW
					\date		09/2013

					\returns	The PopupEntry object for the currently selected item
				*/
				virtual std::unique_ptr<PopupEntry> currentEntry() const;
				
				/**
					\brief		Gets the user data for the current entry
					\author		GW
					\date		09/2013

					\param		data__	Return-by-reference for the current user data; ignore if false is returned
					\returns	true if the method succeeds in acquiring the contained entry data, false otherwise
				*/
				virtual bool currentUserData(void*& data__) const;

				/**
					\brief		Gets the entries vector
					\author		GW
					\date		09/2013

					\returns	The entries vector for the target flyout, containing all currently known entries
					
					\note		Modifying any value in an entry will not cause the menu to update automatically; call
								setEntries() to update changes to entry data, or call update() to update the UI for
								changes made to entry string values.
				*/
				virtual const std::vector< std::shared_ptr<PopupEntry> >* const entries() const;
				
				/**
					\brief	Allows the entries in an flyout menu to be changed out entirely
					\author	GW
					\date	09/2013

					\param	entries_	New set of entries that the menu will contain
				*/
				virtual void setEntries(const EntryVector& entries_);
				
				/**
					\brief		Gets whether the entries vector already contains an entry, based on its ID
					\author		GW
					\date		09/2013

					\param		id_		ID of the entry to search for, as UTF-8
					\returns	true if the menu already contains an entry with the given ID, false otherwise
				*/
				virtual bool hasEntry(const std::string& id_) const;

				/**
					\brief		Gets the entry with the given ID, provided it exists within the widget
					\author		GW
					\date		02/2023

					\param		id_		ID of the entry to search for
					\returns	The entry with the given ID, or NULL if not found
				*/
				virtual std::shared_ptr<PopupEntry> entryWithID(const std::string& id_) const;
				
				/**
					\brief		Pushes an entry onto the list (i.e. at the end)
					\author		GW
					\date		09/2013

					\param		entry_	New entry object to insert into the list of entries
					\returns	true if the entry is pushed successfully, false otherwise

					\note		If the entry is already present in the list (based on its ID), it will not be inserted.
				*/
				virtual bool pushEntry(const PopupEntry& entry_);

				/**
					\brief		Pops an entry off the list (i.e. from the end)
					\author		GW
					\date		09/2013

					\param		entry__		Return-by-reference for the popped entry, so the caller can inspect which
											one was removed
					\returns	true if an entry was popped and acquired, false otherwise
				*/
				virtual bool popEntry(PopupEntry& entry__);

				/**
					\brief		Inserts an entry into the list of entries at a given position
					\author		GW
					\date		09/2013

					\param		id_		Existing entry immediately before the position at which the new entry will be
										inserted, as UTF-8
					\param		entry_	New entry object to insert into the list of entries
					\returns	true if the entry is inserted successfully, false otherwise

					\note		If the entry for id_ doesn't exist in the list, the new entry will not be inserted.
					\note		If the passed entry is already present in the list (based on its ID), it will not be
								inserted.
				*/
				virtual bool insertEntry(const std::string& id_, const PopupEntry& entry_);

				/**
					\brief		Removes a specific entry from the list of entries
					\author		GW
					\date		09/2013

					\param		id_			ID of the entry to be removed, as UTF-8
					\param		entry__		Return-by-reference for the removed entry, so the caller can inspect which
											one was removed
					\returns	true if an entry was removed and acquired, false otherwise
				*/
				virtual bool removeEntry(const std::string& id_, PopupEntry& entry__);

				/**
					\brief	Clears all entries contained by the flyout
					\author	GW
					\date	09/2013
				*/
				virtual void clearEntries();

				/**
					\brief		Gets the parent panel of the flyout
					\author		GW
					\date		09/2013

					\returns	Parent panel object pointer (NULL if the flyout hasn't been added to one yet)
				*/
				virtual Panel* const panel() const;

				/**
					\brief	Simulates the flyout menu having one of its entries selected
					\author	GW
					\date	09/2013

					\note	When the callback (registered with the setSelectionMadeCallback() method) is called, you can
							determine what entry was selected by using any of the current*() methods (e.g. currentID(),
							currentEntry(), etc.)
				*/
				virtual void selectionMade();
				
				/**
					\brief		Gets the flyout entry selection callback
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered selection callback
				*/
				virtual Callback* const selectionMadeCallback() const;
				
				/**
					\brief	Sets the flyout entry selection callback
					\author	GW
					\date	09/2013

					\param	callback_	New callback for when a flyout menu entry is selected
				*/
				virtual void setSelectionMadeCallback(const Callback& callback_);
				
				/**
					\brief		Gets the showing callback for the flyout menu
					\author		GW
					\date		04/2015

					\returns	A pointer to the currently registered flyout-about-to-show callback
				*/
				virtual Callback* const showingCallback() const;
				
				/**
					\brief	Sets the showing callback for the flyout menu
					\author	GW
					\date	04/2015

					\param	callback_	New callback for when a the flyout is about to be shown

					\pre	The flyout must already be set on its associated panel for this method to function.
				*/
				virtual void setShowingCallback(const Callback& callback_);
				
				/**
					\brief		Gets whether the specified flyout entry is checked
					\author		GW
					\date		03/2014
					
					\param		index_	The index number of the entry in question
					\returns	true if the entry is checked, false otherwise
				*/
				virtual bool entryChecked(const int32_t index_);
				
				/**
					\brief		Sets whether the specified flyout entry is checked
					\author		GW
					\date		03/2014
					
					\param		index_	The index number of the entry in question
					\param		check_	true to check the entry, false to uncheck
					\returns	true if the entry's checkmark state is set, false otherwise
				*/
				virtual bool checkEntry(const int32_t index_, const bool check_);
				
				/**
					\brief		Gets whether the specified flyout entry is checked
					\author		GW
					\date		03/2014
					
					\param		id_		ID string of the entry in question, as UTF-8
					\returns	true if the entry is checked, false otherwise
				*/
				virtual bool entryChecked(const std::string& id_);
				
				/**
					\brief		Sets whether the specified flyout entry is checked
					\author		GW
					\date		03/2014
					
					\param		id_		ID string of the entry in question, as UTF-8
					\param		check_	true to check the entry, false to uncheck
					\returns	true if the entry's checkmark state is set, false otherwise
				*/
				virtual bool checkEntry(const std::string& id_, const bool check_);
				
				/**
					\brief	Forces the flyout menu to update its entries
					\author	GW
					\date	04/2015
					
					\pre	The flyout must already be set on its associated panel for this method to function.
				*/
				virtual void update();


			private:
				friend class Panel;
				friend pui::PopupMenu* __accessImpl(const Flyout&);

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	09/2013
				*/
				void* __impl() const;
		};
		
		typedef std::unique_ptr<Flyout> FlyoutUP;
		typedef std::shared_ptr<Flyout> FlyoutSP;
		typedef std::weak_ptr<Flyout> FlyoutWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_FLYOUT__
