/**
	\file
	\brief		Header file for font preference data storage
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_FONT_PREF__
#define __HDI_CORE_FONT_PREF__

#include "hdicorePrefData.h"

namespace hdi
{
	namespace core
	{
		class Font;
	
		/**
			\brief	Allows for storage of font faces and sizes
		*/
		class FontPref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty FontPref object
					\author	GW
					\date	09/2013
					
					\note	To test if a FontPref object is empty, call isEmpty() on it
					\note	Empty FontPref objects do not relate to any actual preference data; they are designed to be
							"receivers" of some other FontPref object via the overloaded assignment operator. Empty
							FontPref objects are useless until such time (though it is safe to call any of their methods)
				*/
				FontPref();
			
				/**
					\brief	Constructs a new FontPref object from an existing FontPref object (copy constructor)
					\author	GW
					\date	09/2013

					\param	f_	Existing FontPref object
				*/
				FontPref(const FontPref& f_);

				/**
					\brief	Constructs a FontPref object, with new data, to be added to a container at a later time
					\author	GW
					\date	09/2013

					\param	face_	Initial font face
					\param	size_	Initial font size
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				FontPref(const Font& face_, const double size_, const std::string& name_ = "");

				/**
					\brief	Destructs a FontPref object
					\author	GW
					\date	09/2013
				*/
				virtual ~FontPref();

				/**
					\brief		Allows one FontPref object to be assigned from another
					\author		GW
					\date		09/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target FontPref object, but with its value updated to match that of rhs_
				*/
				virtual FontPref& operator=(const FontPref& rhs_);
				
				/**
					\brief		Convenience method to clone an FontPref object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new FontPref object
					
					\note		If you subclass FontPref, you MUST overload this method yourself! If you don't and/or 
								your clone() method does not return an instance of your FontPref subclass, you will
								experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned FontPref object.
				*/
				virtual FontPref* clone() const;

				/**
					\brief		Gets the font face for this object
					\author		GW
					\date		09/2013

					\returns	The Font object being described by this wrapper
				*/
				virtual std::unique_ptr<Font> font() const;

				/**
					\brief		Sets the font face for this object
					\author		GW
					\date		09/2013

					\param		font_	New font face
					\returns	true if the font is successfully updated, or false otherwise
				*/
				virtual bool setFont(const Font& font_);

				/**
					\brief		Gets the font size for this object
					\author		GW
					\date		09/2013

					\returns	The font size being described by this wrapper
				*/
				virtual double size() const;

				/**
					\brief		Sets the font size for this object
					\author		GW
					\date		09/2013

					\param		size_	New font size value
					\returns	true if the font size is successfully updated, or false otherwise
				*/
				virtual bool setSize(const double size_);
		};
		
		typedef std::unique_ptr<FontPref> FontPrefUP;
		typedef std::shared_ptr<FontPref> FontPrefSP;
		typedef std::weak_ptr<FontPref> FontPrefWP;
	}
}

#endif
// __HDI_CORE_FONT_PREF__
