/**
	\file
	\brief		Header file for manipulating the Illustrator document grid
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_GRID__
#define __HDI_CORE_GRID__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"
#include "hdicoreColor.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Allows manipulation of Illustrator's built-in grids
		*/
		class Grid
		{
			public:
				/**
					\brief	Indicates how the grid is drawn
				*/
				enum Style
				{
					UnknownStyle	= 0,
					GraphStyle		= 10,
					DottedStyle		= 20
				};
				
				/**
					\brief	Describes the spacing and density of gridlines
				*/
				class Spacing
				{
					public:
						/**
							\brief	Space between major gridlines, in points
						*/
						double spacing;
						
						/**
							\brief	Number of minor gridlines between the major gridlines
						*/
						int32_t ticks;
						
						/**
							\brief	Constructs a new Spacing object with default values
							\author	GW
							\date	08/2013
						*/
						Spacing();
						
						/**
							\brief	Destructs a Spacing object
							\author	GW
							\date	08/2013
						*/
						virtual ~Spacing();
				};
		
				typedef std::unique_ptr<Spacing> SpacingUP;
				typedef std::shared_ptr<Spacing> SpacingSP;
				typedef std::weak_ptr<Spacing> SpacingWP;

				/**
					\brief	Destructs a Grid object
					\author	GW
					\date	08/2013
				*/
				~Grid();

				/**
					\brief	Shows the grid
					\author	GW
					\date	08/2013
				*/
				void show();
				
				/**
					\brief	Hides the grid
					\author	GW
					\date	08/2013
				*/
				void hide();
				
				/**
					\brief		Gets whether the grid is visible or not
					\author		GW
					\date		08/2013

					\returns	true if the grid is visible, false otherwise
				*/
				bool visible() const;
				
				/**
					\brief	Sets whether the grid is visible
					\author	GW
					\date	08/2013

					\param	visible_	true for visible, false otherwise
					
					\note	This method simply calls show() or hide() as appropriate
				*/
				void setVisible(const bool visible_);
				
				/**
					\brief		Gets whether the gridlines can be snapped to
					\author		GW
					\date		08/2013

					\returns	true to snap the cursor to gridlines, false otherwise
				*/
				bool snaps() const;
				
				/**
					\brief	Sets whether the gridlines can be snapped to
					\author	GW
					\date	08/2013

					\param	snaps_	true for snapping, false otherwise
				*/
				void setSnaps(const bool snaps_);
				
				/**
					\brief		Gets whether the grid draws behind art
					\author		GW
					\date		08/2013

					\returns	true if the grid is behind art, false otherwise
				*/
				bool behindArt() const;

				/**
					\brief	Sets whether the grid draws behind art
					\author	GW
					\date	08/2013

					\param	behind_		true for gridline drawing at the "backmost" of the document, false for "frontmost"
				*/
				void setBehindArt(const bool behind_);

				/**
					\brief		Gets the gridline style, either kAIGridStyleGraph and kAIGridStyleDotted
					\author		GW
					\date		08/2013

					\returns	The current gridline styling
				*/
				Style style() const;

				/**
					\brief	Sets the gridline style, either kAIGridStyleGraph and kAIGridStyleDotted
					\author	GW
					\date	08/2013

					\param	style_	New gridline styling
				*/
				void setStyle(const Style style_);
				
				/**
					\brief		Gets the spacing of the gridlines
					\author		GW
					\date		08/2013

					\returns	Current gridline spacing settings
				*/
				Spacing spacing() const;
				
				/**
					\brief	Sets the spacing of the gridlines from an AIGridSpacing struct
					\author	GW
					\date	08/2013

					\param	spacing_	New (major) gridline spacing and number of ticks (minor lines) between them
				*/
				void setSpacing(const Spacing& spacing_);
				
				/**
					\brief	Sets the spacing of the gridlines from a spacing distance and the number of ticks between
					\author	GW
					\date	08/2013

					\param	spacing_	New (major) gridline spacing
					\param	ticks_		Number of ticks (minor lines) between the gridlines
				*/
				void setSpacing(const double spacing_, const int32_t ticks_);
				
				/**
					\brief		Gets the current "dark" color (for major gridlines)
					\author		GW
					\date		08/2013

					\returns	Current color for the major gridlines
				*/
				Color darkColor() const;
				
				/**
					\brief	Sets the current "dark" color (for major gridlines)
					\author	GW
					\date	08/2013

					\param	color_	New color for the major gridlines
				*/
				void setDarkColor(const Color& color_);
				
				/**
					\brief		Gets the current "light" color (for minor gridlines, also called "ticks")
					\author		GW
					\date		08/2013

					\returns	Current color for the minor gridlines
				*/
				Color lightColor() const;
				
				/**
					\brief	Sets the current "light" color (for minor gridlines, also called "ticks")
					\author	GW
					\date	08/2013

					\param	color_	New color for the minor gridlines
				*/
				void setLightColor(const Color& color_);


			private:
				// Only CurrentDocument can construct a Grid object
				friend class CurrentDocument;
				
				/**
					\brief	Constructs a Grid object; at the moment, this can only be done by the CurrentDocument class
					\author	GW
					\date	08/2013
				*/
				Grid();
		};
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_GRID__
