/**
	\file
	\brief		Header file for Illustrator hit tests
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_HIT_DATA__
#define __HDI_CORE_HIT_DATA__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreArtboardPoint.h"
#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class HitData;
	}

	namespace core
	{
		class Art;
		class Bezier;
		class TextRange;

		/**
			\brief	Allows for hit data creation, metadata access, snapping, etc.
		*/
		class HitData
		{
			public:
				/**
					\brief	Describes the type hit that was actually detected from a request
					
					\note	See the comment for each enum value for what type of request might result in the indicated
							hit.
				*/
				enum HitType
				{
					NothingHit		= 0,	// Any hit request might result in this, if nothing is hit
					SegPointHit		= 10,	// SegPointRequest, AnySegControlRequest, SegPointOrInteriorRequest, etc.
					SegInPointHit	= 20,	// SegInPointRequest, AnySegControlRequest, AnyHitRequestNoGuides, etc.
					SegOutPointHit	= 30,	// SegOutPointRequest, AnySegControlRequest, AnyHitRequestNoGuides, etc.
					SegInteriorHit	= 40,	// SegInteriorRequest, SegPointOrInteriorRequest, etc.
					FillHit			= 50	// FillRequest, AnyHitRequestNoGuides, AnyHitRequest
				};
			
				/**
					\brief	Constructs an empty HitData object
					\author	GW
					\date	08/2013
					
					\note	To test if a HitData object is empty, call isEmpty() on it
					\note	Empty HitData objects do not relate to any actual hit on the Illustrator artboard; they are
							designed to be "receivers" of some other HitData object via the overloaded assignment
							operator. Empty HitData objects are useless until such time (though it is safe to call any
							of their methods).
				*/
				HitData();

				/**
					\brief	Constructs a HitData object from an existing HitData object (copy constructor)
					\author	GW
					\date	08/2013

					\param	hitData_	Existing HitData object
				*/
				HitData(const HitData& hitData_);

				/**
					\brief		Constructs a HitData object from a hit test, with an optional tolerance and zoom
								adjustment
					\author		GW
					\date		08/2013

					\param		art_			Root art of a "tree" to restrict the hit, or NULL to search to whole
												document
					\param		pt_				Point at which to perform the hit test
					\param		request_		Type of hit test to perform
					\param		tolerance_		Point radius around pt_ in which a hit is valid
					\param		adjustToZoom_	Adjust the tolerance_ argument to take document zoom into account

					\note		If adjustToZoom_ is true, then the tolerance_ argument specifies the point radius around
								pt_ at 100% document zoom. If the document is zoomed to 200%, tolerance_ will be
								adjusted to 1/2 the given value; if zoomed to 400%, tolerance_ will be adjusted to 1/4
								the given value; etc.

			 		\warning	This method does not assume responsibility of the memory for the art_ argument.
				*/
				HitData(
					const Art* const art_,
					const ArtboardPoint& pt_,
					const HitRequest request_,
					const double tolerance_ = 2.0,
					const bool adjustToZoom_ = true
				);

				/**
					\brief	Destructs a HitData object
					\author	GW
					\date	08/2013
				*/
				virtual ~HitData();

				/**
					\brief		Gets whether the target HitData object is empty (constructed with the default ctor)
					\author		GW
					\date		08/2013
					
					\returns	true if the target HitData object is empty, false otherwise
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Assigns one HitData object to another
					\author		GW
					\date		08/2013

					\param		rhs_	Existing HitData object to copy values from
					\returns	The target HitData object, but with its values updated to match that of the rhs_ argument
				*/
				virtual HitData& operator=(const HitData& rhs_);
				
				/**
					\brief		Tests whether a given HitData object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	HitData to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same data, false otherwise
				*/
				virtual bool operator==(const HitData& rhs_) const;
				
				/**
					\brief		Tests whether a given HitData object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	HitData to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different data, false otherwise
				*/
				virtual bool operator!=(const HitData& rhs_) const;

				/**
					\brief		Gets the original type of hit test performed
					\author		GW
					\date		08/2013

					\returns	Type of hit request originally provided to a HitData constructor, or UnknownRequest if
								none
				*/
				virtual HitRequest request() const;

				/**
					\brief		Reports whether some art was hit
					\author		GW
					\date		08/2013

					\returns	true if some art was hit, false otherwise
				*/
				virtual bool hit() const;

				/**
					\brief		Gets the type of hit detected
					\author		GW
					\date		08/2013

					\returns	The type of hit actually detected; see HitType enum docs for more info
				*/
				virtual HitType hitType() const;

				/**
					\brief		Gets the art that was hit (or NULL if none)
					\author		GW
					\date		08/2013

					\returns	The art that was hit, or NULL if none
				*/
				virtual std::unique_ptr<Art> art() const;

				/**
					\brief		Gets the segment point hit
					\author		GW
					\date		08/2013

					\returns	The index of the segment point that was hit, or (generally) -1 if the hit type is not
								SegPointHit, SegInPointHit, or SegOutPointHit

					\note		If the hit type is SegInteriorHit, this method attempts to provide a meaningful value;
								typically, this value will be the same as calling the segIndex() method.
				*/
				virtual int16_t segPointIndex() const;

				/**
					\brief		Gets the segment number of a path that was hit
					\author		GW
					\date		08/2013

					\returns	The index of the segment that was hit, or (generally) -1 if the hit type is not
								SegInteriorHit

					\note		If the hit type is SegPointHit, SegInPointHit, or SegOutPointHit, this method attempts
								to provide a meaningful value; typically, the "segment index" hit will be the same as
								calling segPointIndex() with the exception of the last point being hit in an open path
								(then it's n-1).
				*/
				virtual int16_t segIndex() const;

				/**
					\brief		Gets the t-value of the hit along a path
					\author		GW
					\date		08/2013

					\returns	The t-value that was hit, or (generally) 0.0 if the hit type is not SegInteriorHit
					
					\note		If the hit type is SegPointHit, SegInPointHit, or SegOutPointHit, this method attempts
								to provide a meaningful value; typically, the t-value hit will be 0.0 with the exception
								of the last point being hit in an open path (then it's 1.0).
				*/
				virtual double tValue() const;

				/**
					\brief		Gets the point hit
					\author		GW
					\date		08/2013

					\returns	Either the point at which a hit was detected within some given tolerance, or the
								original point at which the hit test was performed (in the case of no hit or, often, a
								fill hit)
				*/
				virtual ArtboardPoint point() const;

				/**
					\brief		Gets the bezier for the segment hit
					\author		GW
					\date		08/2013

					\returns	A Bezier object for the segment hit, or NULL if no segment was hit
				*/
				virtual std::unique_ptr<Bezier> bezier() const;
				
				/**
					\brief		Gets the text range that was hit (preset to the range of the character that is at the
								hit point)
					\author		GW
					\date		08/2014
					
					\returns	The text range that was hit if the art is of type ArtTypeText, NULL otherwise
				*/
				virtual std::unique_ptr<TextRange> textRange() const;

				/**
					\brief		Convenience method to modify and "snap" a given point if a hit has occurred
					\author		GW
					\date		08/2013

					\param		pt__	Existing point, only modified if a hit has occurred
					\returns	true if pt__ was "snapped" to point(), false otherwise

					\note		This method is useful for e.g. snapping the actual cursor location to some art that was
								hit before drawing new art.
				*/
				virtual bool snapForHit(ArtboardPoint& pt__) const;

				/**
					\brief		Convenience method to modify and "snap" a given point only if a certain type of hit has
								occurred
					\author		GW
					\date		08/2013

					\param		type_	Type of hit desired for snapping
					\param		pt__	Existing point, only modified if the correct type of hit has occurred
					\returns	true if pt__ was "snapped" to point(), false otherwise

					\note		This method is useful for e.g. snapping the actual cursor location to some art that was
								hit before drawing new art.
				*/
				virtual bool snapForHitType(const HitType type_, ArtboardPoint& pt__) const;
			
			
			private:
				friend aip::HitData* __accessImpl(const HitData&);
				friend HitData __accessCtor(const aip::HitData&);

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				void* __impl() const;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				HitData(const aip::HitData&);
		};
		
		typedef std::unique_ptr<HitData> HitDataUP;
		typedef std::shared_ptr<HitData> HitDataSP;
		typedef std::weak_ptr<HitData> HitDataWP;

		extern aip::HitData* __accessImpl(const HitData&);
		extern HitData __accessCtor(const aip::HitData&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_HIT_DATA__
