/**
	\file
	\brief		Header file for Illustrator app-related methods
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_ILLUSTRATOR__
#define __HDI_CORE_ILLUSTRATOR__

#if defined(HDI_CORE_AIP_MODE)

#include <map>
#include <vector>

#include "hdicoreColor.h"
#include "hdicoreTypes.h"

/**
	\brief	Slightly shorter way to access the Illustrator instance, but still utilizing a non-colliding name scheme
*/
#define HDI_CORE_ILLUSTRATOR	hdi::core::Illustrator::instance()

namespace hdi
{
	namespace core
	{
		namespace ai
		{
			class Preferences;
		}

		class CurrentDocument;
		class Document;
		class FileFormat;
		class Font;
		class LiveEffect;
		class MenuGroup;
		class ThirdPartyPlugin;
		class Typeface;
		
		struct SuiteInfo;

		/**
			\brief	Allows access to a wide variety of Illustrator app-related runtime attributes, documents, fonts, etc.
		*/
		class Illustrator
		{
			public:
				typedef std::vector< std::shared_ptr<Font> > FontVector;
				typedef std::map<std::string, SuiteInfo> SuiteInfoMap;
				typedef std::map< std::string, ThirdPartyPlugin* > ThirdPartyPluginMap;
			
				/**
					\brief	Used when querying for current UI colors
				*/
				enum ThemeWindowType
				{
					PanelOrControlBarType		= 10,
					ModalOrFloatingDialogType	= 20
				};

				/**
					\brief	Used to describe a given builtin Illustrator toolbox
				*/
				enum BuiltinToolbox
				{
					UnknownBuiltinToolbox	= 0,
					BasicBuiltinToolbox		= 1 << 0,
					AdvancedBuiltinToolbox	= 1 << 1
				};
				
				/**
					\brief	Describes a particular folder in an Illustrator installation, whose path can be acquired via
							the Illustrator class
				*/
				enum FolderType
				{
					UnknownFolderType										= 0,
					ApplicationFolderType									= 10,
					PluginsFolderType										= 20,
					PrimaryScratchFolderType								= 30,
					SecondaryScratchFolderType								= 40,
					PreferencesFolderType									= 50,
					UserSupportFolderType									= 60,
					UserSupportAIFolderType									= 70,
					UserSupportAIPluginsFolderType							= 80,
					ApplicationSupportCommonFolderType						= 90,
					ApplicationSupportCommonColorFolderType					= 100,
					ApplicationSupportCommonTypeSupportFolderType			= 110,
					ApplicationSupportCommonFontsFolderType					= 120,
					ApplicationSupportCommonFontsRequiredFolderType			= 130,
					ApplicationSupportCommonFontsRequiredCMapsFolderType	= 140,
					RequiredFontsFolderType									= 150,
					FontsFolderType											= 160,
					MyDocumentsFolderType									= 170,
					ApplicationSupportCommonWorkflowFolderType				= 180,
					PrinterDescriptionsFolderType							= 190,
					RequiredPluginsFolderType								= 200,
					SettingsFolderType										= 210,
					ColorTableSettingsFolderType							= 220,
					OptimizeSettingsFolderType								= 230,
					HelpFolderType											= 240,
					RootFolderType											= 250,
					PresetsFolderType										= 260,
					PresetActionsFolderType									= 270,
					PresetBrushesFolderType									= 280,
					PresetGradientsFolderType								= 290,
					PresetKeyboardShortcutsFolderType						= 300,
					PresetPatternsFolderType								= 310,
					PresetScriptsFolderType									= 320,
					PresetStylesFolderType									= 330,
					PresetSwatchesFolderType								= 340,
					DictionariesFolderType									= 350,
					LegalFolderType											= 360,
					SampleFilesFolderType									= 370,
					UtilitiesFolderType										= 380,
					PackageFolderType										= 390,
					ApplicationSupportCommonFontsRequiredBaseFolderType		= 400,
					HelpersFolderType										= 410,
					PreviewInFolderType										= 420,
					RidersFileFolderType									= 430,
					HyphenationDictFolderType								= 440,
					ApplicationSupportCommonPDFLFolderType					= 450,
					ApplicationSupportCommonPDFL5FolderType					= 460,
					ApplicationSupportCommonPDFL5CMapsFolderType			= 470,
					ApplicationSupportCommonPDFL5FontsFolderType			= 480,
					ApplicationSupportCommonPrintSupportFolderType			= 490,
					ApplicationSupportCommonColorProfilesFolderType			= 500,
					ApplicationSupportCommonColorSettingsFolderType			= 510,
					ContentsFolderType										= 520,
					HelpImagesFolderType									= 530,
					FontsCMapsFolderType									= 540,
					PresetSymbolsFolderType									= 550,
					TsumeFolderType											= 560,
					SpellingDictFolderType									= 570,
					PresetTemplatesFolderType								= 580,
					FontsCFFolderType										= 590,
					ApplicationSupportCommonKinsokuSetFolderType			= 600,
					ApplicationSupportCommonMojikumeSetFolderType			= 610,
					PresetBlankDocumentsFolderType							= 620,
					UserSupportCommonFontsFolderType						= 630,
					FontsCFTempFolderType									= 640,
					LogsFolderType											= 650,
					SampleArtFolderType										= 660,
					SampleSVGFolderType										= 670,
					SampleGraphDesignsFolderType							= 680,
					SampleDataDrivenGraphicsFolderType						= 690,
					WorkspacesFolderType									= 700,
					PresetColorBooksFolderType								= 710,
					PresetLegacyColorBooksFolderType						= 720,
					PresetSwatchExchangeFolderType							= 730,
					ApplicationSupportCommonLinguisticsFolderType			= 740,
					ApplicationSupportCommonLinguisticsProvidersFolderType	= 750,
					DemonstratorFolderType									= 760,
					ResourcesFolderType										= 770,
					CoolExtrasFolderType									= 780,
					OutputSettingsFolderType								= 790,
					StartupScriptsFolderType								= 800,
					DesktopFolderType										= 810,
					IllustratorFormatsFolderType							= 820,
					AdditionalAIPluginsFolderType							= 830,
					UserWritableStartupFileFolderType						= 840,
					UserWritablePresetBrushesFolderType						= 850,
					UserWritablePresetStylesFolderType						= 860,
					UserWritablePresetSwatchesFolderType					= 870,
					UserWritablePresetSwatchExchangeFolderType				= 880,
					UserWritablePresetSymbolsFolderType						= 890,
					PresetFlashPanelsFolderType								= 900,
					UserWritablePresetSettingsFolderType					= 910,
					UserWritablePresetOptimizeSettingsFolderType			= 920,
					UserWritablePresetOutputSettingsFolderType				= 930,
					UserWritablePresetColorTableSettingsFolderType			= 940,
					RequiredStartupProfilesFolderType						= 950,
					PresetWorkspacesFolderType								= 960,
					PresetWelcomeScreenFolderType							= 970,
					RequiredFolderType										= 980,
					RequiredResourcesFolderType								= 990,
					RequiredLocalizedResourcesFolderType					= 1000,
					WSMgrCfgFolderType										= 1010,
					ModifiedWorkspacesFolderType							= 1020,
					ToolsFolderType											= 1030,
					PreferencesRootFolderType								= 1040,
					RequiredLinguisticsFolderType							= 1050,
					TemporaryFolder											= 1060
				};
							
				/**
					\brief		Acquires an instance of the Illustrator class
					\author		GW
					\date		08/2013
					
					\returns	The Illustrator instance object
				*/
				static Illustrator* instance();

				/**
					\brief	Destructs an Illustrator object
					\author	GW
					\date	08/2013
				*/
				~Illustrator();

				/**
					\brief		Gets a pointer to the platform object representing the current Illustrator instance
					\author		GW
					\date		08/2013

					\returns	A pointer to the platform object for the Illustrator instance
				*/
				PlatformAppPtr platformApp() const;
				
				#if defined(WIN_ENV)
					/**
						\brief		Gets a pointer to the platform object representing the main Illustrator window
						\author		GW
						\date		06/2023

						\returns	A pointer to the platform object for the Illustrator main window
					*/
					PlatformAppWindowPtr platformAppWindow() const;
				#endif
				
				/**
					\brief		Allocates a block of memory in Illustrator's memory heap (rather than the caller's
								memory heap); required for data that Illustrator will store for you, or data passed to
								another plugin
					\author		GW
					\date		04/2014
					
					\param		size_	Number of bytes to allocate
					\returns	A pointer to the beginning of the block of memory, or NULL for error
					
					\note		This method is commonly used like:
								SPMessageData* data = (SPMessageData*) HDI_CORE_ILLUSTRATOR->alloc(sizeof(SPMessageData));
				*/
				void* alloc(const int32_t size_) const;
				
				/**
					\brief		Reallocates a block of memory in Illustrator's memory heap (that was previously
								allocated using the Illustrator::alloc() method), while attempting to resize the block
								rather than relocate it.
					\author		GW
					\date		04/2014
					
					\param		block_		Pointer to the block previously allocated with Illustrator::alloc()
					\param		newSize_	The new size of the block, in bytes
					\returns	A pointer to the beginning of a new block of memory, or block_ if the original block
								could be resized, or NULL for error

					\note		If the return value is not equal to block_, then block_ is no longer valid (you must use
								the return value instead).
				*/
				void* realloc(void* const block_, const int32_t newSize_) const;
				
				/**
					\brief	Frees a block of memory previously allocated with the Illustrator::alloc() method
					\author	GW
					\date	04/2014
					
					\param	block_	Pointer to the block of memory to free
				*/
				void free(void* const block_) const;

				/**
					\brief		Gets the name of the registered Illustrator user
					\author		GW
					\date		08/2013

					\returns	The user's name, according to Illustrator, as UTF-8
				*/
				std::string userName() const;

				/**
					\brief		Gets the organization/company name of the registered Illustrator user
					\author		GW
					\date		08/2013

					\returns	The user's company name, according to Illustrator, as UTF-8
				*/
				std::string userOrganization() const;

				/**
					\brief		Gets the serial number of the registered Illustrator user
					\author		GW
					\date		08/2013

					\returns	A "stringified" version of the Illustrator serial number (likely a hash)
				*/
				std::string serialNumber() const;

				/**
					\brief		Gets the Illustrator major version number (e.g. "16" from "16.0.2")
					\author		GW
					\date		08/2013

					\returns	Major version number of Illustrator
				*/
				int32_t majorVersion() const;

				/**
					\brief		Gets the Illustrator minor version number (e.g. "0" from "16.0.2")
					\author		GW
					\date		08/2013

					\returns	Minor version number of Illustrator
				*/
				int32_t minorVersion() const;

				/**
					\brief		Gets the Illustrator revision/bugfix version number (e.g. "2" from "16.0.2")
					\author		GW
					\date		08/2013

					\returns	Revision version number of Illustrator
				*/
				int32_t revisionVersion() const;
				
				/**
					\brief		Gets the bitness of Illustrator at runtime
					\author		GW
					\date		08/2014
					
					\returns	32 or 64
					
					\note		Mac Illustrator versions 16 and above will be 64-bit.
					\note		Windows Illustrator versions 16 and above will be either 32- or 64-bit.
				*/
				int16_t bitness() const;
			
				/**
					\brief		Gets the Illustrator version as a string (e.g. "16.0" for Illustrator 16)
					\author		GW
					\date		08/2013
					
					\param		useRevision_	Whether to show the revision number at the end
					\returns	The full Illustrator version string, e.g. "16.0" with useRevision_ as false or "16.0.2"
								as true
				*/
				std::string versionString(const bool useRevision_ = false) const;
				
				/**
					\brief		Gets the Illustrator version description, a.k.a. the marketed version name
					\author		GW
					\date		09/2014
					
					\returns	"CS6" if the major version is 16, "CC 2013" if the major version is 17, etc.
				*/
				std::string versionDescription() const;

				/**
					\brief		Gets a map of all available suites, keyed on their name
					\author		GW
					\date		04/2014
					
					\returns	A map of all available suites' names and version numbers
				*/
				SuiteInfoMap allSuites() const;
				
				/**
					\brief		Gets a map of all available plugins, keyed on their name
					\author		GW
					\date		04/2014
					
					\returns	A map of all available plugins, keyed on their name (as UTF-8)
					
					\warning	The caller assumes responsibility of the memory for the ThirdPartyPlugin objects in the
								map. Be sure to iterate over the map and delete the objects when you're finished with it.
				*/
				ThirdPartyPluginMap allPlugins() const;
				
				/**
					\brief		Gets a specific plugin by its name (must be an exact match)
					\author		GW
					\date		04/2014
					
					\param		name_	Exact name of the plugin in question, as UTF-8
					\returns	A smart pointer for the plugin (will contain NULL if the plugin isn't found)
				*/
				std::unique_ptr<ThirdPartyPlugin> pluginWithName(const std::string& name_) const;

				/**
					\brief		Convenience method to return the ai::Preferences instance
					\author		GW
					\date		07/2014

					\returns	The Illustrator preferences instance
				*/
				ai::Preferences* const preferences() const;

				/**
					\brief		Acquires an instance of the CurrentDocument class for the current document
					\author		GW
					\date		08/2013
					
					\returns	A pointer to the current document instance
				*/
				CurrentDocument* const currentDocument();

				/**
					\brief		Gets the open document count
					\author		GW
					\date		08/2013

					\returns	Number of documents currently open in Illustrator
				*/
				uint32_t documentCount() const;

				/**
					\brief		Gets an open document by its index
					\author		GW
					\date		08/2013

					\param		index_	Index number of the document
					\returns	The document indicated by the index
				*/
				std::unique_ptr<Document> documentAtIndex(const uint32_t index_) const;

				/**
					\brief		Opens a document and returns it
					\author		GW
					\date		08/2013

					\param		path_			Path to the file, as UTF-8
					\param		colorModel_		Color mode for the file
					\param		showDialog_		Whether to show the new document dialog to query the user for parameters
					\param		forceCopy_		If true, duplicates the file and names the copy "Untitled"
					\returns	The opened document, or NULL for none
				*/
				std::unique_ptr<Document> openDocument(
					const std::string& path_,
					const DocumentColorModel colorModel_,
					const bool showDialog_,
					const bool forceCopy_ = false
				) const;

				/**
					\brief		Causes the OS's default browser to open a given URL
					\author		GW
					\date		08/2013

					\param		url_	URL to open, as UTF-8
					\returns	true if the URL is opened, false otherwise
				*/
				bool openURL(const std::string& url_) const;

				/**
					\brief	Caches all the available fonts in an internal map until clearFontCache() is called
					\author	GW
					\date	07/2014
					
					\note	Acquiring fonts by name in Illustrator is notoriously slow. If you expect to be using the
							fontWithName() method a lot in a short period of time, it would be worthwhile to cache the
							fonts beforehand. Remember to call clearFontCache() when you are done.
				*/
				void cacheFonts();
				
				/**
					\brief	Clears fonts previously cached by the cacheFonts() method
					\author	GW
					\date	07/2014
				*/
				void clearFontCache();

				/**
					\brief		Gets the count of available fonts
					\author		GW
					\date		08/2013
					
					\returns	Count of available fonts
				*/
				uint32_t fontCount() const;

				/**
					\brief		Gets a Font object via its index
					\author		GW
					\date		08/2013

					\param		index_	Index number of the desired font
					\returns	The font at the given index
				*/
				std::unique_ptr<Font> fontAtIndex(const uint32_t index_) const;

				/**
					\brief		Gets a Font object via its name (case-insensitive)
					\author		GW
					\date		08/2013

					\param		faceName_	Name of the desired font, as UTF-8
					\returns	The font with the given name
				*/
				std::unique_ptr<Font> fontWithName(const std::string& faceName_) const;

				/**
					\brief		Gets a vector containing Font objects for all available fonts
					\author		GW
					\date		08/2013

					\returns	A vector containing Font objects for all available fonts
				*/
				FontVector allFonts() const;
				
				/**
					\brief		Gets the count of available typefaces
					\author		GW
					\date		08/2014
					
					\returns	Count of available typefaces
				*/
				uint32_t typefaceCount() const;
				
				/**
					\brief		Gets a Typeface object via its index
					\author		GW
					\date		08/2014
					
					\param		index_	Index number of the desired typeface
					\returns	The typeface at the given index
				*/
				std::unique_ptr<Typeface> typefaceAtIndex(const uint32_t index_) const;

				/**
					\brief		Gets the current locale of Illustrator
					\author		GW
					\date		08/2013

					\returns	Illustrator's locale
				*/
				Locale locale() const;
				
				/**
					\brief		Gets a string describing the current locale of Illustrator
					\author		GW
					\date		08/2013
					
					\returns	Illustrator locale description string
				*/
				std::string localeString() const;

				/**
					\brief		Gets the count of available file formats
					\author		GW
					\date		12/2014
					
					\returns	Count of available file formats
				*/
				uint32_t fileFormatCount() const;

				/**
					\brief		Gets a file format object via its index
					\author		GW
					\date		12/2014

					\param		index_	Index number of the desired file format
					\returns	The file format at the given index
				*/
				std::unique_ptr<FileFormat> fileFormatAtIndex(const uint32_t index_) const;
				
				/**
					\brief		Registers a new file format with Illustrator, which will be displayed in Illustrator's
								file dialog
					\author		GW
					\date		12/2014
					
					\param		ff_		New file format to be registered
					\returns	true if the format was registered, false otherwise
				*/
				bool registerFileFormat(FileFormat& ff_);

				/**
					\brief		Gets the count of available live effects
					\author		GW
					\date		08/2014
					
					\returns	Count of available live effects
				*/
				uint32_t liveEffectCount() const;

				/**
					\brief		Gets a live effect object via its index
					\author		GW
					\date		08/2014

					\param		index_	Index number of the desired live effect
					\returns	The live effect at the given index
				*/
				std::unique_ptr<LiveEffect> liveEffectAtIndex(const uint32_t index_) const;

				/**
					\brief		Gets the count of menu groups in Illustrator (at all levels)
					\author		GW
					\date		09/2014
					
					\returns	Count of available menu groups
				*/
				uint32_t menuGroupCount() const;
				
				/**
					\brief		Gets a menu group object via its index
					\author		GW
					\date		09/2014

					\param		index_	Index number of the desired menu group
					\returns	The menu group at the given index
					
					\note		It's worth noting that most groups, due to Illustrator's implementation of "groups", are
								not actually real UI menu groups at all. Most are just a specific region of some top-
								level menu or other subgroup. This can be a bit confusing, as some "groups" do not
								actually behave like proper groups (e.g. their parent cannot be acquired, text value
								cannot be acquired, etc). See also the isPartial() method in the MenuGroup class.
				*/
				std::unique_ptr<MenuGroup> menuGroupAtIndex(const uint32_t index_) const;

				/**
					\brief	Suspends the existing Illustrator app context, storing it for later resuming
					\author	GW
					\date	08/2013
				*/
				void suspendExistingContext();

				/**
					\brief	Resumes a previously suspended Illustrator app context
					\author	GW
					\date	08/2013
				*/
				void resumeSuspendedContext();

				/**
					\brief	Pushes a new Illustrator app context onto the stack
					\author	GW
					\date	08/2013
				*/
				void pushContext();

				/**
					\brief	Pops a previously pushed Illustrator app context from the stack
					\author	GW
					\date	08/2013
				*/
				void popContext();

				/**
					\brief		Checks whether an Illustrator app context currently exists
					\author		GW
					\date		11/2016

					\returns	true if currently within an Illustrator app context, false otherwise
				*/
				bool hasContext() const;

				/**
					\brief		Gets the folder path for a given illustrator folder
					\author		GW
					\date		08/2013

					\param		folder_		The Illustrator folder type whose path is desired
					\returns	A string for the path to the Illustrator folder type in question, as UTF-8
				*/
				std::string folderPath(const FolderType folder_) const;

				/**
					\brief		Checks whether a given file exists within one of Illustrator's support folders
					\author		GW
					\date		08/2013

					\param		folder_		The illustrator folder type to look in
					\param		fileName_	Name of file whose existence (in folder_) is in question, as UTF-8
					\returns	true if a file with the provided name exists in folder_, false otherwise
				*/
				bool checkFolderForFile(const FolderType folder_, const std::string& fileName_) const;

				/**
					\brief		Verifies that a file is located in one of the Illustrator plugins folders
					\author		GW
					\date		08/2013

					\param		fileName_	Name of the file (e.g. "MyPlugin.aip"), as UTF-8
					\returns	true if the file exists in an Illustrator plugins folder, false otherwise
				*/
				bool existsInPluginsFolder(const std::string& fileName_) const;

				/**
					\brief		Verifies that a file is located in the Illustrator tools folder
					\author		GW
					\date		08/2013

					\param		fileName_	Name of the file (e.g. "MyToolsPlugin.aip"), as UTF-8
					\returns	true if the file exists in the Illustrator tools folder, false otherwise
				*/
				bool existsInToolsFolder(const std::string& fileName_) const;
				
				/**
					\brief		Displays to the user a prompt to choose a location and file name for some data to be
								written to disk
					\author		GW
					\date		02/2015
					
					\param		title_		Title of the dialog, as UTF-8
					\param		fileName_	Default file name, as UTF-8
					\param		path__		Return-by-reference for the path to the file to be saved, as UTF-8
					\returns	true if the user completed the prompt, false if they canceled it
				*/
				bool fileSavePrompt(const std::string& title_, const std::string& fileName_, std::string& path__) const;
				
				/**
					\brief		Displays to the user a prompt to choose a file to be read from disk
					\author		GW
					\date		02/2015
					
					\param		title_	Title of the dialog, as UTF-8
					\param		path__	Return-by-reference for the path to the file to be read, as UTF-8
					\returns	true if the user completed the prompt, false if they canceled it
				*/
				bool fileOpenPrompt(const std::string& title_, std::string& path__) const;
				
				/**
					\brief		Displays to the user a prompt to choose a folder on disk
					\author		GW
					\date		02/2015
					
					\param		title_		Title of the dialog, as UTF-8
					\param		path__		Return-by-reference for the path of the folder, as UTF-8
					\returns	true if the user completed the prompt, false if they canceled it
				*/
				bool folderPrompt(const std::string& title_, std::string& path__) const;

				/**
					\brief		Gets which built-in toolbox(es) is/are currently showing
					\author		GW
					\date		09/2018

					\returns	A bitwise OR'd combination of values from the BuiltinToolbox enum
				*/
				BuiltinToolbox visibleBuiltinToolboxes() const;

				/**
					\brief		Gets whether the current Illustrator UI theme color is considered dark
					\author		GW
					\date		08/2013

					\returns	true if the theme is dark, false if light (dark meaning the background color is dark)
				*/
				bool uiThemeDark() const;

				/**
					\brief		Gets the current color for a given widget component
					\author		GW
					\date		08/2013

					\param		window_		Indicates whether the component is within a panel (or control bar) or
											modal dialog
					\param		component_	The type of widget, in window_, whose color is needed
					\returns	Ideal color values for the window_ and component_ combo
				*/
				Color uiThemeColor(const ThemeWindowType window_, const ThemeComponentType component_) const;
			
				/**
					\brief		Gets whether the Illustrator UI should be scaled at all (reads the Illustrator pref for
								this purpose, if necessary/available)
					\author		GW
					\date		02/2017
					
					\returns	true if the Illustrator UI should be scaled for HiDPI displays, false otherwise
					
					\note		This method always returns true on Mac
					\note		As this pref can be changed at runtime, but the UI does not actually update to reflect
								the new setting until relaunch, the pref value is read at startup and cached for the
								duration of the plugin's lifecycle.
				*/
				bool uiShouldScale() const;

				/**
					\brief		Gets the factor by which the Illustrator UI is scaled
					\author		GW
					\date		08/2018
				 
					\returns	Illustrator's UI scaling factor
				 
					\note		This is independent of the operating system's UI scaling factor. For example, on a
								Mac with a Retina display, the OS scaling factor will likely be 2.0; however,
								Illustrator could have a scaling factor of 1.5 itself - this results in a final
								scaling factor of 3.0. In that case, this method will return 1.5 (to find the scaling
								factor of a specific window you must call the associated method for said window,
								after which you can multiply the two numbers together to determine the final scaling
								factor).
				*/
				double uiScalingFactor() const;


			private:
				/**
					\brief	Singleton instance for this class
				*/
				static Illustrator* __instance;

				/**
					\brief	Private implementation data
				*/
				void* __data;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				Illustrator();
				
				/**
					\brief	Unused
					\author	GW
					\date	01/2015
				*/
				Illustrator(const Illustrator&);
				
				/**
					\brief	Unused
					\author	GW
					\date	01/2015
				*/
				Illustrator& operator=(const Illustrator&);
		};
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_ILLUSTRATOR__
