/**
	\file
	\brief		Header file for basic internet HTTP(S) POST requests
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_INTERNET_POST__
#define __HDI_CORE_INTERNET_POST__

#include <string>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace plat
	{
		class InternetPOST;
	}

	namespace core
	{
		class Callback;
		class DictionaryPref;

		/**
			\brief	Allows for the creation of simple HTTP(S) POST clients to post to, and acquire data from, a given URL
		*/
		class InternetPOST
		{
			public:
				/**
					\brief		Creates a synchronous (blocking) InternetPOST object to perform HTTP(S) POST requests, in
								a "send and forget" fashion
					\author		GW
					\date		09/2019
				 
					\param		url_	HTTP(S) URL to read, as UTF-8
					\param		data_	Dictionary of data to be JSON-encoded and sent to url_; root-level members of
										the dictionary will be used as individual POST arguments, and the member values
										will be individually JSON-encoded as POST argument values
				 
					\warning	As this method doesn't take any callback arguments, the request will simply be sent and
				 				the caller will never be notified of it finishing/failing.
				*/
				InternetPOST(const std::string& url_, const DictionaryPref& data_);
			
				/**
					\brief	Creates a synchronous (blocking) InternetPOST object to perform HTTP(S) POST requests
					\author	GW
					\date	08/2014
				 
					\param	url_		HTTP(S) URL to read, as UTF-8
					\param	data_		Dictionary of data to be JSON-encoded and sent to url_; root-level members of
										the dictionary will be used as individual POST arguments, and the member values
										will be individually JSON-encoded as POST argument values
					\param	success_	Callback to perform when the request completes successfully
					\param	failure_	Callback to perform when the request fails
				*/
				InternetPOST(
					const std::string& url_,
					const DictionaryPref& data_,
					const Callback& success_,
					const Callback& failure_
				);
			
				/**
					\brief		Creates an asynchronous (non-blocking) InternetPOST object to perform HTTP(S) POST
								requests
					\author		GW
					\date		09/2019
				 
					\param		url_		HTTP(S) URL to read, as UTF-8
					\param		data_		Dictionary of data to be JSON-encoded and sent to url_; root-level members
											of the dictionary will be used as individual POST arguments, and the member
											values will be individually JSON-encoded as POST argument values
					\returns	A new InternetPOST object that can execute its HTTP request asynchronously
				 
					\warning	As this method doesn't take any callback arguments, the request will simply be sent and
				 				the caller will never be notified of it finishing/failing.
					\warning	The responsibility for the memory of the returned InternetPOST pointer lies with the
								caller, and it must not be deleted until after either the success or failure callbacks
								are called.

					\note		Asynchronous InternetPOST objects must be allocated using operator new and have several
								internal flags set (only synchronous InternetPOST objects can be allocated on the stack),
								which this function does automatically.
				*/
				static InternetPOST* Async(const std::string& url_, const DictionaryPref& data_);
			
				/**
					\brief		Creates an asynchronous (non-blocking) InternetPOST object to perform HTTP(S) POST
								requests
					\author		GW
					\date		08/2014
				 
					\param		url_		HTTP(S) URL to read, as UTF-8
					\param		data_		Dictionary of data to be JSON-encoded and sent to url_; root-level members
											of the dictionary will be used as individual POST arguments, and the member
											values will be individually JSON-encoded as POST argument values
					\param		success_	Callback to perform when the request completes successfully
					\param		failure_	Callback to perform when the request fails
					\returns	A new InternetGET object that can execute its HTTP request asynchronously
				 
					\warning	The responsibility for the memory of the returned InternetPOST pointer lies with the
								caller, and it must not be deleted until after either the success or failure callbacks
								are called.

					\note		Asynchronous InternetPOST objects must be allocated using operator new and have several
								internal flags set (only synchronous InternetPOST objects can be allocated on the stack),
								which this function does automatically.
					\note		The callbacks will not be called if their scheduling comes during or after plugin
								pre-shutdown (this would lead to a crash on quit).
				*/
				static InternetPOST* Async(
					const std::string& url_,
					const DictionaryPref& data_,
					const Callback& success_,
					const Callback& failure_
				);

				/**
					\brief	Destructs an InternetPOST object
					\author	GW
					\date	08/2014
				*/
				virtual ~InternetPOST();
				
				/**
					\brief		Gets whether the target object is asynchronous (i.e. executes in a background thread)
					\author		GW
					\date		08/2014
					
					\returns	true for asynchronous, false for synchronous
				*/
				virtual bool asynchronous() const;
				
				/**
					\brief	Starts the POST request
					\author	GW
					\date	08/2014
				*/
				virtual void run();
				
				/**
					\brief		Gets whether the POST request is currently running
					\author		GW
					\date		08/2014
					
					\returns	true if the request is running, false otherwise
				*/
				virtual bool running() const;
				
				/**
					\brief		Gets the content acquired via the POST request
					\author		GW
					\date		08/2014
					
					\returns	The resultant content of the network request, encoded in a string of bytes
					
					\note		The std::string return type is slightly misleading; you are not limited to requesting
								string data. Requesting binary data is perfectly acceptable, as the std::string return
								type is merely a convenient container.
					\note		The results of this method are only valid when it is called from the "success" callback
								(see the InternetPOST constructor)
				*/
				virtual std::string content() const;
			
				/**
					\brief		Gets a translated platform error code, if possible, that was received by the implementation
					\author		GW
					\date		06/2017
					
					\returns	The translated platform error code

					\note		The results of this method are only valid when it is called from the "failure" callback
								(see the InternetGET constructor).
				*/
				virtual InternetError errorCode() const;
			
				/**
					\brief		Gets the raw platform error code that was received by the implementation
					\author		GW
					\date		06/2017
					
					\returns	The translated platform error code
					
					\note		See the file NSURLError.h on Mac, and WinINet.h on Windows, for a list of possible platform
								error codes.
					\note		The results of this method are only valid when it is called from the "failure" callback
								(see the InternetGET constructor).
				*/
				virtual int32_t platformErrorCode() const;
				
				/**
					\brief		Gets an error description if the POST request failed
					\author		GW
					\date		08/2014
					
					\returns	The resultant error of the request, as UTF-8

					\note		The results of this method are only valid when it is called from the "failure" callback
								(see the InternetPOST constructor).
				*/
				virtual std::string error() const;
				

			private:
				/**
					\brief	Private implementation object
				*/
				plat::InternetPOST* __impl;
				
				/**
					\brief	Empty InternetPOST objects cannot be constructed
					\author	GW
					\date	08/2014
				*/
				InternetPOST();
			
				/**
					\brief	InternetPOST objects cannot be copied
					\author	GW
					\date	08/2014
				*/
				InternetPOST(const InternetPOST&);
			
				/**
					\brief	InternetPOST objects cannot be copied
					\author	GW
					\date	08/2014
				*/
				InternetPOST& operator=(const InternetPOST&);
		};
	}
}

#endif
// __HDI_CORE_INTERNET_POST__
