/**
	\file
	\brief		Header file for plugin user interface label widgets
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_LABEL__
#define __HDI_CORE_LABEL__

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		class Color;
		class TextField;
		class TextView;
		
		#if defined(HDI_CORE_AIP_MODE)
			class Panel;
		#endif

		/**
			\brief	Label widget, for simply displaying text or images inside a window
		*/
		class Label : public Widget
		{
			public:
				friend class TextField;
				friend class TextView;
			
				/**
					\brief	Text alignment options
				*/
				enum Alignment
				{
					UnknownAlignment		= 0,
					LeftAlignment			= 10,
					CenterAlignment			= 20,
					RightAlignment			= 30
				};
			
				/**
					\brief	Background frame shape drawing options
				*/
				enum FrameShape
				{
					NoFrameShape			= 0,
					RectangleFrameShape		= 10,
					RoundedRectFrameShape	= 20,
					CapsuleFrameShape		= 30
				};

				/**
					\brief	Constructs an empty Label object
					\author	GW
					\date	09/2013
					
					\note	To test if a Label object is empty, call isEmpty() on it
					\note	Empty Label objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other Label object via the overloaded assignment operator. Empty Label
							objects are useless until such time (though it is safe to call any of their methods).
				*/
				Label();

				/**
					\brief	Constructs a new Label object from an existing Label object (copy constructor)
					\author	GW
					\date	09/2013

					\param	label_	Existing Label object
				*/
				Label(const Label& label_);

				/**
					\brief	Constructs a static text label
					\author	GW
					\date	09/2013

					\param	loc_			Top-left location of the label, in 1x resolution coordinates
					\param	title_			Title of the label, as UTF-8
					\param	containerType_	Type of window that will contain the widget
					
					\note	The containerType_ argument is necessary for this constructor because it computes the ideal
							width for the new label from the initial label title string. However, Adobe uses different
							font sizes for different types of windows, and we can't correctly compute the width without
							knowing what font size to use.
				*/
				Label(const Point& loc_, const std::string& title_, const WindowType containerType_);

				/**
					\brief	Constructs a static text label
					\author	GW
					\date	09/2013

					\param	loc_	Top-left location of the label, in 1x resolution coordinates
					\param	title_	Title of the label, as UTF-8
					\param	width_	Width of the label, at 1x resolution
				*/
				Label(const Point& loc_, const std::string& title_, const double width_);

				/**
					\brief	Constructs a static image label
					\author	GW
					\date	09/2013

					\param	frame_		Frame of the label, in 1x resolution coordinates
					\param	imageIDs_	PNGI resource IDs
				*/
				Label(const Rect& frame_, const ImageWithRolloverIDs& imageIDs_);

				/**
					\brief	Constructs a static image label
					\author	GW
					\date	12/2018

					\param	frame_		Frame of the label, in 1x resolution coordinates
					\param	svgIDs_		SVG resource IDs
					\param	dims_		1x dimensions to use when rendering the SVG; the default value causes the widget
										frame dimensions to be used instead
				*/
				Label(const Rect& frame_, const SVGWithRolloverIDs& svgIDs_, const Size& dims_ = Size());

				/**
					\brief	Label destructor
					\author	GW
					\date	09/2013
				*/
				virtual ~Label();

				/**
					\brief		Allows one Label object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target Label object, but with its value updated to match that of rhs_
				*/
				virtual Label& operator=(const Label& rhs_);
				
				/**
					\brief		Convenience method to clone a Label object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new Label object
					
					\note		If you subclass Label, you MUST overload this method yourself! If you don't and/or your
								clone() method does not return an instance of your Label subclass, you will experience
								"object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned Label object.
				*/
				virtual Label* clone() const;
				
				/**
					\brief		Convenience method to duplicate a Label object, creating a new and identical UI element
								to the target (but not belonging to the same containing window)
					\author		GW
					\date		11/2013
					
					\returns	A pointer to the new Label object (and new UI element)
					
					\warning	The caller must manage the memory for the returned Label object.
				*/
				virtual Label* duplicate() const;

				/**
					\brief		Gets the current text value of the label
					\author		GW
					\date		09/2013
					
					\returns	The current text value, as UTF-8

					\note		If the widget is an image label, this returns ""
				*/
				virtual std::string text() const;

				/**
					\brief	Sets the current text value of the label
					\author	GW
					\date	09/2013

					\param	text_	New text value, as UTF-8

					\note	If the widget is an image label, the function will bail early
				*/
				virtual void setText(const std::string& text_);

				/**
					\brief		Gets the current alignment of the label
					\author		GW
					\date		09/2013

					\returns	Current text alignment
				*/
				virtual Alignment alignment() const;

				/**
					\brief	Sets the current alignment of the label
					\author	GW
					\date		09/2013

					\param	alignment_	New text alignment
				*/
				virtual void setAlignment(const Alignment alignment_);
			
				/**
					\brief		Gets whether the text is bold
					\author		GW
					\date		09/2013

					\returns	true if the text is bold, false otherwise
				*/
				virtual bool bold() const;

				/**
					\brief	Sets the font weight
					\author	GW
					\date	09/2013

					\param	bold_	true for bold, false otherwise

					\note	If the widget is an image label, the function will bail early
				*/
				virtual void setBold(const bool bold_);

				/**
					\brief		Gets whether the text is italicized
					\author		GW
					\date		09/2013

					\returns	true if the text is italicized, false otherwise
				*/
				virtual bool italic() const;

				/**
					\brief	Sets the font style
					\author	GW
					\date	09/2013

					\param	italic_		true for italics, false otherwise

					\note	If the widget is an image label, the function will bail early
				*/
				virtual void setItalic(const bool italic_);
			
				/**
					\brief		Gets the shape for the background of the label frame
					\author		GW
					\date		09/2018

					\returns	Current label background frame shape
				 
					\note		Frame shapes only apply to text labels.
				*/
				virtual FrameShape frameShape() const;
			
				/**
					\brief	Sets the shape for the background of the label frame
					\author	GW
					\date	09/2018

					\param	shape_	New label background frame shape (use FrameShapeNone for no background drawing)
				 
					\note	Frame shapes only apply to text labels.
				*/
				virtual void setFrameShape(const FrameShape shape_);
			
				/**
					\brief		Gets the colors for the background of the label frame
					\author		GW
					\date		09/2018

					\param		light__	Return-by-reference for the current label background frame light-UI color
					\param		dark__	Return-by-reference for the current label background frame dark-UI color
				 
					\note		Frame colors only apply to text labels.
				*/
				virtual void frameColors(Color& light__, Color& dark__) const;
			
				/**
					\brief	Sets the colors for the background of the label frame
					\author	GW
					\date	09/2018

					\param	light_	New label background frame light-UI color
					\param	dark_	New label background frame dark-UI color
				 
					\note	Frame colors only apply to text labels.
					\note	These colors should be ignored if the frameShape() method returns FrameShapeNone.
				*/
				virtual void setFrameColors(const Color& light_, const Color& dark_);
			
				/**
					\brief		Gets the colors for the text of the label
					\author		GW
					\date		09/2018

					\param		light__	Return-by-reference for the current label text light-UI color
					\param		dark__	Return-by-reference for the current label text dark-UI color
					\returns	true if the default colors have been overridden (and the arguments have been filled
								appropriately), false otherwise
				 
					\note		Text colors only apply to text labels.
				*/
				virtual bool textColors(Color& light__, Color& dark__) const;
			
				/**
					\brief	Sets the colors for the text of the label
					\author	GW
					\date	09/2018

					\param	light_	New label text light-UI color
					\param	dark_	New label text dark-UI color
				 
					\note	Text colors only apply to text labels.
				*/
				virtual void setTextColors(const Color& light_, const Color& dark_);
				
				/**
					\brief		Gets the text field that the target label has been associated with
					\author		GW
					\date		04/2014
					
					\returns	Text field that will be focused when the label is clicked
				*/
				virtual std::unique_ptr<TextField> textField() const;
				
				/**
					\brief	Sets the text field that should be focused when the label is clicked
					\author	GW
					\date	04/2014
					
					\param	field_	New field to focus when the label is clicked
					
					\note	Will remove any previously linked text view or panel.
				*/
				virtual void setTextField(TextField& field_);
				
				/**
					\brief	Removes the text field that should be focused when the label is clicked
					\author	GW
					\date	04/2014
				*/
				virtual void removeTextField();
				
				/**
					\brief		Gets the text view that the target label has been associated with
					\author		GW
					\date		04/2014
					
					\returns	Text view that will be focused when the label is clicked
				*/
				virtual std::unique_ptr<TextView> textView() const;
				
				/**
					\brief	Sets the text view that should be focused when the label is clicked
					\author	GW
					\date	04/2014
					
					\param	view_	New view to focus when the label is clicked
					
					\note	Will remove any previously linked text field or panel.
				*/
				virtual void setTextView(TextView& view_);
				
				/**
					\brief	Removes the text view that should be focused when the label is clicked
					\author	GW
					\date	04/2014
				*/
				virtual void removeTextView();
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the panel that the target label has been associated with
						\author		GW
						\date		12/2015
						
						\returns	Panel that will be shown in a popdown borderless fashion when the label is clicked
					*/
					virtual std::unique_ptr<Panel> linkedPanel() const;
					
					/**
						\brief	Sets the panel that should be shown when the label is clicked
						\author	GW
						\date	12/2015
						
						\param	p_	New panel to show in a popdown borderless fashion when the label is clicked
						
						\note	Will remove any previously linked text field or text view.
					*/
					virtual void setLinkedPanel(const Panel& p_);
				
					/**
						\brief	Removes the panel that should be shown when the label is clicked
						\author	GW
						\date	12/2015
					*/
					virtual void removeLinkedPanel();
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief	Simulates a click on the label
					\author	GW
					\date	09/2013
				*/
				virtual void click();
				
				/**
					\brief		Gets the current click callback for the label
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered click callback
				*/
				virtual Callback* const clickCallback() const;
				
				/**
					\brief	Sets the click callback for the label
					\author	GW
					\date	09/2013

					\param	callback_	New callback for when the label is clicked
				*/
				virtual void setClickCallback(const Callback& callback_);

				/**
					\brief	Simulates a double-click on the label
					\author	GW
					\date	03/2015
				*/
				virtual void doubleClick();
				
				/**
					\brief		Gets the current double-click callback for the label
					\author		GW
					\date		03/2015

					\returns	A pointer to the currently registered double-click callback
				*/
				virtual Callback* const doubleClickCallback() const;
				
				/**
					\brief	Sets the double-click callback for the label
					\author	GW
					\date	03/2015

					\param	callback_	New callback for when the label is double-clicked
				*/
				virtual void setDoubleClickCallback(const Callback& callback_);
			
				/**
					\brief		Gets the PNGI resource IDs of the widget
					\author		GW
					\date		03/2015

					\returns	The IDs of the PNGI image resources
				*/
				virtual ImageWithRolloverIDs imageIDs() const;
			
				/**
					\brief	Sets the PNGI resource IDs of the widget
					\author	GW
					\date	03/2015

					\param	ids_	New PNGI image resource IDs
				*/
				virtual void setImageIDs(const ImageWithRolloverIDs& ids_);
			
				/**
					\brief		Gets the SVG resource IDs of the widget
					\author		GW
					\date		12/2018

					\returns	The IDs of the SVG image resources
				*/
				virtual SVGWithRolloverIDs svgIDs() const;
			
				/**
					\brief	Sets the SVG resource IDs of the widget
					\author	GW
					\date	12/2018

					\param	ids_	New SVG image resource IDs
				*/
				virtual void setSVGIDs(const SVGWithRolloverIDs& ids_);
				
				/**
					\brief		Gets whether the background of the label should change in appearance upon mouse-over
					\author		GW
					\date		03/2015
					
					\returns	true for showing a rollover effect, false otherwise
				*/
				virtual bool backgroundRollover() const;
				
				/**
					\brief	Sets whether the background of the label should change in appearance upon mouse-over
					\author	GW
					\date	03/2015
					
					\param	ro_		true to show a rollover effect, false otherwise
				*/
				virtual void setBackgroundRollover(const bool ro_);
			
				/**
					\brief		Gets whether dragging support for this particular widget is enabled
					\author		GW
					\date		09/2018
				 
					\returns	true if drag-and-drop source support is enabled, false otherwise
				*/
				virtual bool draggable() const;
			
				/**
					\brief	Sets whether dragging support for this particular widget is enabled
					\author	GW
					\date	09/2018
				 
					\param	enable_		true to enable drag-and-drop source support, false otherwise
				*/
				virtual void setDraggable(const bool enable_);
			
				/**
					\brief		Gets the data that will be passed when the widget is dropped on a receiver
					\author		GW
					\date		09/2018
				 
					\returns	The droppable data
				 
					\note	A value of "" (empty string; the default) will result in the text value of the widget itself
							being the drag data.
				*/
				virtual std::string dragData() const;
			
				/**
					\brief	Sets the data that will be passed when the widget is dropped on a receiver
					\author	GW
					\date	09/2018
				 
					\param	data_	Data to be passed to a receiving drag-and-drop widget (when dropped)
				 
					\note	Passing a value of "" (empty string; the default) will result in the text value of the widget
							itself being the drag data.
				*/
				virtual void setDragData(const std::string& data_);
			
				/**
					\brief		Gets whether dropping support for this particular widget is enabled
					\author		GW
					\date		09/2018
				 
					\returns	true if drag-and-drop destination support is enabled, false otherwise
				*/
				virtual bool droppable() const;
			
				/**
					\brief	Sets whether dropping support for this particular widget is enabled
					\author	GW
					\date	09/2018
				 
					\param	enable_		true to enable drag-and-drop destination support, false otherwise
				*/
				virtual void setDroppable(const bool enable_);
			
				/**
					\brief		Gets the current drop callback for the widget
					\author		GW
					\date		09/2018

					\returns	A pointer to the currently registered drop callback
				*/
				virtual Callback* const dropCallback() const;
			
				/**
					\brief	Sets the drop callback for the widget
					\author	GW
					\date	09/2018

					\param	callback_	New callback for when the widget receives a drop
				*/
				virtual void setDropCallback(const Callback& callback_);
			
				/**
					\brief		Gets the data that was passed when the widget received a drop
					\author		GW
					\date		09/2018
				 
					\returns	The dropped data
				 
					\warning	The result of this method is only valid when it is called from within a drop callback.
				*/
				virtual std::string droppedData() const;
			
				/**
					\brief		Gets the point at which the drop occurred, in the widget's coordinate system
					\author		GW
					\date		09/2018
				 
					\returns	The point at which the drop occurred
				 
					\warning	The result of this method is only valid when it is called from within a drop callback.
				*/
				virtual Point droppedPoint() const;
			
				/**
					\brief		Gets the ideal height for a label, given its value and a known width
					\author		GW
					\date		09/2013

					\param		text_			Value for the label, as UTF-8
					\param		width_			Known width into which the label must fit
					\param		containerType_	Type of window for the widget (because font sizes differ)
					\returns	Ideal height for the given text to fit into the given width, at 1x resolution
				*/
				static double idealHeight(const std::string& text_, const double width_, const WindowType containerType_);

				/**
					\brief		Gets the ideal width for a label, given its value and a known height
					\author		GW
					\date		09/2013

					\param		text_			Value for the label, as UTF-8
					\param		height_			Known height into which the label must fit
					\param		containerType_	Type of window for the widget (because font sizes differ)
					\returns	Ideal width for the given text to fit into the given height, at 1x resolution
				*/
				static double idealWidth(const std::string& text_, const double height_, const WindowType containerType_);
		};
		
		typedef std::unique_ptr<Label> LabelUP;
		typedef std::shared_ptr<Label> LabelSP;
		typedef std::weak_ptr<Label> LabelWP;
	}
}

#endif
// __HDI_CORE_LABEL__
