/**
	\file
	\brief		Header file for general application layer manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_LAYER__
#define __HDI_CORE_LAYER__

#include "hdicoreTypes.h"
#include "hdicoreColor.h"
#include "hdicoreRect.h"

namespace hdi
{
	#if defined(HDI_CORE_AIP_MODE)
		namespace aip
		{
			class Layer;
		}
	#elif defined(HDI_CORE_PSP_MODE)
		namespace psp
		{
			class Layer;
		}
	#endif

	namespace core
	{
		#if defined(HDI_CORE_AIP_MODE)
			class Art;
			class Artboard;
		#endif

		/**
			\brief	Allows for layer metadata access, visibility manipulation, locking, etc.
		*/
		class Layer
		{
			public:
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief	Allows for iteration through the siblings of a layer, a la std::vector iteration

						\code
							// Here's an example, iterating through all top-level layers of a document:
							std::unique_ptr<hdi::core::Layer> layer0 = HDI_CORE_ILLUSTRATOR->currentDocument()->firstLayer();
							for(hdi::core::Layer::Iterator iter = layer0->begin(); !iter.atEnd(); ++iter)
							{
								// Do stuff with iter.layer() ...
							}
							
							// Here's another example, reverse iterating through all top-level layers of a document:
							std::unique_ptr<hdi::core::Layer> layerN = HDI_CORE_ILLUSTRATOR->currentDocument()->lastLayer();
							for(hdi::core::Layer::Iterator iter = layerN->begin(); !iter.atEnd(); --iter)
							{
								// Do more stuff with iter.layer() ...
							}
						\endcode
					*/
					class Iterator
					{
						public:
							/**
								\brief	Constructs an Iterator object from an existing Iterator object (copy constructor)
								\author	GW
								\date	10/2013

								\param	iter_	Existing Iterator object
							*/
							Iterator(const Iterator& iter_);
						
							/**
								\brief	Destructs an Iterator object
								\author	GW
								\date	08/2013
							*/
							~Iterator();
							
							/**
								\brief		Assigns one Iterator object to another
								\author		GW
								\date		10/2013

								\param		rhs_	Existing Iterator object to copy values from
								\returns	The target Iterator object, but with its values updated to match that of the
											rhs_ argument
							*/
							Iterator& operator=(const Iterator& rhs_);
							
							/**
								\brief		Tests whether a given Iterator object is the same as another
								\author		GW
								\date		10/2013

								\param		rhs_	Iterator to compare against (righthand side of equality operator)
								\returns	true for the target and rhs_ having the same state, false otherwise
							*/
							bool operator==(const Iterator& rhs_) const;
							
							/**
								\brief		Tests whether a given Iterator object is not the same as another
								\author		GW
								\date		10/2013

								\param		rhs_	Iterator to compare against (righthand side of inequality operator)
								\returns	true for the target and rhs_ having differing states, false otherwise
							*/
							bool operator!=(const Iterator& rhs_) const;

							/**
								\brief		Prefix increment operator for forward iteration
								\author		GW
								\date		08/2013
								
								\returns	A reference to the iterator after its position was changed
							*/
							Iterator& operator++();

							/**
								\brief		Postfix increment operator for forward iteration
								\author		GW
								\date		08/2013
								
								\returns	A copy of the original iterator before its position was changed
							*/
							Iterator operator++(int);

							/**
								\brief		Prefix decrement operator for reverse iteration
								\author		GW
								\date		08/2013
								
								\returns	A reference to the iterator after its position was changed
							*/
							Iterator& operator--();

							/**
								\brief		Postfix decrement operator for reverse iteration
								\author		GW
								\date		08/2013
								
								\returns	A copy of the original iterator before its position was changed
							*/
							Iterator operator--(int);

							/**
								\brief		Determines whether iteration should stop; works well as the conditional in a
											for() loop
								\author		GW
								\date		08/2013
								
								\returns	true if at the end of the list of siblings, false otherwise
							*/
							bool atEnd() const;

							/**
								\brief		Gets the current layer pointed to by the iterator
								\author		GW
								\date		08/2013

								\returns	The current layer, or an empty Layer object if at the end (see default Layer
											constructor for more information)
							*/
							std::unique_ptr<Layer> layer() const;


						private:
							// Only Layer can construct a new Iterator object
							friend class Layer;

							/**
								\brief	Private implementation data
							*/
							void* __data;
					
							/**
								\brief	Empty Iterator objects cannot be constructed
								\author	GW
								\date	08/2013
							*/
							Iterator();
						
							/**
								\brief	Creates a new iterator that starts at some target layer
								\author	GW
								\date	08/2013

								\param	layer_	Initial layer whose parent group will be iterated through; iteration starts
												at layer_
							*/
							explicit Iterator(const Layer& layer_);
					};
			
					typedef std::unique_ptr<Iterator> IteratorUP;
					typedef std::shared_ptr<Iterator> IteratorSP;
					typedef std::weak_ptr<Iterator> IteratorWP;
			
					typedef Color ColorType;
				#endif
				// HDI_CORE_AIP_MODE
			
				#if defined(HDI_CORE_PSP_MODE)
					enum ColorType
					{
						ColorTypeUnknown	= 0,
						ColorTypeNone		= 10,
						ColorTypeRed		= 20,
						ColorTypeOrange		= 30,
						ColorTypeYellow		= 40,
						ColorTypeGreen		= 50,
						ColorTypeBlue		= 60,
						ColorTypeViolet		= 70,
						ColorTypeGray		= 80
					};
				
					enum LockingState
					{
						LockingStateNone				= 0,
						LockingStateArtboardAutonest	= 1 << 0,
						LockingStateComposite			= 1 << 1,
						LockingStatePosition			= 1 << 2,
						LockingStateTransparency		= 1 << 3,
						LockingStateAll					= 0xffffffff
					};
				#endif
				// HDI_CORE_PSP_MODE

				/**
					\brief	Constructs an empty Layer object
					\author	GW
					\date	08/2013
					
					\note	To test if a Layer object is empty, call isEmpty() on it
					\note	Empty Layer objects do not relate to any layer in the current document; they are designed to
							be "receivers" of some other Layer object via the overloaded assignment operator. Empty
							Layer objects are useless until such time (though it is safe to call any of their methods).
				*/
				Layer();
			
				/**
					\brief	Constructs a Layer object from an existing Layer object (copy constructor)
					\author	GW
					\date	08/2013

					\param	layer_	Existing Layer object
				*/
				Layer(const Layer& layer_);
				
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief	Constructs a new Layer object from an AILayerHandle
						\author	GW
						\date	08/2015
						
						\param	aiLayer_	Raw Illustrator AILayerHandle
					*/
					Layer(const AILayerHandle aiLayer_);

					/**
						\brief		Inserts a new layer at a given position in the layer list in the current document
						\author		GW
						\date		08/2013

						\param		order_	Order in which to create the layer
						\param		prep_	Prepositional layer to which order_ will apply

						\warning	This method does not assume responsibility of the memory for the prep_ argument.
					*/
					Layer(const PaintOrder order_, const Layer* const prep_ = NULL);
				#elif defined(HDI_CORE_PSP_MODE)
					/**
						\brief	Constructs a new Layer object from a Photoshop layer ID
						\author	GW
						\date	12/2017
						
						\param	psLayer_	Raw Photoshop layer ID
					*/
					Layer(const int32_t psLayer_);
				#endif

				/**
					\brief	Layer destructor
					\author	GW
					\date	08/2013
				*/
				virtual ~Layer();
				
				/**
					\brief		Assigns one Layer object to another
					\author		GW
					\date		08/2013

					\param		rhs_	Existing Layer object to copy values from
					\returns	The target Layer object, but with its values updated to match that of the rhs_ argument
				*/
				virtual Layer& operator=(const Layer& rhs_);

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the layer handle around which the target object is wrapped
						\author		GW
						\date		11/2013
						
						\returns	AILayerHandle for this object

						\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
									doing. If a specific piece of functionality provided by Illustrator is not handled by
									this class (or related classes), then it should probably be added to the hdi_core library.
					*/
					virtual AILayerHandle aiLayerHandle() const;
				#elif defined(HDI_CORE_PSP_MODE)
					/**
						\brief		Gets the layer ID around which the target object is wrapped
						\author		GW
						\date		12/2017
						
						\returns	Layer ID for this object

						\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
									doing. If a specific piece of functionality provided by Photoshop is not handled by
									this class (or related classes), then it should probably be added to the hdi_core library.
					*/
					virtual int32_t psLayerID() const;
				#endif

				/**
					\brief		Gets whether the target Layer object is empty (constructed with the default ctor)
					\author		GW
					\date		08/2013
					
					\returns	true if the target Layer object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Removes the layer from the document
					\author		GW
					\date		08/2013

					\returns	true for successful disposal, false otherwise

					\note		After calling dispose(), this object is basically useless! The actual layer is gone, so
								obviously none of these methods will work after that.
				*/
				virtual bool dispose();

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets whether the target layer is editable
						\author		GW
						\date		08/2013

						\returns	true for the layer being editable (unlocked), false otherwise

						\note		This method always returns the opposite of locked() method.
					*/
					virtual bool editable() const;

					/**
						\brief	Sets whether the target layer is editable
						\author	GW
						\date	08/2013

						\param	editable_	true to allow editing, false otherwise

						\note	Always sets the layer to be locked.
					*/
					virtual void setEditable(const bool editable_);

					/**
						\brief		Gets whether the target layer is locked
						\author		GW
						\date		08/2013

						\returns	true for the layer being locked (uneditable), false otherwise

						\note		Always returns the opposite of editable() method.
					*/
					virtual bool locked() const;

					/**
						\brief	Sets whether the target layer is locked
						\author	GW
						\date	08/2013

						\param	locked_		true to lock, false otherwise

						\note	Always makes the layer uneditable.
					*/
					virtual void setLocked(const bool locked_);

					/**
						\brief		Gets whether the target layer has any art
						\author		GW
						\date		08/2013
						
						\returns	true if the layer contains art, false otherwise
					*/
					virtual bool hasArt() const;

					/**
						\brief		Gets whether any of the target layer's art is selected
						\author		GW
						\date		08/2013

						\returns	true if the layer has any art selected, false otherwise
					*/
					virtual bool hasSelectedArt() const;

					/**
						\brief	Deselects all of the target layer's art
						\author	GW
						\date	08/2013
						
						\note	This method can be called whether or not any art is actually selected.
					*/
					virtual void deselectArt() const;

					/**
						\brief		Gets the first (invisible) art group inside the layer
						\author		GW
						\date		08/2013

						\returns	The Art object representing the layer group (layers are merely art groups with a little
									"sugar" added)
					*/
					virtual std::unique_ptr<Art> group() const;

					/**
						\brief		Gets the parent layer object for the target layer
						\author		GW
						\date		08/2013
						
						\returns	The parent layer object, or NULL for none
					*/
					virtual std::unique_ptr<Layer> parent() const;

					/**
						\brief		Gets the first child layer object for the target layer
						\author		GW
						\date		08/2013
						
						\returns	A Layer object for the container's first child, or NULL if none
					*/
					virtual std::unique_ptr<Layer> firstChild() const;

					/**
						\brief		Gets the next sibling layer object for the target layer
						\author		GW
						\date		08/2013
						
						\returns	A Layer object for the target's sibling, or NULL if none
					*/
					virtual std::unique_ptr<Layer> sibling() const;

					/**
						\brief		Gets the previous sibling layer object for the target layer
						\author		GW
						\date		08/2013
						
						\returns	A Layer object for the target's previous sibling, or NULL if none
					*/
					virtual std::unique_ptr<Layer> priorSibling() const;

					/**
						\brief		Gets the next layer in preorder traversal, which returns descendents first
						\author		GW
						\date		08/2013
						
						\returns	A Layer object for the target's next preorder child or sibling (for recursive looping),
									or NULL if none
					*/
					virtual std::unique_ptr<Layer> nextPreorderLayer() const;

					/**
						\brief		Gets the target layer's preview or outline mode setting
						\author		GW
						\date		08/2013

						\returns	true for preview, false for outline
					*/
					virtual bool preview() const;

					/**
						\brief	Sets the target layer to preview or outline mode
						\author	GW
						\date	08/2013

						\param	preview_	true for preview, false for outline
					*/
					virtual void setPreview(const bool preview_);

					/**
						\brief		Gets whether the target layer is considered printable when printing the document
						\author		GW
						\date		08/2013

						\returns	true for printable, false otherwise
					*/
					virtual bool printed() const;

					/**
						\brief	Sets whether the target layer is considered printable when printing the document
						\author	GW
						\date	08/2013

						\param	printed_	true for printable, false otherwise
					*/
					virtual void setPrinted(const bool printed_);

					/**
						\brief		Gets whether images in the target layer are automatically dimmed
						\author		GW
						\date		08/2013

						\returns	true for dimming images, false otherwise
					*/
					virtual bool dimsPlacedImages() const;
					
					/**
						\brief	Sets whether images in the target layer are automatically dimmed
						\author	GW
						\date	08/2013

						\param	dim_	true to dim images, false otherwise
					*/
					virtual void setDimsPlacedImages(const bool dim_);
				
					/**
						\brief		Converts the target layer to an artwork group
						\author		GW
						\date		02/2016
						
						\returns	true if the conversion was successful, false otherwise
						
						\warning	Conversion is a destructive process. Once the target layer has been converted, the Layer
									object will become empty (i.e. its isEmpty() method will return true).
						
						\note		Top-level layers cannot be converted; only sublayers.
						\note		Sublayers that themselves contain sublayers cannot be converted.
					*/
					virtual bool convertToGroup();

					/**
						\brief		Gets an iterator for the layer in the target's parent layer, starting with the target
						\author		GW
						\date		08/2013
						
						\returns	An Iterator object, for iterating through the target's siblings
						
						\code
							// Here's an example, iterating through all top-level layers of a document:
							hdi::core::Layer layer0;
							HDI_CORE_ILLUSTRATOR->currentDocument()->firstLayer(layer0);
							for(hdi::core::Layer::Iterator iter = layer0.begin(); !iter.atEnd(); ++iter)
							{
								// Do stuff with iter.layer() ...
							}
							
							// Here's another example, reverse iterating through all top-level layers of a document:
							hdi::core::Layer layerN;
							HDI_CORE_ILLUSTRATOR->currentDocument()->lastLayer(layerN);
							for(hdi::core::Layer::Iterator iter = layerN.begin(); !iter.atEnd(); --iter)
							{
								// Do more stuff with iter.layer() ...
							}
						\endcode
					*/
					Iterator begin() const;
				#endif
				// HDI_CORE_AIP_MODE
			
				#if defined(HDI_CORE_PSP_MODE)
					/**
						\brief		Gets whether the target layer is locked
						\author		GW
						\date		12/2017

						\returns	LockingState enum values bitwise OR'd together to indicate locking state
					*/
					virtual LockingState locked() const;

					/**
						\brief	Sets whether the target layer is locked
						\author	GW
						\date	12/2017

						\param	locked_		New LockingState enum values bitwise OR'd together
					*/
					virtual void setLocked(const LockingState locked_);

					/**
						\brief		Gets whether the target layer is background
						\author		GW
						\date		12/2017

						\returns	true for the layer being background, false otherwise
					*/
					virtual bool background() const;

					/**
						\brief		Gets whether the target layer has effects visible
						\author		GW
						\date		12/2017
						
						\returns	true for effects being visible, false for hidden
					*/
					virtual bool fxVisible() const;

					/**
						\brief	Sets whether the target layer has effects visible
						\author	GW
						\date	12/2017

						\param	visible_	true to make effects visible, false to hide
					*/
					virtual void setFXVisible(const bool visible_);

					/**
						\brief		Gets the opacity level of the target layer
						\author		GW
						\date		12/2017
						
						\returns	Opacity value, in the range [0.0,1.0]
					*/
					virtual double opacity() const;

					/**
						\brief	Sets the opacity level of the target layer
						\author	GW
						\date	12/2017

						\param	o_	New opacity value, in the range [0.0,1.0]
					*/
					virtual void setOpacity(const double o_);

					/**
						\brief		Gets whether the target layer preserves transparency
						\author		GW
						\date		12/2017
						
						\returns	true for preserving transparency, false otherwise
					*/
					virtual bool preserveTransparency() const;

					/**
						\brief	Sets whether the target layer preserves transparency
						\author	GW
						\date	12/2017

						\param	pres_	true to preserve transparency, false otherwise
					*/
					virtual void setPreserveTransparency(const bool pres_);

					/**
						\brief		Gets the the blend mode for the target layer
						\author		GW
						\date		12/2017
						
						\returns	The layer blend mode
					*/
					virtual BlendMode blendMode() const;
				
					/**
						\brief	Sets the the blend mode for the target layer
						\author	GW
						\date	12/2017
						
						\param	bm_		New layer blend mode
					*/
					virtual void setBlendMode(const BlendMode bm_);
			
					/**
						\brief		Gets the total layer bounds
						\author		GW
						\date		10/2017
						
						\returns	Layer bounds rect
					*/
					virtual Rect bounds() const;
				
					/**
						\brief		Gets the layer bounds without effects
						\author		GW
						\date		10/2017
						
						\returns	Layer non-effects bounds rect
					*/
					virtual Rect boundsNoEffects() const;
					
					/**
						\brief		Gets the layer bounds without mask
						\author		GW
						\date		10/2017
						
						\returns	Layer non-mask bounds rect
					*/
					virtual Rect boundsNoMask() const;
				#endif
				// HDI_CORE_PSP_MODE

				/**
					\brief		Gets the name for the target layer
					\author		GW
					\date		08/2013

					\returns	The name of the layer, as UTF-8
				*/
				virtual std::string name() const;

				/**
					\brief	Sets the name for the target layer
					\author	GW
					\date	08/2013

					\param	name_	New layer name, as UTF-8
				*/
				virtual void setName(const std::string& name_);

				/**
					\brief	Shows the layer
					\author	GW
					\date	08/2013
				*/
				virtual void show();

				/**
					\brief	Hides the layer
					\author	GW
					\date	08/2013
				*/
				virtual void hide();

				/**
					\brief		Gets whether the target layer is visible
					\author		GW
					\date		08/2013
					
					\returns	true for visible, false for hidden
				*/
				virtual bool visible() const;

				/**
					\brief	Sets whether the target layer is visible
					\author	GW
					\date	08/2013

					\param	visible_	true to make visible, false to hide

					\note	This method simply calls show() or hide() as appropriate
				*/
				virtual void setVisible(const bool visible_);

				/**
					\brief		Gets whether the target Layer object is the active (current) layer
					\author		GW
					\date		08/2013
					
					\returns	true if the layer is active, false otherwise
				*/
				virtual bool active() const;

				/**
					\brief		Gets the color of the layer in the layer list
					\author		GW
					\date		08/2013

					\returns	A Color object representing the RGB color of the layer in the app Layers panel
				*/
				virtual ColorType color() const;

				/**
					\brief	Sets the color of the layer in the layer list
					\author	GW
					\date	08/2013

					\param	color_	New layer color
				*/
				virtual void setColor(const ColorType color_);
				
				/**
					\brief		Tests whether a given Layer object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Layer to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same layer, false otherwise
				*/
				virtual bool operator==(const Layer& rhs_) const;
				
				/**
					\brief		Tests whether a given Layer object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Layer to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different layers, false otherwise
				*/
				virtual bool operator!=(const Layer& rhs_) const;

				/**
					\brief		Tests the validity of the target layer by checking its topmost "invisible" group
					\author		GW
					\date		08/2013
					
					\returns	true if the layer's group is valid, false otherwise
				*/
				virtual bool valid() const;


			private:
				#if defined(HDI_CORE_AIP_MODE)
					friend class CurrentDocument;
					
					friend aip::Layer* __accessImpl(const Layer&);
					friend Layer __accessCtor(aip::Layer*&);

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					Layer(aip::Layer*&);
				#elif defined(HDI_CORE_PSP_MODE)
					friend class Document;
					
					friend psp::Layer* __accessImpl(const Layer&);
					friend Layer __accessCtor(psp::Layer*&);

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					Layer(psp::Layer*&);
				#endif

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				void* __impl() const;
		};
		
		typedef std::unique_ptr<Layer> LayerUP;
		typedef std::shared_ptr<Layer> LayerSP;
		typedef std::weak_ptr<Layer> LayerWP;

		#if defined(HDI_CORE_AIP_MODE)
			extern aip::Layer* __accessImpl(const Layer&);
			extern Layer __accessCtor(aip::Layer*&);
		#elif defined(HDI_CORE_PSP_MODE)
			extern psp::Layer* __accessImpl(const Layer&);
			extern Layer __accessCtor(psp::Layer*&);
		#endif
	}
}

#if defined(HDI_CORE_PSP_MODE)
	hdi::core::Layer::LockingState operator|(
		const hdi::core::Layer::LockingState lhs_,
		const hdi::core::Layer::LockingState rhs_
	);

	hdi::core::Layer::LockingState& operator|=(
		hdi::core::Layer::LockingState& lhs__,
		const hdi::core::Layer::LockingState rhs_
	);
#endif
// HDI_CORE_PSP_MODE

#endif
// __HDI_CORE_LAYER__
