/**
	\file
	\brief		Header file for general Illustrator mask art manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_MASK_ART__
#define __HDI_CORE_MASK_ART__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		class Art;
	
		/**
			\brief	Allows for mask manipulation, metadata access, etc.
		*/
		class MaskArt
		{
			public:
				/**
					\brief	Destructs a MaskArt object
					\author	GW
					\date	08/2014
				*/
				~MaskArt();
				
				/**
					\brief		Gets whether a mask for the related art already exists
					\author		GW
					\date		08/2014
					
					\returns	true if a mask for the art exists, false otherwise
					
					\note		This method is useful in combination with create() and dispose()
				*/
				bool exists() const;
				
				/**
					\brief		Creates a mask for the related art object if one does not already exists
					\author		GW
					\date		08/2014
					
					\returns	true if the mask was created successfully (or already exists)
				*/
				bool create();
				
				/**
					\brief		Disposes the mask for the related art object, if one exists
					\author		GW
					\date		08/2014
					
					\returns	true if the mask was disposed successfully (or never existed)
				*/
				bool dispose();
				
				/**
					\brief		Gets the link state of the target mask
					\author		GW
					\date		08/2014
					
					\returns	true if the mask is linked, false otherwise
					
					\note		When a mask is linked, certain actions on the associated art object, such as rotation,
								are automatically applied to the mask as well.
				*/
				bool linked() const;
				
				/**
					\brief	Sets the link state of the target mask
					\author	GW
					\date	08/2014
					
					\param	link_	true to turn linking on, false to turn it off
					
					\note	When a mask is linked, certain actions on the associated art object, such as rotation, are
							automatically applied to the mask as well.
				*/
				void setLinked(const bool link_);
				
				/**
					\brief		Gets the enabled state of the target mask
					\author		GW
					\date		08/2014
					
					\returns	true if the mask is enabled, false otherwise
					
					\note		When a mask is disabled, it is ignored when rendering the associated art object.
				*/
				bool enabled() const;
				
				/**
					\brief	Sets the enabled state of the target mask
					\author	GW
					\date	08/2014
					
					\param	enable_	true to enable the mask, false to disable
					
					\note	When a mask is disabled, it is ignored when rendering the associated art object.
				*/
				void setEnabled(const bool enable_);
				
				/**
					\brief		Gets the inversion state of the target mask
					\author		GW
					\date		08/2014
					
					\returns	true if the mask is inverted, false otherwise
					
					\note		When a mask is inverted, the opacity is inverted before being applied to the associated
								art.
				*/
				bool inverted() const;
				
				/**
					\brief	Sets the inversion state of the target mask
					\author	GW
					\date	08/2014
					
					\param	invert_	true to invert the mask, false to disable
					
					\note	When a mask is inverted, the opacity is inverted before being applied to the associated art.
				*/
				void setInverted(const bool invert_);
				
				/**
					\brief		Gets the clipping state of the target mask
					\author		GW
					\date		08/2014
					
					\returns	true if the mask is clipped, false otherwise
					
					\note		When a mask is clipped, the opacity value outside the mask art is 0%, thus clipping out
								everything outside the mask art. When clipping is off, the value outside is 100%, thus
								revealing everything outside the boundary of the mask art.
				*/
				bool clipped() const;
				
				/**
					\brief	Sets the clipping state of the target mask
					\author	GW
					\date	08/2014
					
					\param	clip_	true to enable clipping the mask, false to disable
					
					\note	When a mask is clipped, the opacity value outside the mask art is 0%, thus clipping out
							everything outside the mask art. When clipping is off, the value outside is 100%, thus
							revealing everything outside the boundary of the mask art.
				*/
				void setClipped(const bool clip_);
				
				/**
					\brief		Copies the target mask to some destination art
					\author		GW
					\date		08/2014
					
					\param		dest_	Destination art object
					\returns	true if the copy was performed successfully, false otherwise
				*/
				bool copy(const Art& dest_) const;

				/**
					\brief		Gets the art object that comprises the opacity mask
					\author		GW
					\date		08/2014
					
					\returns	The art object, or NULL if there is no mask
				*/
				std::unique_ptr<Art> maskArt() const;


			private:
				// Only Art can construct a new MaskArt object
				friend class Art;

				/**
					\brief	Art to which this mask belongs
				*/
				void* __data;
				
				/**
					\brief	Empty MaskArt objects cannot be constructed
					\author	GW
					\date	08/2014
				*/
				MaskArt();

				/**
					\brief	Constructs a new MaskArt object from a given Art object
					\author	GW
					\date	08/2014

					\param	art_	Pointer for the existing art we're wrapping around
				*/
				MaskArt(void* const art_);
			
				/**
					\brief	MaskArt objects cannot be copied
					\author	GW
					\date	08/2014
				*/
				MaskArt(const MaskArt&);
			
				/**
					\brief	MaskArt objects cannot be copied
					\author	GW
					\date	08/2013
				*/
				MaskArt& operator=(const MaskArt&);
		};
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_MASK_ART__
