/**
	\file
	\brief		Header file for manipulating Illustrator menubar menu groups
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_MENU_GROUP__
#define __HDI_CORE_MENU_GROUP__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		class MenuItem;
	
		/**
			\brief	Wraps around an Illustrator menu group
		*/
		class MenuGroup
		{
			public:
				struct Child;
			
				typedef std::vector< std::shared_ptr<Child> > ChildVector;
			
				/**
					\brief	Menu groups can have both groups and items as children, so this acts as a simple wrapper to
							store one or the other
				*/
				struct Child
				{
					/**
						\brief	Stores the child menu group
					*/
					std::unique_ptr<MenuGroup> group;
					
					/**
						\brief	Stores the child menu item
					*/
					std::unique_ptr<MenuItem> item;
					
					/**
						\brief	Constructs a new Child object with neither a group nor item
						\author	GW
						\date	09/2014
					*/
					Child();
					
					/**
						\brief	Copy constructor for a Child object
						\author	GW
						\date	09/2014
						
						\param	kid_	Existing Child object to copy a group or item from
					*/
					Child(const Child& kid_);
					
					/**
						\brief	Child object destructor
						\author	GW
						\date	09/2014
					*/
					~Child();
					
					/**
						\brief		Overloaded assignment operator for Child objects
						\author		GW
						\date		09/2014
						
						\param		rhs_	Righthand side of the assignment operator; existing Child object to copy
											values from
						\returns	The target Child object, but with its members updated to match that of rhs_
					*/
					Child& operator=(const Child& rhs_);
				};
		
				typedef std::unique_ptr<Child> ChildUP;
				typedef std::shared_ptr<Child> ChildSP;
				typedef std::weak_ptr<Child> ChildWP;
			
				/**
					\brief	Constructs an empty MenuGroup object
					\author	GW
					\date	11/2013
					
					\note	To test if a MenuGroup object is empty, call isEmpty() on it
					\note	Empty MenuGroup objects do not relate to any actual Illustrator menu group; they are
							designed to be "receivers" of some other MenuGroup object via the overloaded assignment
							operator. Empty MenuGroup objects are useless until such time (though it is safe to call any
							of their methods).
				*/
				MenuGroup();
			
				/**
					\brief	Constructs a new MenuGroup object from an existing MenuGroup object (copy constructor)
					\author	GW
					\date	11/2013

					\param	m_	Existing MenuGroup object
				*/
				MenuGroup(const MenuGroup& m_);
					
				/**
					\brief	Constructs a new MenuGroup object
					\author	GW
					\date	11/2013

					\param	group_		Built-in Illustrator menu group to contain the new MenuGroup
					\param	text_		String that will show as the menu group title, as UTF-8
					\param	sort_		Whether to sort the children of the new group alphabetically, or in order of
										creation
					\param	mergeTop_	Whether to merge the new group with group_ and force any children of the new
										group to be at the top of the list
					\param	divBelow_	Whether to automatically place a separator item below the new group
					\param	divAbove_	Whether to automatically place a separator item above the new group
					
					\note	divBelow_ and divAbove_ are only applicable if mergeTop_ is true.
				*/
				MenuGroup(
					const BuiltinMenuGroup group_,
					const std::string& text_,
					const bool sort_ = false,
					const bool mergeTop_ = false,
					const bool divBelow_ = false,
					const bool divAbove_ = false
				);

				/**
					\brief	Constructs a new MenuGroup object
					\author	GW
					\date	11/2013

					\param	group_		User-created menu group to contain the new MenuGroup
					\param	text_		String that will show as the menu group title, as UTF-8
					\param	sort_		Whether to sort the children of the new group alphabetically, or in order of
										creation
					\param	mergeTop_	Whether to merge the new group with group_ and force any children of the new
										group to be at the top of the list
					\param	divBelow_	Whether to automatically place a separator item below the new group
					\param	divAbove_	Whether to automatically place a separator item above the new group
					
					\note	divBelow_ and divAbove_ are only applicable if mergeTop_ is true.
				*/
				MenuGroup(
					const MenuGroup& group_,
					const std::string& text_,
					const bool sort_ = false,
					const bool mergeTop_ = false,
					const bool divBelow_ = false,
					const bool divAbove_ = false
				);

				/**
					\brief	Destructs a MenuGroup object
					\author	GW
					\date	11/2013
				*/
				virtual ~MenuGroup();

				/**
					\brief	Destroys the UI menu group, converting the target object to an empty MenuGroup object (see
							the default constructor for more info on empty MenuGroup objects)
					\author	GW
					\date	11/2013

					\note	This method must be called for any existing menu groups by the time the plugin is shutting
							down. If you no longer retain any of your own references to a menu group then the lib will
							call this method for you as it is shutting down.
				*/
				virtual void destroy();
			
				/**
					\brief		Assigns one MenuGroup object to another
					\author		GW
					\date		11/2013

					\param		rhs_	Existing MenuGroup object to copy values from
					\returns	The target MenuGroup object, but with its values updated to match that of the rhs_
								argument
				*/
				virtual MenuGroup& operator=(const MenuGroup& rhs_);
				
				/**
					\brief		Tests whether a given MenuGroup object is the same as another
					\author		GW
					\date		11/2013

					\param		rhs_	MenuGroup to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same menu group, false otherwise
				*/
				virtual bool operator==(const MenuGroup& rhs_) const;
				
				/**
					\brief		Tests whether a given MenuGroup object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	MenuGroup to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different menu groups, false otherwise
				*/
				virtual bool operator!=(const MenuGroup& rhs_) const;

				/**
					\brief		Gets the menu group around which the target object is wrapped
					\author		GW
					\date		09/2014
					
					\returns	AIMenuGroup for this object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual AIMenuGroup aiMenuGroup() const;

				/**
					\brief		Gets the menu item handle around which the target object is wrapped
					\author		GW
					\date		11/2013
					
					\returns	AIMenuItemHandle for this object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual AIMenuItemHandle aiMenuItemHandle() const;

				/**
					\brief		Gets whether the target MenuGroup object is empty (constructed with the default ctor)
					\author		GW
					\date		11/2013
					
					\returns	true if the target MenuGroup object is empty, false otherwise
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Gets whether the menu group is a built-in Illustrator group
					\author		GW
					\date		09/2014
					
					\returns	true if the target represents a built-in Illustrator menu group, false otherwise
					
					\note		Most of Illustrator's menu groups are not actually groups at all, but rather a specific
								region of some top-level menu or other subgroup. This can be a bit confusing, as some
								"groups" do not actually behave like proper groups (e.g. their parent cannot be acquired,
								text value cannot be acquired, etc). If a specific method of this class is not behaving
								the way you expect, it would be wise to check this method as a hint. Methods that are
								affected by the return value of isPartial() are denoted in their documentation.
				*/
				virtual bool isPartial() const;
				
				/**
					\brief		Gets the parent of the target menu group
					\author		GW
					\date		09/2014
					
					\returns	The parent menu group, or NULL for none/error
					
					\note		Cannot be acquired if isPartial() returns true.
				*/
				virtual std::unique_ptr<MenuGroup> parent() const;
				
				/**
					\brief		Gets the children of the target menu group
					\author		GW
					\date		09/2014
					
					\returns	A vector of Child objects, with either the group or item members filled in each (never
								both)

					\note		Illustrator offers no direct calls to obtain menu group children, so this method must
								iterate over all menu items in order to locate the children of the target group. As such,
								any e.g. recursive operations on a menu item hierarchy is inherently O(n^2).
				*/
				virtual ChildVector children() const;

				/**
					\brief		Gets whether the menu group is enabled
					\author		GW
					\date		11/2013

					\returns	true if the group is enabled, false otherwise
					
					\note		Ignore this method if isPartial() returns true.
				*/
				virtual bool enabled() const;

				/**
					\brief	Sets the menu group as enabled or disabled
					\author	GW
					\date	11/2013

					\param	enabled_	true to enable the group, false to disable
					
					\note	Cannot be set if isPartial() returns true.
				*/
				virtual void setEnabled(const bool enabled_);

				/**
					\brief		Gets the text of the menu group itself
					\author		GW
					\date		11/2013

					\returns	Current text value of the menu group, as UTF-8
					
					\note		Ignore this method if isPartial() returns true.
				*/
				virtual std::string text() const;

				/**
					\brief	Sets the text of the menu group itself
					\author	GW
					\date	11/2013

					\param	text_	New text value for the menu group, as UTF-8

					\note	Cannot be set if isPartial() returns true.
				*/
				virtual void setText(const std::string& text_);

				/**
					\brief		Gets whether the children of the target menu group are sorted
					\author		GW
					\date		09/2014
					
					\returns	true if sorted alphabetically, false otherwise
				*/
				virtual bool sorted() const;
				
				/**
					\brief	Sets whether the children of the target menu group art sorted alphabetically or in order of
							creation
					\author	GW
					\date	09/2014
					
					\param	sort_	true to sort alphabetically, false otherwise
				*/
				virtual void setSorted(const bool sort_);


			private:
				friend MenuGroup __accessCtor(AIMenuGroup, AIMenuItemHandle);

				/**
					\brief	Private implementation data
				*/
				void* __data;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				MenuGroup(AIMenuGroup, AIMenuItemHandle);
		};
		
		typedef std::unique_ptr<MenuGroup> MenuGroupUP;
		typedef std::shared_ptr<MenuGroup> MenuGroupSP;
		typedef std::weak_ptr<MenuGroup> MenuGroupWP;
		
		extern MenuGroup __accessCtor(AIMenuGroup, AIMenuItemHandle);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_MENU_GROUP__
