/**
	\file
	\brief		Header file for Illustrator notification data wrapper classes
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_NOTIFICATION__
#define __HDI_CORE_NOTIFICATION__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		class Message;
		class Symbol;
	
		/**
			\brief	Base class for all *Notification subclasses
		*/
		class Notification
		{
			public:
				/**
					\brief	Destructs a Notification object
					\author	GW
					\date	07/2017
				*/
				virtual ~Notification();
			
				/**
					\brief		Gets the type of notification data the target object represents
					\author		GW
					\date		07/2017

					\returns	Notifier type (useful for determining how to cast a Notification object to its subclass)
				*/
				virtual NotifierType type() const;
			
			
			protected:
				/**
					\brief	Stores private implementation data
				*/
				void* _data;

				/**
					\brief	Internal use only
					\author	GW
					\date	07/2017
				*/
				Notification();
				
				/**
					\brief	Unused
					\author	GW
					\date	07/2017
				*/
				Notification(const Notification&);
				
				/**
					\brief		Unused
					\author		GW
					\date		07/2017

					\returns	The target Notification object, but with its values updated to match that of the param
				*/
				Notification& operator=(const Notification&);
		};
		
		/**
			\brief	Received for ArtObjectsChangedNotifierType
		*/
		class ArtObjectsChangedNotification : public Notification
		{
			public:
				typedef std::vector< std::string > UUIDVector;
			
				/**
					\brief	Destructs a ArtObjectsChangedNotification object
					\author	GW
					\date	07/2017
				*/
				virtual ~ArtObjectsChangedNotification();
			
				/**
					\brief		Gets a vector containing the UUID of each added piece of art
					\author		GW
					\date		07/2017
					
					\returns	A vector of each UUID that was added
				*/
				const UUIDVector* const addedArt() const;
			
				/**
					\brief		Gets a vector containing the UUID of each removed piece of art
					\author		GW
					\date		07/2017
					
					\returns	A vector of each UUID that was removed
				*/
				const UUIDVector* const removedArt() const;
			
				/**
					\brief		Gets a vector containing the UUID of each changed piece of art
					\author		GW
					\date		07/2017
					
					\returns	A vector of each UUID that was changed
				*/
				const UUIDVector* const changedArt() const;


			private:
				// Only NotifyMessage can construct an ArtObjectsChangedNotification object
				friend class NotifyMessage;

				/**
					\brief	Internal use only
					\author	GW
					\date	07/2017
				*/
				ArtObjectsChangedNotification(const NotifierType, void*);
				
				/**
					\brief	Unused
					\author	GW
					\date	07/2017
				*/
				ArtObjectsChangedNotification(const ArtObjectsChangedNotification&);
				
				/**
					\brief	Unused
					\author	GW
					\date	07/2017
				*/
				ArtObjectsChangedNotification& operator=(const ArtObjectsChangedNotification&);
		};

		/**
			\brief	Received for ArtSymbolSetDetailedChangeNotifierType
		*/
		class ArtSymbolSetChangedNotification : public Notification
		{
			public:
				typedef std::vector< std::shared_ptr<Symbol> > SymbolVector;
			
				/**
					\brief	Destructs a ArtSymbolSetChangedNotification object
					\author	GW
					\date	07/2017
				*/
				virtual ~ArtSymbolSetChangedNotification();
			
				/**
					\brief		Gets a vector containing each added symbol
					\author		GW
					\date		07/2017
					
					\returns	A vector of each symbol that was added
				*/
				SymbolVector addedSymbols() const;
			
				/**
					\brief		Gets a vector containing each removed symbol
					\author		GW
					\date		07/2017
					
					\returns	A vector of each symbol that was removed
				*/
				SymbolVector removedSymbols() const;
			
				/**
					\brief		Gets a vector containing each changed symbol
					\author		GW
					\date		07/2017
					
					\returns	A vector of each symbol that was changed
				*/
				SymbolVector changedSymbols() const;
			
				/**
					\brief		Gets a vector containing each renamed symbol
					\author		GW
					\date		07/2017
					
					\returns	A vector of each symbol that was renamed
				*/
				SymbolVector renamedSymbols() const;


			private:
				// Only NotifyMessage can construct an ArtSymbolSetChangedNotification object
				friend class NotifyMessage;

				/**
					\brief	Internal use only
					\author	GW
					\date	07/2017
				*/
				ArtSymbolSetChangedNotification(const NotifierType, void*);
				
				/**
					\brief	Unused
					\author	GW
					\date	07/2017
				*/
				ArtSymbolSetChangedNotification(const ArtSymbolSetChangedNotification&);
				
				/**
					\brief	Unused
					\author	GW
					\date	07/2017
				*/
				ArtSymbolSetChangedNotification& operator=(const ArtSymbolSetChangedNotification&);
		};
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_NOTIFICATION__
