/**
	\file
	\brief		Header file for Illustrator notifier manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_NOTIFIER__
#define __HDI_CORE_NOTIFIER__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		
		/**
			\brief	Acts as a receiver for specific notifications, passing them along to caller-supplied callbacks
		*/
		class Notifier
		{
			public:
				/**
					\brief	Constructs an empty Notifier object
					\author	GW
					\date	08/2013
					
					\note	To test if a Notifier object is empty, call isEmpty() on it
					\note	Empty Notifier objects do not relate to any Illustrator notifier; they are designed to be
							"receivers" of some other Notifier object via the overloaded assignment operator. Empty
							Notifier objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				Notifier();
			
				/**
					\brief	Constructs a new Notifier object from an existing Notifier object (copy constructor)
					\author	GW
					\date	08/2013

					\param	n_	Existing Notifier object
				*/
				Notifier(const Notifier& n_);

				/**
					\brief	Constructs a Notifier object to fire a responder (target and action) when Illustrator sends
							a notification
					\author	GW
					\date	08/2013

					\param	type_		Type of notifier
					\param	responder_	Target and action to fire
				*/
				Notifier(const NotifierType type_, const Callback& responder_);

				/**
					\brief	Destructs a Notifier object, deactivating it first
					\author	GW
					\date	08/2013
				*/
				virtual ~Notifier();
			
				/**
					\brief		Assigns one Notifier object to another
					\author		GW
					\date		08/2013

					\param		rhs_	Existing Notifier object to copy values from
					\returns	The target Notifier object, but with its values updated to match that of the rhs_
								argument
				*/
				virtual Notifier& operator=(const Notifier& rhs_);
				
				/**
					\brief		Tests whether a given Notifier object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Notifier to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same notifier, false otherwise
				*/
				virtual bool operator==(const Notifier& rhs_) const;
				
				/**
					\brief		Tests whether a given Notifier object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Notifier to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different notifiers, false otherwise
				*/
				virtual bool operator!=(const Notifier& rhs_) const;

				/**
					\brief		Gets whether the target Notifier object is empty (constructed with the default ctor)
					\author		GW
					\date		08/2013
					
					\returns	true if the target Notifier object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Gets the type of the notifier
					\author		GW
					\date		08/2013

					\returns	Notifier type
				*/
				virtual NotifierType type() const;

				/**
					\brief		Gets whether the notifier is active
					\author		GW
					\date		08/2013

					\returns	true if the notifier is enabled, false otherwise
				*/
				virtual bool active() const;

				/**
					\brief	Sets whether the notifier is active
					\author	GW
					\date	08/2013

					\param	active_		true to enable the notifier, false to disable

					\note	This method simply calls activate() or deactivate() as appropriate
				*/
				virtual void setActive(const bool active_);

				/**
					\brief	Activates (enables) the notifier
					\author	GW
					\date	08/2013
				*/
				virtual void activate() const;

				/**
					\brief	Deactivates (disables) the notifier
					\author	GW
					\date	08/2013
				*/
				virtual void deactivate() const;

				/**
					\brief		Gets the registered callback for the notifier
					\author		GW
					\date		08/2013

					\returns	Currently registered notifier callback
				*/
				virtual Callback* const callback() const;
				
				/**
					\brief	Sets the callback for the notifier
					\author	GW
					\date	08/2013

					\param	cb_		New callback to be performed when the notifier fires
				*/
				virtual void setCallback(const Callback& cb_);


			private:
				/**
					\brief	Private implementation data
				*/
				void* __data;
		};
		
		typedef std::unique_ptr<Notifier> NotifierUP;
		typedef std::shared_ptr<Notifier> NotifierSP;
		typedef std::weak_ptr<Notifier> NotifierWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_NOTIFIER__
