/**
	\file
	\brief		Header file for general Illustrator path art manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_PATH_ART__
#define __HDI_CORE_PATH_ART__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreArtboardPoint.h"
#include "hdicorePathStyle.h"

namespace hdi
{
	namespace core
	{
		class Angle;
		class Art;
		class Bezier;
		class HitData;
		class ArtboardSegPoint;
	
		/**
			\brief	Allows for path manipulation, metadata access, trimming, extending, etc.
		*/
		class PathArt
		{
			public:
				typedef std::vector<ArtboardSegPoint> ArtboardSegPointVector;
				typedef std::vector< std::shared_ptr<HitData> > HitDataVector;
			
				/**
					\brief	Maximum number of segments allowed in a path
				*/
				static const int16_t maxSegments = 32000;

				/**
					\brief	Describes which direction the (closed) art is wound
				*/
				enum WindingOrderType
				{
					WindingUnknown		= 0,
					Clockwise			= 1,	// determined by positive area
					CounterClockwise	= 2		// determined by negative area
				};

				/**
					\brief	Describes how an individual segment point is selected, if at all

					\note	There are a limited number of ways that a segment point can be selected. Either the point is
							not selected at all; its anchor, in, and out controls are selected; its in and out controls
							are selected; its in control is selected; or its out control is selected. Any other
							combination is not possible.
				*/
				enum SegPointSelectionState
				{
					SegPointNotSelected		= 0,
					SegPointAnchorSelected	= 1 << 0,
					SegPointInSelected		= 1 << 1,
					SegPointOutSelected		= 1 << 2,
					SegPointAllSelected		= (SegPointAnchorSelected | SegPointInSelected | SegPointOutSelected)
				};

				/**
					\brief	Describes which direction a trim should take place (to the greater, lesser, or both sides of
							a given reference point)
				*/
				enum TrimDirectionType
				{
					TrimGreaterDirectionOnly	= 1 << 0,
					TrimLesserDirectionOnly		= 1 << 1,
					TrimBothDirections			= (TrimGreaterDirectionOnly | TrimLesserDirectionOnly)
				};
				
				/**
					\brief	Describes various errors that can occur when trimming a path
				*/
				enum TrimError
				{
					NoTrimError							= 0,
					HitDataArtMustMatchTrimPathArtError	= 10,
					TrimPathIsClosedError				= 20,
					TrimPathNeedsAtLeast2PointsError	= 30,
					InvalidSourcePathForTrimming		= 40
				};
				
				/**
					\brief	Describes various errors that can occur when trimming a path
				*/
				enum ExtendError
				{
					NoExtendError							= 0,
					HitDataArtMustMatchExtendPathArtError	= 110,
					NoExtendIntersectionsFoundError			= 120,
					ExtendPathEndSegmentsOnlyError			= 130,
					ExtendPathIsClosedError					= 140,
					ExtendPathNeedsAtLeast2PointsError		= 150,
					InvalidSourcePathForExtending			= 160
				};

				/**
					\brief	Destructs a PathArt object
					\author	GW
					\date	08/2013
				*/
				~PathArt();
				
				/**
					\brief		Gets whether the path is a guide
					\author		GW
					\date		08/2013
					
					\returns	true if the path is a guide path, false otherwise
				*/
				bool isGuide() const;

				/**
					\brief		Gets whether the target path is closed
					\author		GW
					\date		08/2013
					
					\returns	true if the path is closed, false if open
				*/
				bool closed() const;

				/**
					\brief	Sets whether the target path is closed
					\author	GW
					\date	08/2013

					\param	closed_		true to close the path, false to open it
				*/
				void setClosed(const bool closed_);

				/**
					\brief		Gets the target path's style
					\author		GW
					\date		08/2013
					
					\returns	An PathStyle object describing the path's styling info
				*/
				PathStyle style() const;

				/**
					\brief	Sets the target path's style
					\author	GW
					\date	08/2013

					\param	style_	New path styling info
				*/
				void setStyle(const PathStyle& style_);

				/**
					\brief		Gets the count of segments in the path, taking into account phantom segments of closed
								paths
					\author		GW
					\date		08/2013
					
					\returns	The count of segments in the path
				*/
				int16_t segCount() const;

				/**
					\brief		Gets the count of individual segment points in the path
					\author		GW
					\date		08/2013
					
					\returns	The count of segment points in the path
				*/
				int16_t segPointCount() const;

				/**
					\brief	Sets the count of segments in the path
					\author	GW
					\date	08/2013

					\param	count_	New segment count

					\pre	Path closedness/openness must already be set to the desired value, otherwise unexpected
							results may occur. If the closed/open state of a (new) path is not yet determinable, use the
							setSegPointCount() method instead.

					\note	For new path art, this method should be called before any segment point manipulation method,
							e.g. setSegs() or setSegPoints(), as it "primes" the path to have the requisite number of
							segments available for said manipulation.
					\note	For new path art, if this method is called and then e.g. setSegs() is never called, then all
							the segment points will simply lie at (0,0).
				*/
				void setSegCount(const int16_t count_);

				/**
					\brief	Sets the count of individual segment points in the path
					\author	GW
					\date	08/2013

					\param	count_	New segment point count
					
					\note	For new path art, this method should be called before any segment point manipulation method,
							e.g. setSegs() or setSegPoints(), as it "primes" the path to have the requisite number of
							segment points available for said manipulation.
					\note	For new path art, if this method is called and then e.g. setSegPoints() is never called,
							then all the segment points will simply lie at (0,0).
				*/
				void setSegPointCount(const int16_t count_);

				/**
					\brief		Gets a Bezier object for a given segment index
					\author		GW
					\date		08/2013

					\param		segIndex_	Segment index number in question
					\returns	The Bezier object that describes the path segment (if the seg index is valid), NULL
								otherwise
				*/
				std::unique_ptr<hdi::core::Bezier> bezier(const int16_t segIndex_) const;

				/**
					\brief		Gets the midpoint along a path
					\author		GW
					\date		08/2013
					
					\returns	The midpoint of the path (a point equidistant from either end point of the path)
				*/
				ArtboardPoint midPoint() const;

				/**
					\brief	Gets the end points of a path
					\author	GW
					\date	08/2013

					\param	start__		Return-by-reference for the start point (i.e. segment index 0, t-value 0.0)
					\param	end__		Return-by-reference for the end point (i.e. segment index n-1, t-value 1.0)
					
					\note	This method will also work for closed paths, giving the start and end points around the
							phantom segment.
				*/
				void endPoints(ArtboardPoint& start__, ArtboardPoint& end__) const;

				/**
					\brief		Gets whether the path has any length
					\author		GW
					\date		08/2013
					
					\returns	true if the path has length, false otherwise
					
					\note		A path might have zero length if it consists of a single point or all the points lie
								atop one another.
				*/
				bool hasLength() const;

				/**
					\brief		Gets the total length of the path
					\author		GW
					\date		08/2013
					
					\returns	The length of the path in points
				*/
				double length() const;

				/**
					\brief		Gets the length of a subpath from one segment point to another
					\author		GW
					\date		08/2013

					\param		startSegPt_		Starting segment point number for calculating path length
					\param		endSegPt_		Ending segment point number for calculating path length
					\returns	The length in points of the subpath from startSegPt_ to endSegPt_
				*/
				double sublength(const int16_t startSegPt_, const int16_t endSegPt_) const;

				/**
					\brief		Gets the length of a subpath, from segment point at a t-value to another
					\author		GW
					\date		08/2013

					\param		startSeg_	Starting segment point number for calculating path length
					\param		startT_		T-value along startSeg_ seg to begin the calculation
					\param		endSeg_		Ending segment point number for calculating path length
					\param		endT_		T-value along endSeg_ seg to end the calculation
					\returns	The length in points of the subpath from startSeg_ at startT_ to endSeg_ at endT_
				*/
				double sublength(
					const int16_t startSeg_,
					const double startT_,
					const int16_t endSeg_,
					const double endT_
				) const;

				/**
					\brief		Gets the t-value at a given length of a given path segment
					\author		GW
					\date		08/2013
					
					\param		segNum_		Segment number in which the given sublength resides
					\param		length_		Length within the given segment number for which a t-value is desired
					\returns	The bezier t-value in the given segment at which the given length is reached
				*/
				double tAtLength(const int16_t segNum_, const double length_) const;

                /**
                    \brief		Gets the area of the path
                    \author		GW
					\date		08/2013

					\returns	The area formed by the path, in square points
                 */
                double area() const;
                
				/**
					\brief		Gets whether each individual path segment is straight
					\author		GW
					\date		08/2013
					
					\returns	true if the entire path is composed of straight segments, false otherwise
				*/
				bool hasStraightSegs() const;

				/**
					\brief		Gets whether the entire path forms a perfectly straight line
					\author		GW
					\date		08/2013
					
					\returns	true if the entire path is straight and composes the same angle relative to 0.0°
				*/
				bool isStraight() const;

				/**
					\brief		Gets whether the entire path forms a rectangular shape
					\author		GW
					\date		08/2013
					
					\returns	true if the path forms an open or closed rectangle (with straight segments)
				*/
				bool isRectangular() const;

				/**
					\brief		Gets whether the entire path forms a circle
					\author		GW
					\date		08/2013
				 
					\returns	true if the path forms a closed circle, false otherwise
				*/
				bool isCircular() const;

				/**
					\brief		Gets whether the entire path forms an ellipse
					\author		GW
					\date		03/2021
				 
					\returns	true if the path forms a closed ellipse, false otherwise
				*/
				bool isEllipse() const;

				/**
					\brief		Calculates the centroid of the path
					\author		GW
					\date		04/2023
					
					\returns	The centroid of the path, via approximating its various edges as triangles and computing
								the sum of each of their centroids
					
					\note		If the path has a single point, then that point will be returned. If the path has two
								points, then the midpoint will be returned.
				*/
				ArtboardPoint centroid() const;

				/**
					\brief		Gets the angle of the path relative to the x-axis, if the path is a perfectly straight line
					\author		GW
					\date		08/2013
					
					\param		angle__		Return-by-reference for the angle of the path
					\returns	true if the path is straight and, therefore, the value of angle__ is valid
					
					\note		See isStraight() method.
				*/
				bool angle(Angle& angle__) const;

				/**
					\brief		Gets a series of points describing the requested path segments
					\author		GW
					\date		08/2013

					\param		segIndex_	Index number of the first segment in question
					\param		count_		Number of segments requested from segIndex_ forward (inclusive)
					\param		wrap_		Whether to wrap around the path, if count_ - segIndex_ goes beyond the total
											number of segments in the path (useful for closed paths, regarding the
											phantom)
					\returns	A vector of segment points describing the segments in question

					\note		The count of the resultant points in the vector will typically be equal to count_ + 1
								(since two points are needed to describe a single segment). This will be overridden in
								the case of a requested count that is larger than the path's segment count, or the path
								being closed (the phantom segment is at the "end" and wraps around).
					\note		You can acquire just the phantom segment of a path (closed or open) by passing in -1 (or
								segPtCount() - 1) for segIndex_ and 1 for count_
					\note		The maximum value for count_ is equal to maxSegments
				*/
				ArtboardSegPointVector segs(
					const int16_t segIndex_,
					const int16_t count_,
					const bool wrap_ = false
				) const;

				/**
					\brief		Gets a series of path segment points
					\author		GW
					\date		08/2013

					\param		segPtIndex_		Index number of the first segment point in question
					\param		count_			Number of segment points requested from segIndex_ forward (inclusive)
					\param		wrap_			Whether to wrap around the path, if count_ - segPtIndex_ goes beyond the
												total number of segment points in the path (useful for closed paths,
												regarding the phantom)
					\returns	A vector of segment points describing the segments in question
					
					\note		The maximum value for count_ is equal to maxSegments
				*/
				ArtboardSegPointVector segPoints(
					const int16_t segPtIndex_,
					const int16_t count_,
					const bool wrap_ = false
				) const;

				/**
					\brief		Gets two segment points describing the phantom segment of a path
					\author		GW
					\date		08/2013

					\returns	A vector of the two segment points describing the segment

					\note		This method is indentical to the phantomSegPoints() method.
				*/
				ArtboardSegPointVector phantomSeg() const;

				/**
					\brief		Gets the two phantom segment points
					\author		GW
					\date		08/2013

					\returns	A vector of the two segment points describing the segment

					\note		This method is indentical to the phantomSeg() method.
				*/
				ArtboardSegPointVector phantomSegPoints() const;

				/**
					\brief	Sets a series of segments in the path to new values
					\author	GW
					\date	08/2013

					\param	segIndex_	Index number of the first segment to set
					\param	segPts_		Vector of new segment points describing the segments
					
					\pre	Path closedness/openness must already be set to the desired value, otherwise unexpected
							results may occur. If the closed/open state of a (new) path is not yet determinable, use the
							setSegPoints() method instead.

					\note	This method will either change existing segments, or create new ones if they don't exist yet.
					\note	The number of segments modified will be equal to the count of the points in segPts_ for
							closed paths, or the count minus one for open paths (since two points are needed to describe
							a single segment).
					\note	The maximum count for segPts_ is equal to maxSegments
				*/
				void setSegs(const int16_t segIndex_, const ArtboardSegPointVector& segPts_);

				/**
					\brief	Sets a series of segment points in the path to new values
					\author	GW
					\date	08/2013

					\param	segPtIndex_		Index number of the first segment point to set
					\param	segPts_			Vector of new segment points

					\note	This method will either change existing segment points, or create new ones if they don't
							exist yet.
					\note	The number of points modified will be equal to the count of the points in segPts_
					\note	The maximum count for segPts_ is equal to maxSegments
				*/
				void setSegPoints(const int16_t segPtIndex_, const ArtboardSegPointVector& segPts_);
				
				/**
					\brief		Gets whether all the segments points of a path are selected
					\author		GW
					\date		08/2013
					
					\returns	true if all the segments points are selected, false otherwise
				*/
				bool allSegPointsSelected() const;

				/**
					\brief		Gets whether a given segment is selected
					\author		GW
					\date		08/2013

					\param		segIndex_	Index number of the segment in question
					\returns	true if the segment is selected, false otherwise
						
					\note		A segment is selected if both its endpoints are fully selected
				*/
				bool segSelected(const int16_t segIndex_) const;

				/**
					\brief		Gets the selection state of a given segment point
					\author		GW
					\date		08/2013

					\param		segPtIndex_		Index number of the segment point in question
					\returns	A segment point selection state enum value to describe how the point is selected
					
					\note		It is worth noting that a segment point being selected individually by the direct
								selection tool will result in all control points being selected (and visibly so on the
								artboard), but a path's segment points being selected en masse by the regular selection
								tool will result in the Illustrator SDK saying that all control points are selected too
								(though only the anchor point will be visibly selected on the artboard). It is not
								possible to determine with the SDK whether a given segment point was selected by one
								tool or another, nor is it possible to determine that only the anchor point is visibly
								shown to be selected (and not the other two control points). As such, if you need to
								make a distinction between these two situations, it is best to make an initial pass
								through all of the segment points of a path and count how many appear to have all of
								their control points selected. If it is exactly one, then that individual segment point
								will indeed visibly show all of its control points as being selected. If the count is
								not one, then all of the "fully selected" control points will only visibly show their
								anchor point as being selected.
				*/
				SegPointSelectionState segPointSelected(const int16_t segPtIndex_) const;

				/**
					\brief		Sets the selection state of a given segment point
					\author		GW
					\date		03/2015

					\param		segPtIndex_		Index number of the segment point in question
					\param		selected_		New segment point selection state enum value
					\returns	true if the point was selected as indicated, false otherwise (see the documentation for
								the SegPointSelectionState enum for why this might occur)
				*/
				bool setSegPointSelected(const int16_t segPtIndex_, const SegPointSelectionState selected_);

				/**
					\brief	Inserts new segments into the target path at a given segment index
					\author	GW
					\date	08/2013

					\param	segIndex_	Segment index, in which the new segs will start
					\param	segPts_		Vector of new segment points describing the segments

					\note	For example, passing 0 for segIndex_ means that the new segs will be inserted between the
							existing seg pts 0 and 1.
					\note	The number of segments inserted will be equal to the count of the points in segPts_, minus
							one (since two points are needed to describe a single segment).
					\note	Negative index values, to insert points before the start of the path, can only be passed to
							the insertSegPoints() method.
				*/
				void insertSegs(const int16_t segIndex_, const ArtboardSegPointVector& segPts_);

				/**
					\brief	Inserts new segment points into the target path after a given segment point index
					\author	GW
					\date	08/2013

					\param	segPtIndex_		Segment point index, after which the new seg points will be inserted
					\param	segPts_			Vector of new segment points

					\note	For example, passing 0 for segPtIndex_ means that the new seg pts will be inserted between
							the existing seg pts 0 and 1.
					\note	The number of points inserted will be equal to the count of the points in segPts_
					\note	If a negative number is passed for segPtIndex_, and the count of segPts_ is equal to its
							absolute value, these new points will be inserted before the first point in the path.
				*/
				void insertSegPoints(const int16_t segPtIndex_, const ArtboardSegPointVector& segPts_);

				/**
					\brief	Deletes a given number of segments from a given segment index forward
					\author	GW
					\date	08/2013

					\param	segIndex_	Index number of the first segment to delete
					\param	count_		Number of segments to delete from segIndex_ forward (inclusive)
				*/
				void deleteSegs(const int16_t segIndex_, const int16_t count_);

				/**
					\brief	Deletes a given number of segment points from a given segment point index forward
					\author	GW
					\date	08/2013

					\param	segPtIndex_		Index number of the first segment point to delete
					\param	count_			Number of segment points to delete from segIndex_ forward (inclusive)
				*/
				void deleteSegPoints(const int16_t segPtIndex_, const int16_t count_);

				/**
					\brief	Rotates segment point indices by a given delta, while keeping the points at the same
							location in space
					\author	GW
					\date	08/2013

					\param	delta_	The number of positions to move each segment point

					\note	The rotation changes the segment point index by the given delta; e.g. if a path had point
							indices and locations 0:(0,1), 1:(2,3), 2:(4,5) and delta_ was 1, the points would shift to
							become 0:(2,3), 1:(4,5), 2:(0,1) instead.
				*/
				void rotateSegPoints(const int16_t delta_);

				/**
					\brief	Reverses the winding order of the segment points in the path
					\author	GW
					\date	08/2013
				*/
				void reverseSegs();

				/**
					\brief		Gets the winding order of the path
					\author		GW
					\date		08/2013
					
					\returns	An enum value for clockwise or counter-clockwise winding order
				*/
				WindingOrderType windingOrder() const;

				/**
					\brief	Sets the winding order of the path
					\author	GW
					\date	08/2013

					\param	winding_	New path winding order
					
					\note	If a specific winding order is desired, this method offers more control than the
							reverseSegs() method. For example, if a path is already clockwise, setting it to clockwise
							has no effect. If a path is counter-clockwise, setting it to clockwise will switch the
							winding order.
				*/
				void setWindingOrder(const WindingOrderType winding_);
				
				/**
					\brief	Describes how a path's stroke should be aligned
				*/
				enum StrokeAlignment
				{
					UnknownStrokeAlignment	= 0,
					CenterStrokeAlignment	= 10,
					InsideStrokeAlignment	= 20,
					OutsideStrokeAlignment	= 30
				};
			
				/**
					\brief	Sets the stroke alignment for the target path
					\author	GW
					\date	08/2014
					
					\param	align_	New stroke alignment setting
					
					\note	As of Illustrator 26, there is no getter for stroke alignment.
				*/
				void setStrokeAlignment(const StrokeAlignment align_);

				/**
					\brief	"Safely" inserts a new segment point in a given segment
					\author	GW
					\date	08/2013

					\param	segIndex_	Index number of the segment in which the new point will be added
					\param	t_			T-value along the given path segment at which the new point will be added
					
					\note	This method is considered "safe" because it keeps the same bezier curvature present at the
							provided t-value by calculating the proper in- and out-point positions.
				*/
				void insertSegPointAtT(const int16_t segIndex_, const double t_);

				/**
					\brief		Gets the index of the segment point that is closest to a given point in space
					\author		GW
					\date		08/2013

					\param		pt_		The point in question
					\returns	The segment point index closest to the provided point, or -1 for error
				*/
				int16_t segPointClosestToPoint(const ArtboardPoint& pt_) const;

				/**
					\brief		Gets whether a given point is contained inside the path
					\author		GW
					\date		08/2013
					
					\param		pt_		The point in question
					\returns	true if the given point lies within the path, false otherwise
				*/
				bool containsPoint(const ArtboardPoint& pt_) const;

				/**
					\brief		Attempts to reduce the number of segment points in the path by converting them to a
								series of points and then re-fitting curves to the points. Any corner segments are
								preserved.
					\author		GW
					\date		09/2021
				 
				 	\param		flatness_	An upper bound on the error between the line segments and the true curve
					\returns	true if the simplification was successful, false otherwise
				*/
				bool simplify(const double flatness_);

				/**
					\brief		Gets all path segment intersections along the target path
					\author		GW
					\date		08/2013

					\param		max_		If desired, limits the max count of intersections, in whichever order they
											are found first; pass 0 for "unlimited"
					\param		guides_		Whether guides should be included in the search
					\returns	A vector of HitData objects describing each intersection point along the target path
								(including intersections with itself). Inspect each HitData object in the vector for
								info about which piece of art intersects the target path, and how it intersects the path.
				*/
				HitDataVector getIntersections(const int16_t max_ = 0, const bool guides_ = false) const;

				/**
					\brief		Gets path segment intersections along the target path
					\details	This method only gets intersections between a path and a provided target piece of art
					\author		GW
					\date		08/2013

					\param		target_		Piece of art to check against for intersections with the target path
					\param		max_		If desired, limits the max count of intersections, in whichever order they
											are found first; pass 0 for "unlimited"
					\param		guides_		Whether guides should be included in the search
					\returns	A vector of HitData objects describing each intersection point along the target path.
								Each intersection, if any, will only contain target_ as the intersecting art. Inspect
								each HitData object in the vector for info about how it intersects the path.
				*/
				HitDataVector getIntersections(
					const Art& target_,
					const int16_t max_ = 0,
					const bool guides_ = false
				) const;

				/**
					\brief		Trims a path around a given "center" point, either to one or both sides
					\author		GW
					\date		08/2013

					\param		center_		HitData object describing the "center" from which the trims will take place
					\param		delete_		Whether portion(s) of the path beyond the intersection(s) should be deleted
					\param		dir_		Which direction the trim(s) should take place
					\param		guides_		Whether guides should be taken into account for trimming
					\param		split__		Return-by-reference for the second piece of path art, if necessary
					\returns	An error code for the result of the trim operation. Closed paths cannot be trimmed, and
								"paths" with only one segment point cannot be trimmed. Successful cases return
								NoTrimError.
					
					\note		Intersection(s) nearest center_, on the path, will be trimmed. The art that center_
								contains must be the same as the art to which the target path belongs.
					\note		Pass true for delete_ to remove portions of the path beyond the intersection(s), or
								false to insert new segment points where the deletion would have otherwise occurred.
					\note		The dir_ arg can be used to perform a trim on only one side of a line, regardless of
								intersections existing on the other side.
					\note		split__ will be set to the secondary piece of path art after a trim that results in the
								path being split (i.e. deleting pts in the middle). It will first be set to an empty Art
								object by default.
					\note		If some art should not be considered for trimming, simply lock or hide it before calling
								the trim() method.
				*/
				TrimError trim(
					const HitData& center_,
					const bool delete_,
					const TrimDirectionType dir_,
					const bool guides_,
					Art& split__
				);

				/**
					\brief		Extends a path from a given "point" to another "point"
					\author		GW
					\date		08/2013

					\param		from_	HitData object describing the point from which the extension should take place
					\param		to_		HitData object describing the path and segment to which the target should be
										extended
					\param		guides_	Whether guides should be included when extending
					\returns	An error code for the result of the extend operation. Closed paths cannot be extended,
								"paths" with only one segment point cannot be extended, only the endpoints of a path can
								be extended, and a path cannot be extended to another path parallel to it. Successful
								cases return NoExtendError.

					\note		The art that from_ contains must be the same as the art to which the target path belongs.
					\note		If some art should not be considered for extending, simply lock or hide it before
								calling the extend() method.
				*/
				ExtendError extend(const HitData& from_, const HitData& to_, const bool guides_);


			private:
				// Only Art can construct a new PathArt object
				friend class Art;

				/**
					\brief	Art to which this path belongs
				*/
				void* __data;
				
				/**
					\brief	Empty PathArt objects cannot be constructed
					\author	GW
					\date	08/2013
				*/
				PathArt();

				/**
					\brief	Constructs a new PathArt object from a given Art object
					\author	GW
					\date	08/2013

					\param	art_	Pointer for the existing art we're wrapping around
				*/
				PathArt(void* const art_);
			
				/**
					\brief	PathArt objects cannot be copied
					\author	GW
					\date	08/2013
				*/
				PathArt(const PathArt&);
			
				/**
					\brief	PathArt objects cannot be copied
					\author	GW
					\date	08/2013
				*/
				PathArt& operator=(const PathArt&);
		};
	}
}

hdi::core::PathArt::SegPointSelectionState operator|(
	const hdi::core::PathArt::SegPointSelectionState lhs_,
	const hdi::core::PathArt::SegPointSelectionState rhs_
);

hdi::core::PathArt::SegPointSelectionState& operator|=(
	hdi::core::PathArt::SegPointSelectionState& lhs__,
	const hdi::core::PathArt::SegPointSelectionState rhs_
);

hdi::core::PathArt::TrimDirectionType operator|(
	const hdi::core::PathArt::TrimDirectionType lhs_,
	const hdi::core::PathArt::TrimDirectionType rhs_
);

hdi::core::PathArt::TrimDirectionType& operator|=(
	hdi::core::PathArt::TrimDirectionType& lhs__,
	const hdi::core::PathArt::TrimDirectionType rhs_
);

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_PATH_ART__
