/**
	\file
	\brief		Header file for path style preference data storage
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_PATH_STYLE_PREF__
#define __HDI_CORE_PATH_STYLE_PREF__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicorePathStyle.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Allows for storage of path styles
		*/
		class PathStylePref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty PathStylePref object
					\author	GW
					\date	11/2015
					
					\note	To test if a PathStylePref object is empty, call isEmpty() on it
					\note	Empty PathStylePref objects do not relate to any actual preference data; they are designed to
							be "receivers" of some other PathStylePref object via the overloaded assignment operator.
							Empty PathStylePref objects are useless until such time (though it is safe to call any of
							their methods)
				*/
				PathStylePref();
			
				/**
					\brief	Constructs a new PathStylePref object from an existing PathStylePref object (copy constructor)
					\author	GW
					\date	11/2015

					\param	ps_		Existing PathStylePref object
				*/
				PathStylePref(const PathStylePref& ps_);

				/**
					\brief	Constructs a PathStylePref object, with new data, to be added to a container at a later time
					\author	GW
					\date	11/2015

					\param	value_	Initial path style value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				PathStylePref(const PathStyle& value_, const std::string& name_ = "");

				/**
					\brief	Destructs a PathStylePref object
					\author	GW
					\date	11/2015
				*/
				virtual ~PathStylePref();

				/**
					\brief		Allows one PathStylePref object to be assigned from another
					\author		GW
					\date		11/2015

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target PathStylePref object, but with its value updated to match that of rhs_
				*/
				virtual PathStylePref& operator=(const PathStylePref& rhs_);
				
				/**
					\brief		Convenience method to clone an PathStylePref object on the heap
					\author		GW
					\date		11/2015

					\returns	A pointer to the new PathStylePref object
					
					\note		If you subclass PathStylePref, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your PathStylePref subclass, you will
								experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned PathStylePref object.
				*/
				virtual PathStylePref* clone() const;

				/**
					\brief		Gets the type of persistent data
					\author		GW
					\date		11/2015

					\returns	PathStyleType

					\note		This is useful for knowing how to cast a PrefData object to its proper subclass
				*/
				virtual Type dataType() const;

				/**
					\brief		Gets the path style value for this object
					\author		GW
					\date		11/2015

					\returns	The path style value being described by this wrapper
				*/
				virtual PathStyle pathStyle() const;

				/**
					\brief		Sets the path style value for this object
					\author		GW
					\date		11/2015

					\param		value_	New path style value
					\returns	true if the path style is successfully updated, or false otherwise
				*/
				virtual bool setPathStyle(const PathStyle& value_);
		};
		
		typedef std::unique_ptr<PathStylePref> PathStylePrefUP;
		typedef std::shared_ptr<PathStylePref> PathStylePrefSP;
		typedef std::weak_ptr<PathStylePref> PathStylePrefWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_PATH_STYLE_PREF__
