/**
	\file
	\brief		Header file for general Illustrator pattern manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_PATTERN__
#define __HDI_CORE_PATTERN__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreArtboardRect.h"
#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class Pattern;
	}

	namespace core
	{
		class Art;
	
		/**
			\brief	Handles general pattern-related functionality
		*/
		class Pattern
		{
			public:
				/**
					\brief	Indicates the type of "pattern" an object is representing
				*/
				enum Type
				{
					UnknownPatternType	= 0,
					NormalPatternType	= 10,
					SymbolPatternType	= 20,
					BrushPatternType	= 30
				};

				/**
					\brief	Constructs an empty Pattern object
					\author	GW
					\date	11/2013
					
					\note	To test if a Pattern object is empty, call isEmpty() on it
					\note	Empty Pattern objects do not relate to any pattern in a Illustrator document; they are
							designed to be "receivers" of some other Pattern object via the overloaded assignment
							operator. Empty Pattern objects are useless until such time (though it is safe to call any
							of their methods).
				*/
				Pattern();
			
				/**
					\brief	Constructs a new Pattern object from an existing Pattern object (copy constructor)
					\author	GW
					\date	11/2013

					\param	patt_	Existing Pattern object
				*/
				Pattern(const Pattern& patt_);
				
				/**
					\brief		Creates a new Pattern object (and Illustrator pattern); set the source art for it with
								the setSourceArt() method
					\author		GW
					\date		11/2013
					
					\returns	A new Pattern object
				*/
				static Pattern create();

				/**
					\brief	Pattern object destructor
					\author	GW
					\date	11/2013
				*/
				virtual ~Pattern();
				
				/**
					\brief		Assigns one Pattern object to another
					\author		GW
					\date		11/2013

					\param		rhs_	Existing Pattern object to copy values from
					\returns	The target Pattern object, but with its values updated to match that of the rhs_ argument
				*/
				virtual Pattern& operator=(const Pattern& rhs_);

				/**
					\brief		Removes the pattern from its document
					\author		GW
					\date		11/2013
					
					\returns	true for successful disposal, false otherwise

					\note		After calling dispose(), this object is basically useless! The actual pattern is gone,
								so obviously none of these methods will work after that.
				*/
				virtual bool dispose();

				/**
					\brief		Gets the pattern handle around which the target object is wrapped
					\author		GW
					\date		11/2013
					
					\returns	AIPatternHandle for this object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual AIPatternHandle aiPatternHandle() const;

				/**
					\brief		Gets whether the target Pattern object is empty (constructed with the default ctor)
					\author		GW
					\date		11/2013
					
					\returns	true if the target Pattern object is empty, false otherwise
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Tests the validity of the target pattern by searching for it in the document
					\author		GW
					\date		11/2013
					
					\returns	true for a valid pattern, false otherwise
				*/
				virtual bool valid() const;
				
				/**
					\brief		Gets the type of pattern the target object is representing
					\author		GW
					\date		12/2013
					
					\returns	A Type enum value indicating the type of the underlying pattern
					
					\note		Typically, when SymbolPatternType is returned, you can cast the target to a Symbol
								object for more refined manipulation. However, it is recommended that you still use
								dynamic_cast() to be certain the cast is warranted.
				*/
				virtual Type patternType() const;
				
				/**
					\brief		Tests whether a given Pattern object is the same as another
					\author		GW
					\date		11/2013

					\param		rhs_	Pattern to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same pattern, false otherwise
				*/
				virtual bool operator==(const Pattern& rhs_) const;
				
				/**
					\brief		Tests whether a given Pattern object is not the same as another
					\author		GW
					\date		11/2013

					\param		rhs_	Pattern to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different patterns, false otherwise
				*/
				virtual bool operator!=(const Pattern& rhs_) const;
				
				/**
					\brief		Gets the Art object that defines the source art for the pattern
					\author		GW
					\date		11/2013
					
					\returns	An Art object for the source definition of the pattern
				*/
				virtual std::unique_ptr<Art> sourceArt() const;
				
				/**
					\brief	Sets the Art object that defines the source art for the pattern
					\author	GW
					\date	11/2013
					
					\param	art_	New source art for the target Pattern object
				*/
				virtual void setSourceArt(const Art& art_);
				
				/**
					\brief		Gets the name of the pattern
					\author		GW
					\date		11/2013
					
					\returns	The pattern's name, as UTF-8
				*/
				virtual std::string name() const;
				
				/**
					\brief	Sets the name of the pattern
					\author	GW
					\date	11/2013
					
					\param	name_	New name for the pattern, as UTF-8
				*/
				virtual void setName(const std::string& name_);
				
				/**
					\brief		Gets the bounds rect for the target pattern
					\author		GW
					\date		11/2013
					
					\returns	A rect object describing the pattern's bounds
				*/
				virtual ArtboardRect bounds() const;

				/**
					\brief		Gets whether the target pattern is currently being edited in pattern editing mode
					\author		GW
					\date		11/2015
					
					\returns	true if currently being edited, false otherwise
				*/
				virtual bool isBeingEdited() const;

				/**
					\brief		Puts the document into pattern editing mode, with the target pattern being the one to edit
					\author		GW
					\date		11/2015
					
					\param		inst_	Ignored
					\returns	true if pattern editing mode was entered successfully, false otherwise
				*/
				virtual bool setBeingEdited(const Art* const inst_ = NULL);


			protected:
				/**
					\brief	Private implementation data
				*/
				void* _data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	11/2013
				*/
				void* _impl() const;


			private:
				friend aip::Pattern* __accessImpl(const Pattern&);
				friend Pattern __accessCtor(aip::Pattern*&);

				/**
					\brief	Internal use only
					\author	GW
					\date	11/2013
				*/
				Pattern(aip::Pattern*&);
		};
		
		typedef std::unique_ptr<Pattern> PatternUP;
		typedef std::shared_ptr<Pattern> PatternSP;
		typedef std::weak_ptr<Pattern> PatternWP;
		
		extern aip::Pattern* __accessImpl(const Pattern&);
		extern Pattern __accessCtor(aip::Pattern*&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_PATTERN__
