/**
	\file
	\brief		Header file for manipulation of the plugin itself
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_PLUGIN__
#define __HDI_CORE_PLUGIN__

#include <vector>

#include "hdicoreMain.h"

/**
	\brief	Slightly shorter way to access the Plugin instance, but still utilizing a non-colliding name scheme
*/
#define HDI_CORE_PLUGIN	hdi::core::Plugin::instance()

namespace hdi
{
	namespace core
	{
		class Callback;
		class Dispatcher;
		class MenuItem;
		class ModalDialog;
		class Tool;
	
		/**
			\brief		Base plugin class; the heart of any plugin project
			\details	Receives startup/shutdown messages, tool messages, menu messages, etc.; allows for preference
						access, registration access, etc.
		*/
		class Plugin
		{
			public:
				typedef std::vector<byte> ByteVector;
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief	Allows for configuration of various plug-in behaviors
					*/
					enum Options
					{
						NoOptions							= 0,
						AutoSelectsResultsOption			= 1 << 0,
						RetainsPartialPathSelectionOption	= 1 << 1
					};
					
					/**
						\brief	Allows for setting of the undo context to other behaviors
					*/
					enum UndoContext
					{
						UnknownUndoContext	= 0,
						
						// A standard undo context results in the addition of a new transaction which can be undone/redone
						// by the user
						StandardUndoContext	= 10,
						
						// A silent context does not cause redos to be discarded and is skipped over when undoing and
						// redoing. An example is a selection change.
						SilentUndoContext	= 20,
						
						// An appended context is like a standard context, except that it is combined with the preceding
						// transaction. It does not appear as a separate transaction. Used, for example, to collect
						// sequential changes to the color of an object into a single undo/redo transaction.
						AppendUndoContext	= 30
					};
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Allows access to the singleton Plugin instance object
					\author		GW
					\date		09/2013
					
					\returns	The singleton Plugin object
				*/
				static Plugin* const instance();

				/**
					\brief		Gets the plugin ref around which the target object is wrapped
					\author		GW
					\date		11/2013
					
					\returns	SPPluginRef for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by the app is not handled by this
								class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual SPPluginRef spPluginRef() const;

				/**
					\brief		Gets a pointer to the platform object representing the current Plugin instance
					\author		GW
					\date		12/2015

					\returns	A pointer to the platform object for the plugin instance (bundle on Mac, DLL on Windows)
				*/
				PlatformPluginPtr platformPlugin() const;

				/**
					\brief		Gets the full path of the plugin file at runtime
					\author		GW
					\date		09/2013
					
					\returns	Plugin file path as UTF-8, e.g. "/Applications/Adobe Illustrator CS6/Plug-ins/Tools/MyPlugin.aip"
				*/
				virtual std::string filePath() const;

				/**
					\brief		Gets the name of the plugin file at runtime
					\author		GW
					\date		09/2013
					
					\returns	Plugin file name as UTF-8, e.g. "MyPlugin.aip"
				*/
				virtual std::string fileName() const;

				/**
					\brief		Gets the plugin name
					\author		GW
					\date		09/2013
					
					\returns	The plugin name, as UTF-8
				*/
				virtual std::string name() const;

				/**
					\brief	Sets the plugin name
					\author	GW
					\date	09/2013
					
					\param	name_	The new plugin name, as UTF-8
				*/
				virtual void setName(const std::string& name_);

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the plugin's options
						\author		GW
						\date		09/2013
						
						\returns	Current plugin options
					*/
					virtual Options options() const;

					/**
						\brief	Sets the plugin's options
						\author	GW
						\date	09/2013

						\param	options_	New options for the plugin
					*/
					virtual void setOptions(const Options options_);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Convenience method to return the Dispatcher instance
					\author		GW
					\date		09/2013

					\returns	The plugin message dispatcher instance
				*/
				virtual Dispatcher* const dispatcher() const;

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a pointer to the currently selected tool
						\author		GW
						\date		09/2013
						
						\returns	The currently selected tool, or NULL if none
					*/
					virtual Tool* const currentTool() const;

					/**
						\brief	Undoes the last set of artwork changes by this specific plugin
						\author	GW
						\date	09/2013
					*/
					virtual void undo() const;
					
					/**
						\brief		Changes the undo context to another type
						\author		GW
						\date		12/2014
						
						\param		uc_		New undo context
						\returns	true if the context was changed, false otherwise
					*/
					virtual bool setUndoKind(const UndoContext uc_);
					
					/**
						\brief		Sets the text for the Undo and Redo menu items, which Illustrator uses as needed
						\author		GW
						\date		01/2023
						
						\param		undo_	String to display for undo item, as UTF-8
						\param		redo_	String to display for redo item, as UTF-8
					*/
					virtual void setUndoRedoNames(const std::string& undo_, const std::string& redo_);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Acquires a byte vector for a given resource
					\author		GW
					\date		09/2013

					\param		type_	Type of resource (four-byte code, e.g. 'PNGI')
					\param		id_		ID number of the resource
					\param		data__	Return-by-reference for the resource data
					\returns	true if the resource could be read, false otherwise
					
					\note		A maximum size of 10MB is supported; resource data any larger will be truncated.
				*/
				virtual bool resourceData(const int32_t type_, const int16_t id_, ByteVector& data__) const;

				/**
					\brief		Acquires a byte vector for a given resource
					\author		GW
					\date		09/2013
					
					\param		type_	Type of resource (four-byte code, e.g. 'PNGI')
					\param		name_	Name of the resource
					\param		data__	Return-by-reference for the resource data
					\returns	true if the resource could be read, false otherwise
					
					\note		A maximum size of 10MB is supported; resource data any larger will be truncated.
				*/
				virtual bool resourceData(
					const int32_t type_,
					const std::string& name_,
					ByteVector& data__
				) const;
				
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets whether tools belonging to this plugin should snap the cursor to artwork
						\author		GW
						\date		09/2013
						
						\returns	true if tools should snap the cursor to artwork, false otherwise
						
						\note		This is merely an indicator of whether snapping will occur (it happens automatically);
									the caller need not manually snap anything.
					*/
					virtual bool toolsSnapToArt() const;
					
					/**
						\brief	Sets whether tools belonging to this plugin should snap the cursor to artwork
						\author	GW
						\date	09/2013
						
						\param	snap_	true if tools should snap the cursor to artwork, false otherwise
					*/
					virtual void setToolsSnapToArt(const bool snap_);
					
					/**
						\brief		Gets whether tools belonging to this plugin should snap the cursor to artwork bounds
						\author		GW
						\date		09/2013
						
						\returns	true if tools should snap the cursor to artwork bounds, false otherwise
						
						\note		This is merely an indicator of whether snapping will occur (it happens automatically);
									the caller need not manually snap anything.
					*/
					virtual bool toolsSnapToArtBounds() const;
					
					/**
						\brief	Sets whether tools belonging to this plugin should snap the cursor to artwork bounds
						\author	GW
						\date	09/2013
						
						\param	snap_	true if tools should snap the cursor to artwork bounds, false otherwise
					*/
					virtual void setToolsSnapToArtBounds(const bool snap_);
					
					/**
						\brief		Gets whether tools belonging to this plugin should snap the cursor to the artboard edges
						\author		GW
						\date		09/2013
						
						\returns	true if tools should snap the cursor to artboard edges, false otherwise
						
						\note		This is merely an indicator of whether snapping will occur (it happens automatically);
									the caller need not manually snap anything.
					*/
					virtual bool toolsSnapToArtboard() const;
					
					/**
						\brief	Sets whether tools belonging to this plugin should snap the cursor to the artboard edges
						\author	GW
						\date	09/2013
						
						\param	snap_	true if tools should snap the cursor to artboard edges, false otherwise
					*/
					virtual void setToolsSnapToArtboard(const bool snap_);
					
					/**
						\brief		Gets whether tools belonging to this plugin should snap the cursor to the Illustrator
									gridlines
						\author		GW
						\date		09/2013
						
						\returns	true if tools should snap the cursor to gridlines, false otherwise
						
						\note		This is merely an indicator of whether snapping will occur (it happens automatically);
									the caller need not manually snap anything.
					*/
					virtual bool toolsSnapToGrids() const;
					
					/**
						\brief	Sets whether tools belonging to this plugin should snap the cursor to the Illustrator
								gridlines
						\author	GW
						\date	09/2013
						
						\param	snap_	true if tools should snap the cursor to gridlines, false otherwise
					*/
					virtual void setToolsSnapToGrids(const bool snap_);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets whether the plugin is in the process of starting up
					\author		GW
					\date		04/2015

					\returns	true if the startup callback is currently executing, false otherwise
				*/
				virtual bool startingUp() const;

				/**
					\brief		Gets whether the plugin has already completed starting up
					\author		GW
					\date		04/2015

					\returns	true if the startup callback has completed
				*/
				virtual bool hasStartedUp() const;
				
				/**
					\brief		Gets whether the plugin is in the process of post-starting up
					\author		GW
					\date		04/2015

					\returns	true if the postStartup callback is currently executing, false otherwise
				*/
				virtual bool postStartingUp() const;
				
				/**
					\brief		Gets whether the plugin has already completed post-starting up
					\author		GW
					\date		04/2015

					\returns	true if the postStartup callback has completed
				*/
				virtual bool hasPostStartedUp() const;
				
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets whether the plugin is in the process of pre-shutting down
						\author		GW
						\date		04/2015

						\returns	true if the preShutdown callback is currently executing, false otherwise
					*/
					virtual bool preShuttingDown() const;
					
					/**
						\brief		Gets whether the plugin has already completed pre-shutting down
						\author		GW
						\date		04/2015

						\returns	true if the preShutdown callback has completed
					*/
					virtual bool hasPreShutdown() const;
				#endif
				// HDI_CORE_AIP_MODE
				
				/**
					\brief		Gets whether the plugin is in the process of shutting down
					\author		GW
					\date		04/2015

					\returns	true if the shutdown callback is currently executing, false otherwise
				*/
				virtual bool shuttingDown() const;
				
				/**
					\brief		Gets whether the plugin has already completed shutting down
					\author		GW
					\date		04/2015

					\returns	true if the shutdown callback has completed
				*/
				virtual bool hasShutdown() const;

				/**
					\brief		Gets a logging stream, by name, for debugging output for the plugin
					\author		GW
					\date		04/2019

					\param		name_	The unique name of the logging stream
					\returns	A pointer to the logging stream, or NULL for none (the default)

					\warning	The caller should not assume responsibility for the memory of the returned pointer!
				*/
				virtual std::ostream* const loggingStream(const std::string& name_) const;

				/**
					\brief		Sets the logging stream, by name, for debugging output for the plugin
					\author		GW
					\date		04/2019

					\param		name_	The unique name of the logging stream
					\param		str_	A pointer to the new logging stream, or NULL for none

					\warning	This method does not assume responsibility for the memory of the str_ argument; the
								caller should manage its memory and, when ready to delete the original object, call the
								setLoggingStream() method again and pass NULL.
				*/
				virtual void setLoggingStream(const std::string& name_, std::ostream* const str_);
				
				/**
					\brief		Gets the callback that's called when the plugin is starting up
					\author		GW
					\date		09/2013
					
					\returns	The startup callback
				*/
				virtual Callback* const startupCallback() const;
				
				/**
					\brief	Sets the callback that's called when the plugin is starting up
					\author	GW
					\date	09/2013
					
					\param	cb_		New startup callback
				*/
				virtual void setStartupCallback(const Callback& cb_);
				
				/**
					\brief		Gets the callback that's called just after the plugin has started up
					\author		GW
					\date		09/2013
					
					\returns	The post-startup callback
				*/
				virtual Callback* const postStartupCallback() const;
				
				/**
					\brief	Sets the callback that's called just after the plugin has started up
					\author	GW
					\date	09/2013
					
					\param	cb_		New post-startup callback
				*/
				virtual void setPostStartupCallback(const Callback& cb_);
				
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the callback that's called just before the plugin will shutdown
						\author		GW
						\date		09/2013
						
						\returns	The pre-shutdown callback
					*/
					virtual Callback* const preShutdownCallback() const;
					
					/**
						\brief	Sets the callback that's called just before the plugin will shutdown
						\author	GW
						\date	09/2013
						
						\param	cb_		New pre-shutdown callback
					*/
					virtual void setPreShutdownCallback(const Callback& cb_);
				#endif
				// HDI_CORE_AIP_MODE
				
				/**
					\brief		Gets the callback that's called when the plugin is shutting down
					\author		GW
					\date		09/2013
					
					\returns	The shutdown callback
				*/
				virtual Callback* const shutdownCallback() const;
				
				/**
					\brief	Sets the callback that's called when the plugin is shutting down
					\author	GW
					\date	09/2013
					
					\param	cb_		New shutdown callback
				*/
				virtual void setShutdownCallback(const Callback& cb_);

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the callback that's called when a user selects one of the plugin's menu items
						\author		GW
						\date		09/2013
						
						\returns	The menu item selection callback
					*/
					virtual Callback* const menuItemSelectedCallback() const;

					/**
						\brief	Sets the callback that's called when a user selects one of the plugin's menu items
						\author	GW
						\date	09/2013
						
						\param	cb_		New menu item selection callback
					*/
					virtual void setMenuItemSelectedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called when a user hovers over menu groups that will show this
									plugin's menu items
						\author		GW
						\date		09/2013
						
						\returns	The menu items update callback
					*/
					virtual Callback* const menuItemsUpdateCallback() const;

					/**
						\brief	Sets the callback that's called when a user hovers over menu groups that will show this
								plugin's menu items
						\author	GW
						\date	09/2013
						
						\param	cb_		New menu items update callback
					*/
					virtual void setMenuItemsUpdateCallback(const Callback& cb_);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets the callback that's called when the brightness of the app UI is changed via the app
								preference dialog
					\author		GW
					\date		09/2013
					
					\returns	The brightness adjusted callback
				*/
				virtual Callback* const brightnessAdjustedCallback() const;

				/**
					\brief	Sets the callback that's called when the brightness of the app UI is changed via the app
							preference dialog
					\author	GW
					\date	09/2013
					
					\param	cb_		New brightness change callback
				*/
				virtual void setBrightnessAdjustedCallback(const Callback& cb_);

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the callback that's called just before a tool is selected from the toolbar (i.e.
									the tool's selection callback has not been called)
						\author		GW
						\date		09/2013
						
						\returns	The tool pre-selected callback
					*/
					virtual Callback* const toolPreSelectedCallback() const;

					/**
						\brief	Sets the callback that's called just before a tool is selected from the toolbar (i.e. the
								tool's selection callback has not been called)
						\author	GW
						\date	09/2013
						
						\param	cb_		New tool pre-selection callback
					*/
					virtual void setToolPreSelectedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just after a tool is selected from the toolbar (i.e.
									the tool's selection callback has already been called)
						\author		GW
						\date		09/2013
						
						\returns	The tool post-selected callback
					*/
					virtual Callback* const toolPostSelectedCallback() const;

					/**
						\brief	Sets the callback that's called just after a tool is selected from the toolbar (i.e. the
								tool's selection callback has already been called)
						\author	GW
						\date	09/2013
						
						\param	cb_		New tool post-selection callback
					*/
					virtual void setToolPostSelectedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just before a tool is deselected (i.e. the tool's
									deselection callback has not been called)
						\author		GW
						\date		09/2013
						
						\returns	The tool pre-deselected callback
					*/
					virtual Callback* const toolPreDeselectedCallback() const;

					/**
						\brief	Sets the callback that's called just before a tool is deselected (i.e. the tool's deselection
								callback has not been called)
						\author	GW
						\date	09/2013
						
						\param	cb_		New tool pre-deselection callback
					*/
					virtual void setToolPreDeselectedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just after a tool is deselected (i.e. the tool's
									deselection callback has already been called)
						\author		GW
						\date		01/2015

						\returns	The tool post-deselected callback
					*/
					virtual Callback* const toolPostDeselectedCallback() const;

					/**
						\brief	Sets the callback that's called just after a tool is deselected (i.e. the tool's deselection
								callback has already been called)
						\author	GW
						\date	01/2015

						\param	cb_		New tool post-deselection callback
					*/
					virtual void setToolPostDeselectedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just before a tool is reselected (i.e. the tool's
									reselection callback has not been called)
						\author		GW
						\date		09/2013
						
						\returns	The tool pre-reselected callback
					*/
					virtual Callback* const toolPreReselectedCallback() const;

					/**
						\brief	Sets the callback that's called just before a tool is reselected (i.e. the tool's reselection
								callback has not been called)
						\author	GW
						\date	09/2013
						
						\param	cb_		New tool pre-reselection callback
					*/
					virtual void setToolPreReselectedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just after a tool is reselected (i.e. the tool's
									reselection callback has already been called)
						\author		GW
						\date		01/2015

						\returns	The tool post-reselected callback
					*/
					virtual Callback* const toolPostReselectedCallback() const;

					/**
						\brief	Sets the callback that's called just after a tool is reselected (i.e. the tool's reselection
								callback has already been called)
						\author	GW
						\date	01/2015

						\param	cb_		New tool post-reselection callback
					*/
					virtual void setToolPostReselectedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just before the cursor moves (with the mouse button
									released; i.e. the tool's track callback has not been called)
						\author		GW
						\date		09/2013
						
						\returns	The tool pre-tracked callback
					*/
					virtual Callback* const toolPreTrackedCallback() const;

					/**
						\brief	Sets the callback that's called just before the cursor moves (with the mouse button released;
								i.e. the tool's track callback has not been called)
						\author	GW
						\date	09/2013
						
						\param	cb_		New tool pre-tracking callback
					*/
					virtual void setToolPreTrackedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just after the cursor moves (with the mouse button
									released; i.e. the tool's track callback has already been called)
						\author		GW
						\date		01/2015

						\returns	The tool post-tracked callback
					*/
					virtual Callback* const toolPostTrackedCallback() const;

					/**
						\brief	Sets the callback that's called just after the cursor moves (with the mouse button released;
								i.e. the tool's track callback has already been called)
						\author	GW
						\date	01/2015

						\param	cb_		New tool post-tracking callback
					*/
					virtual void setToolPostTrackedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just before the mouse button goes down (i.e. the tool's
									mouse down callback has not been called)
						\author		GW
						\date		09/2013
						
						\returns	The tool pre-mouse down callback
					*/
					virtual Callback* const toolPreMouseDownCallback() const;

					/**
						\brief	Sets the callback that's called just before the mouse button goes down (i.e. the tool's
									mouse down callback has not been called)
						\author	GW
						\date	09/2013
						
						\param	cb_		New tool pre-mouse down callback
					*/
					virtual void setToolPreMouseDownCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just after the mouse button goes down (i.e. the tool's
									mouse down callback has already been called)
						\author		GW
						\date		01/2015

						\returns	The tool post-mouse down callback
					*/
					virtual Callback* const toolPostMouseDownCallback() const;

					/**
						\brief	Sets the callback that's called just after the mouse button goes down (i.e. the tool's mouse
								down callback has already been called)
						\author	GW
						\date	01/2015

						\param	cb_		New tool post-mouse down callback
					*/
					virtual void setToolPostMouseDownCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just before the mouse button is down and the cursor
									moves (i.e. the tool's drag callback has not been called)
						\author		GW
						\date		09/2013
						
						\returns	The tool pre-dragged callback
					*/
					virtual Callback* const toolPreDraggedCallback() const;

					/**
						\brief	Sets the callback that's called just before the mouse button is down and the cursor moves
								(i.e. the tool's drag callback has not been called)
						\author	GW
						\date	09/2013
						
						\param	cb_		New tool pre-dragging callback
					*/
					virtual void setToolPreDraggedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just after the mouse button is down and the cursor moves
									(i.e. the tool's drag callback has already been called)
						\author		GW
						\date		01/2015

						\returns	The tool post-dragged callback
					*/
					virtual Callback* const toolPostDraggedCallback() const;

					/**
						\brief	Sets the callback that's called just after the mouse button is down and the cursor moves
								(i.e. the tool's drag callback has already been called)
						\author	GW
						\date	01/2015

						\param	cb_		New tool post-dragging callback
					*/
					virtual void setToolPostDraggedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just before the mouse button comes up (i.e. the tool's
									mouse up callback has not been called)
						\author		GW
						\date		09/2013
						
						\returns	The tool pre-mouse up callback
					*/
					virtual Callback* const toolPreMouseUpCallback() const;

					/**
						\brief	Sets the callback that's called just before the mouse button comes up (i.e. the tool's
									mouse up callback has not been called)
						\author	GW
						\date	09/2013
						
						\param	cb_		New tool pre-mouse up callback
					*/
					virtual void setToolPreMouseUpCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just after the mouse button comes up (i.e. the tool's
									mouse up callback has already been called)
						\author		GW
						\date		01/2015

						\returns	The tool post-mouse up callback
					*/
					virtual Callback* const toolPostMouseUpCallback() const;

					/**
						\brief	Sets the callback that's called just after the mouse button comes up (i.e. the tool's mouse
								up callback has already been called)
						\author	GW
						\date	01/2015

						\param	cb_		New tool post-mouse up callback
					*/
					virtual void setToolPostMouseUpCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just before the '[' key is hit while a tool is selected
									(i.e. the tool's diameter decrease callback has not been called)
						\author		GW
						\date		09/2013
						
						\returns	The tool diameter pre-decreased callback
					*/
					virtual Callback* const toolPreDiameterDecreasedCallback() const;

					/**
						\brief	Sets the callback that's called just before the '[' key is hit while a tool is selected (i.e.
								the tool's diameter decrease callback has not been called)
						\author	GW
						\date	09/2013
						
						\param	cb_		New tool diameter pre-decreased callback
					*/
					virtual void setToolPreDiameterDecreasedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just after the '[' key is hit while a tool is selected
									(i.e. the tool's diameter decrease callback has already been called)
						\author		GW
						\date		01/2015

						\returns	The tool diameter post-decreased callback
					*/
					virtual Callback* const toolPostDiameterDecreasedCallback() const;

					/**
						\brief	Sets the callback that's called just after the '[' key is hit while a tool is selected (i.e.
								the tool's diameter decrease callback has already been called)
						\author	GW
						\date	01/2015

						\param	cb_		New tool diameter post-decreased callback
					*/
					virtual void setToolPostDiameterDecreasedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just before the ']' key is hit while a tool is selected
									(i.e. the tool's diameter increase callback has not been called)
						\author		GW
						\date		09/2013
						
						\returns	The tool diameter pre-increased callback
					*/
					virtual Callback* const toolPreDiameterIncreasedCallback() const;

					/**
						\brief	Sets the callback that's called just before the ']' key is hit while a tool is selected (i.e.
								the tool's diameter increase callback has not been called)
						\author	GW
						\date	09/2013
						
						\param	cb_		New tool diameter pre-increased callback
					*/
					virtual void setToolPreDiameterIncreasedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just after the ']' key is hit while a tool is selected
									(i.e. the tool's diameter increase callback has already been called)
						\author		GW
						\date		01/2015

						\returns	The tool diameter post-increased callback
					*/
					virtual Callback* const toolPostDiameterIncreasedCallback() const;

					/**
						\brief	Sets the callback that's called just after the ']' key is hit while a tool is selected (i.e.
								the tool's diameter increase callback has already been called)
						\author	GW
						\date	01/2015

						\param	cb_		New tool diameter post-increased callback
					*/
					virtual void setToolPostDiameterIncreasedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just before a tool is double-clicked in the toolbar (i.e.
									the tool's edit callback has not been called)
						\author		GW
						\date		09/2013
						
						\returns	The tool pre-edited callback
					*/
					virtual Callback* const toolPreEditedCallback() const;

					/**
						\brief	Sets the callback that's called just before a tool is double-clicked in the toolbar (i.e.
								the tool's edit callback has not been called)
						\author	GW
						\date	09/2013
						
						\param	cb_		New tool pre-editing callback
					*/
					virtual void setToolPreEditedCallback(const Callback& cb_);

					/**
						\brief		Gets the callback that's called just after a tool is double-clicked in the toolbar (i.e.
									the tool's edit callback has already been called)
						\author		GW
						\date		01/2015

						\returns	The tool post-edited callback
					*/
					virtual Callback* const toolPostEditedCallback() const;

					/**
						\brief	Sets the callback that's called just after a tool is double-clicked in the toolbar (i.e. the
								tool's edit callback has already been called)
						\author	GW
						\date	01/2015

						\param	cb_		New tool post-editing callback
					*/
					virtual void setToolPostEditedCallback(const Callback& cb_);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets the about box modal dialog for the plugin
					\author		GW
					\date		09/2014
					
					\returns	A pointer to the about box dialog for the plugin, or NULL if none/error
					
					\note		For Illustrator, see also the aboutMenuItem() method. It is common for plugins to have
								about box modal dialogs that are shown when an "About <Your Plugin Name>" menu item is
								chosen; however, this process is not automated for the creator of the plugin as not all
								plugins necessitate this feature. The Plugin class merely offers a convenient place to
								store said dialog object.
					\note		For Photoshop, the process of showing an about box is automated for the creator of the
								plugin, so long as this method does not return NULL.
				*/
				virtual ModalDialog* const aboutBox() const;
				
				/**
					\brief	Sets the about box modal dialog for the plugin
					\author	GW
					\date	09/2014
					
					\param	dialog_	New about box modal dialog for the plugin
					
					\note	For Illustrator, see also the setAboutMenuItem() method. It is common for plugins to have
							about box modal dialogs that are shown when an "About <Your Plugin Name>" menu item is chosen;
							however, this process is not automated for the creator of the plugin as not all plugins
							necessitate this feature. The Plugin class merely offers a convenient place to store said
							dialog object.
					\note	For Photoshop, the process of showing an about box is automated for the creator of the plugin,
							so long as the aboutBox() method does not return NULL.
				*/
				virtual void setAboutBox(const ModalDialog& dialog_);

				#if defined(HDI_CORE_PSP_MODE)
					/**
						\brief		Gets the callback that's called just before the Photoshop plugin about box dialog is
									automatically shown
						\author		GW
						\date		01/2018

						\returns	The about box pre-show callback
					*/
					virtual Callback* const aboutBoxShowingCallback() const;
			
					/**
						\brief	Sets the callback that's called just before the Photoshop plugin about box dialog is
								automatically shown
						\author	GW
						\date	01/2018

						\param	cb_		New tool diameter post-decreased callback
					*/
					virtual void setAboutBoxShowingCallback(const Callback& cb_);
				#endif
				// HDI_CORE_PSP_MODE

				/**
					\brief		Gets the global prefs modal dialog for the plugin
					\author		GW
					\date		09/2014
					
					\returns	A pointer to the prefs dialog for the plugin, or NULL if none/error
					
					\note		See also the prefsMenuItem() method.
					\note		It is common for plugins to have global prefs modal dialogs that are shown when a "<Your
								Plugin Name> Preferences" menu item is chosen; however, this process is not automated
								for the creator of the plugin as not all plugins necessitate this feature. The Plugin
								class merely offers a convenient place to store said dialog object.
				*/
				virtual ModalDialog* const prefsDialog() const;
				
				/**
					\brief	Sets the global prefs modal dialog for the plugin
					\author	GW
					\date	09/2014
					
					\param	dialog_	New global prefs modal dialog for the plugin
					
					\note	See also the setPrefsMenuItem() method.
					\note	It is common for plugins to have global prefs modal dialogs that are shown when a "<Your
							Plugin Name> Preferences" menu item is chosen; however, this process is not automated for
							the creator of the plugin as not all plugins necessitate this feature. The Plugin class
							merely offers a convenient place to store said dialog object.
				*/
				virtual void setPrefsDialog(const ModalDialog& dialog_);

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the about box menu item for the plugin
						\author		GW
						\date		09/2014
						
						\returns	A pointer to the about box menu item for the plugin, or NULL if none/error
						
						\note		See also the aboutBox() method.
						\note		It is common for plugins to have about box modal dialogs that are shown when an "About
									<Your Plugin Name>" menu item is chosen; however, this process is not automated for the
									creator of the plugin as not all plugins necessitate this feature. The Plugin class
									merely offers a convenient place to store said menu item object.
					*/
					virtual MenuItem* const aboutMenuItem() const;
					
					/**
						\brief	Sets the about box menu item for the plugin
						\author	GW
						\date	09/2014
						
						\param	mi_	New about box menu item for the plugin
						
						\note	See also the setAboutBox() method.
						\note	Typically, one would want to place this menu item in the built-in group AboutMenuGroup.
						\note	It is common for plugins to have about box modal dialogs that are shown when an "About <Your
								Plugin Name>" menu item is chosen; however, this process is not automated for the creator of
								the plugin as not all plugins necessitate this feature. The Plugin class merely offers a
								convenient place to store said menu item object.
					*/
					virtual void setAboutMenuItem(const MenuItem& mi_);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets the global prefs menu item for the plugin
					\author		GW
					\date		09/2014
					
					\returns	A pointer to the prefs menu item for the plugin, or NULL if none/error
					
					\note		See also the prefsDialog() method.
					\note		It is common for plugins to have global prefs modal dialogs that are shown when a "<Your
								Plugin Name> Preferences" menu item is chosen; however, this process is not automated
								for the creator of the plugin as not all plugins necessitate this feature. The Plugin
								class merely offers a convenient place to store said menu item object.
				*/
				virtual MenuItem* const prefsMenuItem() const;
				
				/**
					\brief	Sets the global prefs menu item for the plugin
					\author	GW
					\date	09/2014
					
					\param	mi_	New global prefs menu item for the plugin
					
					\note	See also the setPrefsDialog() method.
					\note	Typically, one would want to place this menu item in the built-in group PrefsMenuGroup.
					\note	It is common for plugins to have global prefs modal dialogs that are shown when a "<Your
							Plugin Name> Preferences" menu item is chosen; however, this process is not automated for
							the creator of the plugin as not all plugins necessitate this feature. The Plugin class
							merely offers a convenient place to store said menu item object.
				*/
				virtual void setPrefsMenuItem(const MenuItem& mi_);


			private:
				#if defined(HDI_CORE_AIP_MODE)
					friend __MainError (::PluginMain) (const char*, const char*, void*);
				#elif defined(HDI_CORE_PSP_MODE)
					#if defined(MAC_ENV)
						friend void (::hdicoreDylibUnloading) ();
					#endif
					
					friend __MainError (::ActionsPluginMain) (const char*, const char*, void*);
					friend void (::AcquirePluginMain) (const int16_t, void*, intptr_t*, int16_t*);
					friend void (::ExportPluginMain) (const int16_t, void*, intptr_t*, int16_t*);
					friend void (::ExtensionPluginMain) (const int16_t, void*, intptr_t*, int16_t*);
					friend void (::FilterPluginMain) (const int16_t, void*, intptr_t*, int16_t*);
					friend void (::FormatPluginMain) (const int16_t, void*, intptr_t*, int16_t*);
					friend void (::MeasurementPluginMain) (const int16_t, void*, intptr_t*, int16_t*);
					friend void (::PickerPluginMain) (const int16_t, void*, intptr_t*, int16_t*);
					friend void (::SelectionPluginMain) (const int16_t, void*, intptr_t*, int16_t*);
				#endif

				/**
					\brief	Describes a variety of errors that can occur at plugin startup
				*/
				enum __StartupError
				{
					__NoStartupError				= 0,
					__CantLockPluginError			= 1000,
					__CantSetPluginNameError		= 1010,
					__BadApplicationVersionError	= 1020,
					__WrongPluginBinaryError		= 1030,
					__WrongInstallLocationError		= 1040,
					__CantLoadPrefsError			= 1050,
					__CantSetupToolsError			= 1060,
					__CantSetupNotifiersError		= 1070,
					__CantSetupAnnotatorsError		= 1080,
					__CantSetupTimersError			= 1090,
					__CustomStartupErrorOffset		= 1000000
				};
				
				/**
					\brief	Describes a variety of errors that can occur at plugin post-startup
				*/
				enum __PostStartupError
				{
					__NoPostStartupError			= 0,
					__CustomPostStartupErrorOffset	= 2000000
				};
				
				/**
					\brief	Describes a variety of errors that can occur at plugin pre-shutdown
				*/
				enum __PreShutdownError
				{
					__NoPreShutdownError			= 0,
					__CustomPreShutdownErrorOffset	= 3000000
				};
				
				/**
					\brief	Describes a variety of errors that can occur at plugin shutdown
				*/
				enum __ShutdownError
				{
					__NoShutdownError			= 0,
					__CantUnlockPluginError		= 4000,
					__CustomShutdownErrorOffset	= 4000000
				};
			
				/**
					\brief	Singleton instance of the plugin
				*/
				static Plugin* __instance;

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Empty Plugin objects cannot be constructed
					\author	GW
					\date	09/2013
				*/
				Plugin();
				
				/**
					\brief	Plugin objects cannot be copied
					\author	GW
					\date	01/2015
				*/
				Plugin(const Plugin&);

				/**
					\brief	Internal use only
					\author	GW
					\date	09/2013
				*/
				Plugin(const void*, const std::string&);
			
				/**
					\brief	Destructs a Plugin object
					\author	GW
					\date	09/2013
				*/
				virtual ~Plugin();
				
				/**
					\brief	Plugin objects cannot be copied
					\author	GW
					\date	09/2013
				*/
				Plugin& operator=(const Plugin& rhs_);

				/**
					\brief	Starts up the plugin by "locking" the plugin ref, setting the name, etc.
					\author	GW
					\date	09/2013
				*/
				__StartupError __startup();
				
				/**
					\brief	Performs actions necessary to plugin startup, but must take place sometime after startup()
					\author	GW
					\date	09/2013
				*/
				__PostStartupError __postStartup();

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief	Performs actions necessary to plugin shutdown, but must take place sometime before shutdown()
						\author	GW
						\date	09/2013
					*/
					__PreShutdownError __preShutdown();
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief	Shutdowns the plugin by releasing suites, "unlocking" the plugin ref, etc.
					\author	GW
					\date	09/2013
				*/
				__ShutdownError __shutdown();
			
				#if defined(HDI_CORE_PSP_MODE)
					/**
						\brief	Show the about box modal dialog previously registered with setAboutBox()
						\author	GW
						\date	01/2018
					*/
					void __aboutBoxShow();
				#endif
				// HDI_CORE_PSP_MODE
		};
	}
}

#if defined(HDI_CORE_AIP_MODE)
	hdi::core::Plugin::Options operator|(
		const hdi::core::Plugin::Options lhs_,
		const hdi::core::Plugin::Options rhs_
	);

	hdi::core::Plugin::Options& operator|=(
		hdi::core::Plugin::Options& lhs__,
		const hdi::core::Plugin::Options rhs_
	);
#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_PLUGIN__
