/**
	\file
	\brief		Header file for describing 2D points
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_POINT__
#define __HDI_CORE_POINT__

#include <memory>
#include <vector>

#include "hdicoreAngle.h"

namespace hdi
{
	#if defined(HDI_CORE_AIP_MODE)
		namespace aip
		{
			class Point;
		}
	#elif defined(HDI_CORE_PSP_MODE)
		namespace psp
		{
			class Point;
		}
	#endif

	namespace core
	{
		class TransformMatrix;
	
		/**
			\brief	Describes a point in the 2-dimensional (x,y) coordinate system, typically in an Illustrator document
					view or a UI window
		*/
		class Point
		{
			public:
				typedef std::vector<Point> PointVector;
			
				/**
					\brief	Position on the x-axis
				*/
				double x;

				/**
					\brief	Position on the y-axis
				*/
				double y;

				/**
					\brief	Default Point constructor; sets x and y to 0.0
					\author	GW
					\date	08/2013
				*/
				Point();

				/**
					\brief	Point constructor, taking x and y arguments
					\author	GW
					\date	08/2013

					\param	x_	Position on the x-axis
					\param	y_	Position on the y-axis
				*/
				Point(const double x_, const double y_);

				/**
					\brief	Point copy constructor
					\author	GW
					\date	08/2013

					\param	pt_		Existing Point object to copy
				*/
				Point(const Point& pt_);

				/**
					\brief	Point destructor
					\author	GW
					\date	08/2013
				*/
				virtual ~Point();

				/**
					\brief		"Named constructor" for a point that is a given length and angle away from the origin
					\author		GW
					\date		08/2013

					\param		length_		Length away from the origin
					\param		angle_		Angle that the point is rotated about the origin and away from the x-axis
					\returns	A Point at the given distance and angle from the origin (0,0)
				*/
				static Point LengthAngle(const double length_, const Angle& angle_);

				/**
					\brief		Sets a Point object from another Point object
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the = operator; Point object to pull values from
					\returns	The target Point object, but with its value updated
				*/
				virtual Point& operator=(const Point& rhs_);
				
				/**
					\brief		Tests whether a given Point object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Point to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same point, false otherwise
				*/
				virtual bool operator==(const Point& rhs_) const;
				
				/**
					\brief		Tests whether a given Point object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Point to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different points, false otherwise
				*/
				virtual bool operator!=(const Point& rhs_) const;

				/**
					\brief		Allows two points to have their x and y values added using the + operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the + operator
					\returns	A copy of the target Point, with all x and y values summed together
				*/
				virtual Point operator+(const Point& rhs_) const;

				/**
					\brief		Allows two points to have their x and y values added and assigned using the += operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the += operator
					\returns	The lefthand side of the += operator, with all x and y values summed together
				*/
				virtual Point& operator+=(const Point& rhs_);

				/**
					\brief		Allows two points to have their x and y values subtracted using the - operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the - operator
					\returns	A copy of the target Point, with rhs_ x and y values subtracted from this x and y
				*/
				virtual Point operator-(const Point& rhs_) const;

				/**
					\brief		Allows two points to have their x and y values subtracted and assigned using the -= operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the -= operator
					\returns	The lefthand side of the -= operator, with rhs_ x and y values subtracted from this x and y
				*/
				virtual Point& operator-=(const Point& rhs_);

				/**
					\brief		Gets the angle of the vector (this, rhs_), if this were moved to the origin
					\author		GW
					\date		01/2018

					\param		rhs_	Righthand side of the ^ operator (called the angle operator in this case)
					\returns	The angle formed by vector (this, rhs_)
				*/
				virtual Angle operator^(const Point& rhs_) const;

				/**
					\brief		Gets the distance between this and rhs_
					\author		GW
					\date		01/2018

					\param		rhs_	Righthand side of the | operator (called the length operator in this case)
					\returns	The distance between this and rhs_
				*/
				virtual double operator|(const Point& rhs_) const;

				/**
					\brief		Gets whether a given point is within a certain tolerance ("close to") the target point
					\author		GW
					\date		01/2018

					\param		compare_	Point to compare with
					\param		tol_		Tolerance between the points; a definition of what constitutes "close"
					\returns	true if the points are within tol_ distance of one another, false otherwise
				*/
				virtual bool closeTo(const Point& compare_, const double tol_) const;

				/**
					\brief		Gets the dot product of two Point objects
					\author		GW
					\date		06/2018

					\param		p2_		Second Point object with which the dot product is calculated
					\returns	The dot product of the target point and the given point (p2_)
				*/
				virtual double dotProduct(const Point& p2_) const;

				/**
					\brief		Offsets a point along the x- and y-axes by given amounts
					\author		GW
					\date		08/2013

					\param		tx_		Amount to offset along the x-axis
					\param		ty_		Amount to offset along the y-axis
					\returns	A copy of the target point, offset by the given amounts

					\note		Negative values go toward 0.0 if the object to be transformed has a positive x and/or y
								value; positive values go away from 0.0
				*/
				virtual Point offset(const double tx_, const double ty_) const;

				/**
					\brief		Rotates the point about a given "origin" point by a certain angle
					\author		GW
					\date		06/2018

					\param		origin_		New "origin" to rotate around
					\param		angle_		Angle to rotate by
					\returns	A copy of the target point, rotated as specified
				*/
				virtual Point rotate(const Point& origin_, const Angle& angle_) const;

				/**
					\brief		"Moves" a point by a given length and angle away from its current position
					\author		GW
					\date		01/2018

					\param		length_		Length away from the current position
					\param		angle_		Angle that the point is moved from the current position
					\returns	A copy of the target point, moved as specified
				*/
				virtual Point move(const double length_, const Angle& angle_) const;

				/**
					\brief		Transforms a point via a 2-dimensional transformation matrix
					\author		GW
					\date		06/2018

					\param		m_	Transformation matrix to apply to the point
					\returns	A copy of the target point, transformed as specified by the matrix
				*/
				virtual Point transform(const TransformMatrix& m_) const;

				/**
					\brief		Checks if the point is within the provided polygon
					\author		GW
					\date		08/2013

					\param		points_[]	Points that describe the polygon
					\returns	true if the point lies within the provided polygon, false otherwise
				*/
				virtual bool insidePolygon(const PointVector& points_) const;


			private:
				#if defined(HDI_CORE_AIP_MODE)
					friend aip::Point* __accessImpl(const Point&);
					friend Point __accessCtor(const aip::Point&);
					
					/**
						\brief	Private implementation object
					*/
					aip::Point* __impl;
				
					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					Point(const aip::Point&);
				#elif defined(HDI_CORE_PSP_MODE)
					friend psp::Point* __accessImpl(const Point&);
					friend Point __accessCtor(const psp::Point&);
					
					/**
						\brief	Private implementation object
					*/
					psp::Point* __impl;
				
					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					Point(const psp::Point&);
				#endif
		};
		
		typedef std::unique_ptr<Point> PointUP;
		typedef std::shared_ptr<Point> PointSP;
		typedef std::weak_ptr<Point> PointWP;
		
		#if defined(HDI_CORE_AIP_MODE)
			extern aip::Point* __accessImpl(const Point&);
			extern Point __accessCtor(const aip::Point&);
		#elif defined(HDI_CORE_PSP_MODE)
			extern psp::Point* __accessImpl(const Point&);
			extern Point __accessCtor(const psp::Point&);
		#endif
	}
}

#endif
// __HDI_CORE_POINT__
