/**
	\file
	\brief		Header file for plugin user interface popup menu widgets
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_POPUP_MENU__
#define __HDI_CORE_POPUP_MENU__

#include <vector>

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		class PopupEntry;
		class Font;

		/**
			\brief	Popup menu widget, allowing for a single preset entry to be chosen at a time
		*/
		class PopupMenu : public Widget
		{
			public:
				typedef std::vector<PopupEntry> EntryVector;
			
				/**
					\brief	Constructs an empty PopupMenu object
					\author	GW
					\date	09/2013
					
					\note	To test if a PopupMenu object is empty, call isEmpty() on it
					\note	Empty PopupMenu objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other PopupMenu object via the overloaded assignment operator. Empty
							PopupMenu objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				PopupMenu();

				/**
					\brief	Constructs a new PopupMenu object from an existing PopupMenu object (copy constructor)
					\author	GW
					\date	09/2013

					\param	popup_	Existing PopupMenu object
				*/
				PopupMenu(const PopupMenu& popup_);

				/**
					\brief	Constructs an empty PopupMenu widget at a given location of a given width
					\author	GW
					\date	09/2013

					\param	loc_		Position of the popup, in 1x resolution coordinates
					\param	width_		Width of the popup, at 1x resolution
				*/
				PopupMenu(const Point& loc_, const double width_);

				/**
					\brief	Constructs a PopupMenu widget with a list of entries and an initial index selection
					\author	GW
					\date	09/2013

					\param	loc_			Position of the popup, in 1x resolution coordinates
					\param	width_			Width of the popup, at 1x resolution
					\param	entries_		Vector of string entries, whose order in the popup will match the vector
					\param	initialIndex_	Initial index to select (must exist in entries_, and must not be a separator)
				*/
				PopupMenu(
					const Point& loc_,
					const double width_,
					const EntryVector& entries_,
					const int32_t initialIndex_
				);

				/**
					\brief	Constructs a PopupMenu widget with a list of entries and an initial ID selection
					\author	GW
					\date	09/2013

					\param	loc_		Position of the popup, in 1x resolution coordinates
					\param	width_		Width of the popup, at 1x resolution
					\param	entries_	Vector of string entries, whose order in the popup will match that in the vector
					\param	initialID_	Initial ID to select, as UTF-8 (must exist in entries_, and must not be a
										separator)
				*/
				PopupMenu(
					const Point& loc_,
					const double width_,
					const EntryVector& entries_,
					const std::string& initialID_
				);

				/**
					\brief		Static named constructor for creating a font menu with a preselected font object
					\author		GW
					\date		09/2013

					\param		loc_	Position of the popup, in 1x resolution coordinates
					\param		width_	Width of the popup, at 1x resolution 
					\param		font_	Font face to select by default
					\returns	A PopupMenu object, filled with all fonts available to the app

					\note		The PopupEntry objects, with which the PopupMenu is filled, themselves contain pointers
								to Font objects
				*/
				static PopupMenu FontMenu(const Point& loc_, const double width_, const Font& font_);
				
				/**
					\brief	PopupMenu destructor
					\author	GW
					\date	09/2013
				*/
				virtual ~PopupMenu();

				/**
					\brief		Allows one PopupMenu object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target PopupMenu object, but with its value updated to match that of rhs_
				*/
				virtual PopupMenu& operator=(const PopupMenu& rhs_);
				
				/**
					\brief		Convenience method to clone a PopupMenu object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new PopupMenu object
					
					\note		If you subclass PopupMenu, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your PopupMenu subclass, you will
								experience "object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned PopupMenu object.
				*/
				virtual PopupMenu* clone() const;
				
				/**
					\brief		Convenience method to duplicate a PopupMenu object, creating a new and identical UI
								element to the target (but not belonging to the same containing window)
					\author		GW
					\date		11/2013
					
					\returns	A pointer to the new PopupMenu object (and new UI element)
					
					\note		The new popup menu will have copies of all entries present in the target, but be aware
								that copies of the user data in each entry cannot be made (because they are simply void*
								variables). As such, the user data in the copies will simply be set to NULL by this
								method. This does not apply to PopupMenu objects previously created with the FontMenu()
								named constructor.
								
					\warning	The caller must manage the memory for the returned PopupMenu object.
				*/
				virtual PopupMenu* duplicate() const;

				/**
					\brief	In the case of a mixed popup state, this will create and select a blank entry
					\author	GW
					\date	09/2013
				*/
				virtual void selectUnknownEntry();

				/**
					\brief	In the case of a (prior) mixed popup state, this will deselect and destroy the blank entry
					\author	GW
					\date	09/2013
				*/
				virtual void deselectUnknownEntry();

				/**
					\brief		Gets the index of the currently selected entry
					\author		GW
					\date		09/2013

					\returns	The index number of the currently selected entry, or -1 if no entry is selected
				*/
				virtual int32_t currentIndex() const;
				
				/**
					\brief		Sets the currently selected entry by index
					\author		GW
					\date		09/2013

					\param		index_	New index to select
					\returns	true if the selection occurred successfully, or false if e.g. out of range
				*/
				virtual bool setCurrentIndex(const int32_t index_);
				
				/**
					\brief		Gets the ID of the currently selected entry
					\author		GW
					\date		09/2013

					\returns	UTF-8 ID string of the currently selected entry, or "" if no entry is currently selected
				*/
				virtual std::string currentID() const;
				
				/**
					\brief		Sets the currently selected entry by ID
					\author		GW
					\date		09/2013

					\param		id_		ID of new value to select in the popup menu, as UTF-8
					\returns	true if the entry with the given ID was selected, or false if invalid
				*/
				virtual bool setCurrentID(const std::string& id_);
				
				/**
					\brief		Gets the string value of the currently selected entry
					\author		GW
					\date		09/2013

					\returns	Value string of the popup menu, as UTF-8
				*/
				virtual std::string currentValue() const;
				
				/**
					\brief		Sets the currently selected entry by string value
					\author		GW
					\date		09/2013

					\param		value_	New value to set for the popup menu (if it exists as an entry), as UTF-8
					\returns	true if the value was set, false otherwise
				*/
				virtual bool setCurrentValue(const std::string& value_);

				/**
					\brief		Gets the current entry object
					\author		GW
					\date		09/2013

					\returns	The PopupEntry object for the currently selected item
				*/
				virtual std::unique_ptr<PopupEntry> currentEntry() const;
				
				/**
					\brief		Gets the user data for the current entry
					\author		GW
					\date		09/2013

					\param		data__	Return-by-reference for the current user data; ignore if false is returned
					\returns	true if the method succeeds in acquiring the contained entry data, false otherwise
				*/
				virtual bool currentUserData(void*& data__) const;

				/**
					\brief		Gets the entries vector
					\author		GW
					\date		09/2013

					\returns	The entries vector for the target widget, containing all currently known entries
					
					\note		Modifying any value in an entry will not cause the widget to update automatically; call
								update() to update the UI for changes made to entry data.
				*/
				virtual const std::vector< std::shared_ptr<PopupEntry> >* const entries() const;
				
				/**
					\brief	Allows the entries in an popup menu to be changed out entirely
					\author	GW
					\date	09/2013

					\param	entries_	New set of entries that the widget will contain
				*/
				virtual void setEntries(const EntryVector& entries_);
				
				/**
					\brief		Gets whether the entries vector already contains an entry, based on its ID
					\author		GW
					\date		09/2013

					\param		id_		ID of the entry to search for, as UTF-8
					\returns	true if the widget already contains an entry with the given ID, false otherwise
				*/
				virtual bool hasEntry(const std::string& id_) const;

				/**
					\brief		Gets the entry with the given ID, provided it exists within the widget
					\author		GW
					\date		02/2023

					\param		id_		ID of the entry to search for
					\returns	The entry with the given ID, or NULL if not found
				*/
				virtual std::shared_ptr<PopupEntry> entryWithID(const std::string& id_) const;
				
				/**
					\brief		Pushes an entry onto the list (i.e. at the end)
					\author		GW
					\date		09/2013

					\param		entry_	New entry object to insert into the list of entries
					\returns	true if the entry is pushed successfully, false otherwise

					\note		If the entry is already present in the list (based on its ID), it will not be inserted.
				*/
				virtual bool pushEntry(const PopupEntry& entry_);

				/**
					\brief		Pops an entry off the list (i.e. from the end)
					\author		GW
					\date		09/2013

					\param		entry__		Return-by-reference for the popped entry, so the caller can inspect which
											one was removed
					\returns	true if an entry was popped and acquired, false otherwise
					
					\warning	If you created the target PopupMenu object with the FontMenu named constructor, then the
								PopupEntry object will contain a Font object pointer for which the caller will become
								responsible (i.e. the caller must cast the user data void* to a Font* and delete the
								Font object when the PopupEntry object is no longer needed).
				*/
				virtual bool popEntry(PopupEntry& entry__);

				/**
					\brief		Inserts an entry into the list of entries at a given position
					\author		GW
					\date		09/2013

					\param		id_		Existing entry immediately before the position at which the new entry will be
										inserted, as UTF-8
					\param		entry_	New entry object to insert into the list of entries
					\returns	true if the entry is inserted successfully, false otherwise

					\note		If the entry for id_ doesn't exist in the list, the new entry will not be inserted.
					\note		If the passed entry is already present in the list (based on its ID), it will not be
								inserted.
				*/
				virtual bool insertEntry(const std::string& id_, const PopupEntry& entry_);

				/**
					\brief		Removes a specific entry from the list of entries
					\author		GW
					\date		09/2013

					\param		id_			ID of the entry to be removed, as UTF-8
					\param		entry__		Return-by-reference for the removed entry, so the caller can inspect which
											one was removed
					\returns	true if an entry was removed and acquired, false otherwise
					
					\warning	If you created the target PopupMenu object with the FontMenu named constructor, then the
								PopupEntry object will contain a Font object pointer for which the caller will become
								responsible (i.e. the caller must cast the user data void* to a Font* and delete the
								Font object when the PopupEntry object is no longer needed).
				*/
				virtual bool removeEntry(const std::string& id_, PopupEntry& entry__);

				/**
					\brief	Clears all entries contained by the widget
					\author	GW
					\date	09/2013
				*/
				virtual void clearEntries();

				/**
					\brief	Simulates the popup menu having its value changed
					\author	GW
					\date	09/2013
				*/
				virtual void valueChanged();
				
				/**
					\brief		Gets the value changed callback for the popup menu
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered value changed callback
				*/
				virtual Callback* const valueChangedCallback() const;
				
				/**
					\brief	Sets the value changed callback
					\author	GW
					\date	09/2013

					\param	callback_	New callback for when the popup menu's value has changed
				*/
				virtual void setValueChangedCallback(const Callback& callback_);
				
				/**
					\brief	Force the popup menu to update its entries
					\author	GW
					\date	12/2013
				*/
				virtual void update();
		};
		
		typedef std::unique_ptr<PopupMenu> PopupMenuUP;
		typedef std::shared_ptr<PopupMenu> PopupMenuSP;
		typedef std::weak_ptr<PopupMenu> PopupMenuWP;
	}
}

#endif
// __HDI_CORE_POPUP_MENU__
