/**
	\file
	\brief		Header file for general preference data storage
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_PREF_DATA__
#define __HDI_CORE_PREF_DATA__

#include "hdicoreTypes.h"
	
namespace hdi
{
	namespace plat
	{
		class PersistentData;
	}

	namespace core
	{
		/**
			\brief	Base class for persistent data objects, stored via JSON
		*/
		class PrefData
		{
			public:
				/**
					\brief	Describes the type of the data the PrefData object contains; useful for determining which
							subclass to cast a PrefData object to.
				*/
				enum Type
				{
					UnknownType			=   0,
					
					ArrayType			=  10,
					
					#if defined(HDI_CORE_AIP_MODE)
						ArtboardPointType	=  20,
						ArtboardRectType	=  30,
						ArtColorType		=  40,
					#endif
					
					BoolType			=  50,
					ColorType			=  60,
					
					#if defined(HDI_CORE_AIP_MODE)
						DashStyleType		=  70,
						DashStyleMapType	=  80,
					#endif
					
					DictionaryType		=  90,
					
					#if defined(HDI_CORE_AIP_MODE)
						FillStyleType		= 100,
						FillStyleMapType	= 110,
					#endif
					
					FontType			= 120,
					NumberType			= 130,
					
					#if defined(HDI_CORE_AIP_MODE)
						PathType			= 140,	// Reserved for future use
						PathStyleType		= 150,
						PathStyleMapType	= 160,
					#endif
					
					PointType			= 170,
					RectType			= 180,
					SizeType			= 190,
					StringType			= 200,
					
					#if defined(HDI_CORE_AIP_MODE)
						StrokeStyleType		= 210,
						StrokeStyleMapType	= 220,
						TransformMatrixType	= 230,
					#endif
					
					CustomTypeOffset	= 10000
				};
			
				/**
					\brief	Constructs an empty PrefData object
					\author	GW
					\date	09/2013
					
					\note	To test if a PrefData object is empty, call isEmpty() on it
					\note	Empty PrefData objects do not relate to any actual preference data; they are designed to be
							"receivers" of some other PrefData object via the overloaded assignment operator. Empty
							PrefData objects are useless until such time (though it is safe to call any of their methods)
				*/
				PrefData();
				
				/**
					\brief		Parses the given JSON string to a best-fit representation of PrefData subclass objects
								in a tree.
					\author		GW
					\date		11/2013
					
					\param		jsonStr_	JSON-encoded string, as UTF-8, to be parsed into a tree structure of JSON
											nodes
					\returns	The root node of the PrefData subclass object tree, or NULL if none/error
				*/
				static std::unique_ptr<PrefData> parse(const std::string& jsonStr_);
			
				/**
					\brief	Destructs a PrefData object
					\author	GW
					\date	09/2013
				*/
				virtual ~PrefData();

				/**
					\brief		Allows one PrefData object to be assigned from another
					\author		GW
					\date		09/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target PrefData object, but with its value updated to match that of rhs_
					
					\note		This method will fail in the case that the target and rhs_ are not of the same type (see
								the dataType() method).
				*/
				virtual PrefData& operator=(const PrefData& rhs_);
				
				/**
					\brief		Convenience method to clone a PrefData object, taking its subclass type into account for
								proper memory management
					\author		GW
					\date		10/2013

					\returns	A pointer to the new PrefData object
					
					\warning	The caller must manage the memory for the returned PrefData object.
				*/
				virtual PrefData* clone() const = 0;

				/**
					\brief		Gets whether the target PrefData object is empty (constructed with the default ctor)
					\author		GW
					\date		09/2013
					
					\returns	true if the target PrefData object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Tests whether a given PrefData object is the same as another
					\author		GW
					\date		09/2013

					\param		rhs_	PrefData to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same data object, false otherwise
					
					\note		This overloaded equality operator does not determine whether the stored values of two
								PrefData objects are the same; rather, it determines whether the data objects themselves
								have the same underlying container.
				*/
				virtual bool operator==(const PrefData& rhs_) const;

				/**
					\brief		Tests whether a given PrefData object is not the same as another
					\author		GW
					\date		09/2013

					\param		rhs_	PrefData to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different data objects, false otherwise

					\note		This overloaded operator does not determine whether the stored values of two PrefData
								objects are different; rather, it determines whether the data objects themselves have
								different underlying containers.
				*/
				virtual bool operator!=(const PrefData& rhs_) const;

				/**
					\brief		Gets the type of persistent data
					\author		GW
					\date		09/2013

					\returns	The type of the persistent data

					\note		This is useful for knowing how to cast a PrefData object to its proper subclass
				*/
				virtual Type dataType() const;

				/**
					\brief		Gets the persistent data object's key
					\author		GW
					\date		09/2013

					\returns	The key/name of the persistent data object, as UTF-8

					\note		This method will return "" in the case of a persistent data object contained within an
								ArrayPref object.
				*/
				virtual std::string key() const;

				/**
					\brief	Sets the persistent data object's key
					\author	GW
					\date	09/2013

					\param	key_	New key/name for the object, as UTF-8

					\note	This value is ignored in the case of a persistent data object contained within an ArrayPref
							object.
				*/
				virtual void setKey(const std::string& key_);

				/**
					\brief		Recursively serializes the target object as a JSON data (sub)tree, returning the
								appropriate JSON string
					\author		GW
					\date		09/2013
					
					\returns	The data tree serialized and formatted as a string, as UTF-8

					\note		If, at any time, the serializer cannot perform its job, "" (empty string) will be
								returned.
				*/
				virtual std::string stringify() const;


			protected:
				friend plat::PersistentData* __accessImpl(const PrefData&);
				friend PrefData* __accessCtor(plat::PersistentData*&);
				
				/**
					\brief	Private implementation data
				*/
				void* _data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	09/2013
				*/
				void* __impl() const;

				/**
					\brief	Internal use only
					\author	GW
					\date	09/2013
				*/
				PrefData(plat::PersistentData*&);

				/**
					\brief	Constructs a new PrefData object from an existing PrefData object (copy constructor)
					\author	GW
					\date	09/2013

					\param	data_	Existing PrefData object
				*/
				PrefData(const PrefData& data_);
		};
		
		typedef std::unique_ptr<PrefData> PrefDataUP;
		typedef std::shared_ptr<PrefData> PrefDataSP;
		typedef std::weak_ptr<PrefData> PrefDataWP;
		
		extern plat::PersistentData* __accessImpl(const PrefData&);
		extern PrefData* __accessCtor(plat::PersistentData*&);
	}
}

#endif
// __HDI_CORE_PREF_DATA__
