/**
	\file
	\brief		Header file for grouping plugin radio button widgets
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_RADIO_BUTTON_GROUP__
#define __HDI_CORE_RADIO_BUTTON_GROUP__

#include <memory>
#include <vector>

namespace hdi
{
	namespace core
	{
		class RadioButton;

		/**
			\brief	Allows for logically grouping radio buttons together, such that one can be chosen and others
					automatically unchosen, etc.
		*/
		class RadioButtonGroup
		{
			public:
				typedef std::vector< std::shared_ptr<RadioButton> > RadioButtonVector;
			
				/**
					\brief	Constructs a new radio button group
					\author	GW
					\date	09/2013
				*/
				RadioButtonGroup();
				
				/**
					\brief	Constructs a new RadioButtonGroup object from an existing RadioButtonGroup object (copy
							constructor)
					\author	GW
					\date	11/2013

					\param	group_	Existing RadioButtonGroup object
				*/
				RadioButtonGroup(const RadioButtonGroup& group_);
				
				/**
					\brief	RadioButtonGroup destructor
					\author	GW
					\date	09/2013
				*/
				virtual ~RadioButtonGroup();

				/**
					\brief		Allows one RadioButtonGroup object to be assigned from another
					\author		GW
					\date		09/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target RadioButtonGroup object, but with its value updated to match that of rhs_
				*/
				virtual RadioButtonGroup& operator=(const RadioButtonGroup& rhs_);

				/**
					\brief		Tests whether a given RadioButtonGroup object is the same as another
					\author		GW
					\date		09/2013

					\param		rhs_	RadioButtonGroup to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same radio button group, false otherwise
				*/
				virtual bool operator==(const RadioButtonGroup& rhs_) const;

				/**
					\brief		Tests whether a given RadioButtonGroup object is not the same as another
					\author		GW
					\date		09/2013

					\param		rhs_	RadioButtonGroup to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different radio button groups, false otherwise
				*/
				virtual bool operator!=(const RadioButtonGroup& rhs_) const;
				
				/**
					\brief	Adds a button to the group
					\author	GW
					\date	09/2013

					\param	rb_		Radio button object to add

					\note	If the button is already in a group, this method will bail early
				*/
				virtual void add(const RadioButton& rb_);
				
				/**
					\brief	Removes a button from the group
					\author	GW
					\date	09/2013

					\param	rb_		Radio button to remove

					\note	If the button isn't in the group, this method will bail early
				*/
				virtual void remove(const RadioButton& rb_);

				/**
					\brief		Checks for the existence of a button in the group
					\author		GW
					\date		09/2013

					\param		rb_		Radio button in question
					\returns	true if the button is contained in the group, false otherwise
				*/
				virtual bool hasButton(const RadioButton& rb_);
				
				/**
					\brief		Gets a list of the contained buttons
					\author		GW
					\date		09/2013

					\returns	A vector containing all RadioButton objects in the target group
				*/
				virtual RadioButtonVector buttons() const;
				
				/**
					\brief		Gets the default radio button for the group
					\details	A radio button group can have a default button, which will be automatically chosen in
								the case that all other buttons have become unchosen (i.e. the group itself can never
								have all buttons unchosen). A default button is optional (allowing all buttons in the
								group to be unchosen).
					\author		GW
					\date		09/2013

					\returns	The default RadioButton object for the group, or NULL if none
				*/
				virtual RadioButton* const defaultButton() const;
				
				/**
					\brief	Sets the default radio button for the group
					\author	GW
					\date	09/2013

					\param	radio_	The radio button to act as the default for the group

					\pre	This method must be called after all radios have been added to the group.

					\note	For more info about the default radio button functionality, see the defaultButton() method.
				*/
				virtual void setDefaultButton(const RadioButton& radio_);
				
				/**
					\brief	Clears the default radio button for the group (i.e. unsets it)
					\author	GW
					\date	06/2015
				*/
				virtual void clearDefaultButton();


			private:
				/**
					\brief	Private implementation data
				*/
				void* __data;
		};
		
		typedef std::unique_ptr<RadioButtonGroup> RadioButtonGroupUP;
		typedef std::shared_ptr<RadioButtonGroup> RadioButtonGroupSP;
		typedef std::weak_ptr<RadioButtonGroup> RadioButtonGroupWP;
	}
}

#endif
// __HDI_CORE_RADIO_BUTTON_GROUP__
