/**
	\file
	\brief		Header file for general Illustrator raster art manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_RASTER_ART__
#define __HDI_CORE_RASTER_ART__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreArtboardRect.h"
#include "hdicoreTransformMatrix.h"

namespace hdi
{
	namespace core
	{
		class Art;
	
		/**
			\brief	Allows for raster art manipulation, metadata access, etc.
		*/
		class RasterArt
		{
			public:
				/**
					\brief	Describes raster art options
				*/
				enum Flags
				{
					UnknownFlag			= 0,
					
					// Used only for 1-bit gray images. When set, the 1-bit data is to be used as a mask. In this case
					// the path style of the image indicates the color that the bits of the image with a value of 1 are
					// painted.
					MaskImageFlag		= 1 << 0,
					
					// Used only for 1-bit gray images. Reverses the sense of the component values so that for a gray
					// image 0=white and 1=black and for an image mask the 0 value pixels are painted.
					InvertBitsFlag		= 1 << 1,

					// Used only for 8-bit spot color rasters, or 16-bit if there is an extra alpha channel. When set,
					// the gray component values are interpreted in the subtractive color spaces.
					GraySubtractiveFlag	= 1 << 2
				};
				
				/**
					\brief	Describes raster art color spaces
				*/
				enum ColorSpace
				{
					UnknownColorSpace			= 0,
					
					// Each pixel value for a gray color space has a single component describing a grayscale value. The
					// gray color space is additive so the minimum value represents black and the maximum represents
					// white.
					GrayColorSpace				= 1,
					
					// Each pixel value for a RGB color space has three components which represent red, green and blue
					// respectively. The RGB color space is additive.
					RGBColorSpace				= 2,
					
					// Each pixel value for a CMYK color space has four components which represent cyan, magenta, yellow
					// and black respectively. The CMYK color space is subtractive.
					CMYKColorSpace				= 3,
					
					// Not valid as an image type; can occur only in placed linked files
					LabColorSpace				= 4,
					
					// Each pixel value for a separation color space has a single component describing a tint value. The
					// separation color space is subtractive so the minimum value represents white and the maximum
					// represents black.
					SeparationColorSpace		= 5,
					
					// Each pixel value for an NChannel color space has of a variable number of components which
					// represent individual channels in the NChannel color space. The process components of the color
					// space could be either additive or subtractive. The spot components of the color space are
					// subtractive.
					NChannelColorSpace			= 6,
					
					// Each pixel value for an indexed color space has a single component describing an index value into
					// a color lookup table. The number of components in the color lookup table depends on the base
					// color space of the indexed color space.
					IndexedColorSpace			= 7,
					
					// Flag indicating that the color model has an alpha channel. The alpha component appears after the
					// color components.
					ColorSpaceHasAlpha			= 0x10,
					
					// A gray color space with an alpha channel
					AlphaGrayColorSpace			= (GrayColorSpace | ColorSpaceHasAlpha),

					// An RGB color space with an alpha channel
					AlphaRGBColorSpace			= (RGBColorSpace  | ColorSpaceHasAlpha),

					// A CMYK color space with an alpha channel
					AlphaCMYKColorSpace			= (CMYKColorSpace | ColorSpaceHasAlpha),

					// A Lab color space with an alpha channel
					AlphaLabColorSpace			= (LabColorSpace | ColorSpaceHasAlpha),

					// A separation color space with an alpha channel
					AlphaSeparationColorSpace	= (SeparationColorSpace | ColorSpaceHasAlpha),

					// An NChannel color space with an alpha channel
					AlphaNChannelColorSpace		= (NChannelColorSpace | ColorSpaceHasAlpha),

					// An indexed color space with an alpha channel
					AlphaIndexedColorSpace		= (IndexedColorSpace | ColorSpaceHasAlpha),
	
					InvalidColorSpace			= 0xFF
				};
			
				/**
					\brief	Describes more detailed information about raster art
				*/
				struct Record
				{
					/**
						\brief	Bitwise OR of flags defining additional options for the image
					*/
					Flags flags;
					
					/**
						\brief	The width and height of the image data in pixels
					*/
					Size dimensions;
					
					/**
						\brief	The number of bits used to define each pixel in the pixel map
						
						\note	Valid values depend on the color space: 1 for black and white (GrayColorSpace), 8 for
								grayscale (GrayColorSpace), 24 for RGB (8 bits for each color), 48 for RGB (16 bits for
								each color), or 32 for CMYK (8 bits for each color)
					*/
					int16_t bitsPerPx;
					
					/**
						\brief	The number of bytes in a single row of pixel data
						
						\note	Ignored for the setInfo() method.
						\note	1-bit data is packed so that there are 8 pixels per byte.
						\note	If bitsPerPx is 1 (for a black and white bitmap), the value should be
								dimensions.width / 8 + 1. For all other bitsPerPx values, the value of this member
								should be dimensions.width * (bitsPerPx / 8).
					*/
					int32_t rowBytes;
					
					/**
						\brief	The color model of the pixel data, which can include an alpha channel
					*/
					ColorSpace colorSpace;
					
					/**
						\brief	For an image that has been converted from one color space to another, the original color
								space. If the image has not been converted, the value should match colorSpace.
					*/
					ColorSpace originalColorSpace;
				
					/**
						\brief	Constructs a new Record object with default values
						\author	GW
						\date	08/2014
					*/
					Record();
					
					/**
						\brief	Constructs a new Record object from an existing one
						\author	GW
						\date	08/2014
						
						\param	rec_	Existing Record object to copy values from
					*/
					Record(const Record& rec_);
					
					/**
						\brief	Constructs a new Record object from the given values
						\author	GW
						\date	08/2014
						
						\param	flags_			Options for the raster image
						\param	dim_			Dimensions of the raster image
						\param	bitsPerPx_		Bits per pixel of the raster image
						\param	rowBytes_		Bytes per pixel row of the raster image
						\param	colorSpace_		Current color space of the raster image
						\param	origColorSpace_	Original color space of the raster image
					*/
					Record(
						const Flags flags_,
						const Size& dim_,
						const int16_t bitsPerPx_,
						const int32_t rowBytes_,
						const ColorSpace colorSpace_,
						const ColorSpace origColorSpace_
					);
					
					/**
						\brief	Destructs a Record object
						\author	GW
						\date	08/2014
					*/
					virtual ~Record();
					
					/**
						\brief		Record assignment operator
						\author		GW
						\date		08/2014
						
						\param		rhs_	Existing Record object to copy values from (righthand side of the
											assignment operator)
						\returns	The target Record object, but with its values updated
					*/
					Record& operator=(const Record& rhs_);
					
					/**
						\brief		Compares two Record objects for equality
						\author		GW
						\date		08/2014
						
						\param		rhs_	Existing Record object to compare against the target
						\returns	true if the members of the two Record objects are the same, false otherwise
					*/
					bool operator==(const Record& rhs_);
					
					/**
						\brief		Compares two Record objects for inequality
						\author		GW
						\date		08/2014
						
						\param		rhs_	Existing Record object to compare against the target
						\returns	true if the members of the two Record objects differ, false otherwise
					*/
					bool operator!=(const Record& rhs_);
				};
				
				/**
					\brief	Destructs a RasterArt object
					\author	GW
					\date	08/2014
				*/
				~RasterArt();
				
				/**
					\brief		Retrieves the raster record associated with a raster art object
					\author		GW
					\date		08/2014
					
					\returns	The raster record for the art
				*/
				Record info() const;
				
				/**
					\brief	Associates a raster record with a raster art object
					\author	GW
					\date	08/2014
					
					\param	rec_	The new raster record
				*/
				void setInfo(const Record& rec_);
				
				/**
					\brief		Gets the file path for the raster art
					\author		GW
					\date		08/2014
					
					\returns	The file path as UTF-8, or "" for error
				*/
				std::string filePath() const;
				
				/**
					\brief	Sets the file path for the raster art
					\author	GW
					\date	08/2014
					
					\param	path_	New file path, as UTF-8
				*/
				void setFilePath(const std::string& path_);
				
				/**
					\brief		Retrieves the transformation matrix for a raster art object. This is the concatenation
								of all transforms applied to the image after it was created. It typically contains at
								least the initial translation and a scale.
					\author		GW
					\date		08/2014
					
					\returns	The transformation matrix for the placed art
				*/
				core::TransformMatrix matrix() const;
				
				/**
					\brief	Sets the transformation matrix for a raster art object
					\author	GW
					\date	08/2014
					
					\param	matrix_	New transformation matrix
					
					\note	All raster data is internally kept at 72 ppi and uses the object's scale to set a resolution.
				*/
				void setMatrix(const core::TransformMatrix& matrix_);
				
				/**
					\brief		Retrieves the raster image bounding box of a raster art object
					\author		GW
					\date		08/2014
					
					\returns	The bounding rect
				*/
				core::ArtboardRect bounds() const;
				
				/**
					\brief	Sets the raster image bounding box of a raster art object
					\author	GW
					\date	08/2014
					
					\param	b_	The new bounding box
				*/
				void setBounds(const core::ArtboardRect& b_);
				
				/**
					\brief		Retrieves the overprint status for a raster object
					\author		GW
					\date		08/2014
					
					\returns	true if overprint is enabled, false if it is disabled
				*/
				bool overprint() const;
				
				/**
					\brief	Enables or disables overprinting for a raster object
					\author	GW
					\date	08/2014
					
					\param	op_		True to enable overprint, false to disable it
				*/
				void setOverprint(const bool op_);

				/**
					\brief		Gets whether a raster object is a colorized grayscale or bitmap image
					\author		GW
					\date		08/2014
					
					\param		additive__	Return-by-reference for whether the
				raster should be interpreted as additive
					\returns	true if the raster is a colorized grayscale or bitmap image
				*/
				bool colorized(bool& additive__) const;

				/**
					\brief	Clears the colorization on a colorized grayscale or bitmap image
					\author	GW
					\date	08/2014
				*/
				void clearColorization();
				

			private:
				// Only Art can construct a new RasterArt object
				friend class Art;

				/**
					\brief	Art to which this mesh belongs
				*/
				void* __data;
				
				/**
					\brief	Empty RasterArt objects cannot be constructed
					\author	GW
					\date	08/2014
				*/
				RasterArt();

				/**
					\brief	Constructs a new RasterArt object from a given Art object
					\author	GW
					\date	08/2014

					\param	art_	Pointer for the existing art we're wrapping around
				*/
				RasterArt(void* const art_);
			
				/**
					\brief	RasterArt objects cannot be copied
					\author	GW
					\date	08/2014
				*/
				RasterArt(const RasterArt&);
			
				/**
					\brief	RasterArt objects cannot be copied
					\author	GW
					\date	08/2013
				*/
				RasterArt& operator=(const RasterArt&);
		};
	}
}

hdi::core::RasterArt::Flags operator|(
	const hdi::core::RasterArt::Flags lhs_,
	const hdi::core::RasterArt::Flags rhs_
);

hdi::core::RasterArt::Flags& operator|=(
	hdi::core::RasterArt::Flags& lhs__,
	const hdi::core::RasterArt::Flags rhs_
);

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_RASTER_ART__
