/**
	\file
	\brief		Header file for plugin registration storage
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_REGISTRATION__
#define __HDI_CORE_REGISTRATION__

#include "hdicoreTypes.h"

namespace hdi
{
	namespace plat
	{
		class Registration;
	}

	namespace core
	{
		class DictionaryPref;
	
		namespace crypt
		{
			namespace aes
			{
				class Key;
				class InitVector;
			}
		}
	
		/**
			\brief	Aids in user registration by reading/writing and de/encrypting the JSON data in a given file
		*/
		class Registration
		{
			public:
				/**
					\brief	Constructs an empty Registration object
					\author	GW
					\date	09/2013
					
					\note	To test if a Registration object is empty, call isEmpty() on it
					\note	Empty Registration objects do not relate to any actual registration file; they are designed
							to be "receivers" of some other Registration object via the overloaded assignment operator.
							Empty Registration objects are useless until such time (though it is safe to call any of
							their methods).
				*/
				Registration();
			
				/**
					\brief	Constructs a new Registration object from an existing Registration object (copy constructor)
					\author	GW
					\date	09/2013

					\param	reg_	Existing Registration object
				*/
				Registration(const Registration& reg_);

				/**
					\brief	Destructs a Registration object
					\author	GW
					\date	09/2013
				*/
				virtual ~Registration();
				
				/**
					\brief		Assigns one Registration object to another
					\author		GW
					\date		09/2013

					\param		rhs_	Existing Registration object to copy values from
					\returns	The target Registration object, but with its values updated to match that of the rhs_
								argument
				*/
				virtual Registration& operator=(const Registration& rhs_);

				/**
					\brief		Gets whether the target Registration object is empty (constructed with the default ctor)
					\author		GW
					\date		09/2013
					
					\returns	true if the target Registration object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Convenience method to create a Registration object from (encrypted) JSON data in a file
					\author		GW
					\date		09/2013

					\param		path_		Path to the registration file, as UTF-8
					\param		key_		AES crypt key for the file
					\param		iv_			AES initialization vector for the file
					\param		reg__		Return-by-reference for the Registration object for the file located at
											path_ and encrypted with key_ (and iv_)
					\param		create_		Whether to create the file if it cannot be decrypted (or doesn't exist yet)
					\returns	true if the file could be created (if necessary) and if the Registration object was
								acquired, false otherwise

					\note		The file need not exist before calling this method (so long as create_ is true).
				*/
				static bool load(
					const std::string& path_,
					const crypt::aes::Key& key_,
					const crypt::aes::InitVector& iv_,
					const bool create_,
					Registration& reg__
				);

				/**
					\brief		Writes (encrypted) JSON registration data to a file
					\author		GW
					\date		09/2013

					\param		path_	Path to the registration file, as UTF-8
					\param		key_	AES crypt key for the file
					\param		iv_		AES initialization vector for the file
					\returns	true if the save operation was successful, false otherwise
				*/
				virtual bool save(
					const std::string& path_,
					const crypt::aes::Key& key_,
					const crypt::aes::InitVector& iv_
				) const;
			
				/**
					\brief		Reloads the registration data from disk (as it is kept in memory at runtime, this might
								be necessary if another process or plugin manipulates the file)
					\author		GW
					\date		01/2016
					
					\param		path_	Path to the registration file, as UTF-8
					\param		key_	AES crypt key for the file
					\param		iv_		AES initialization vector for the file
					\returns	true if the registration could be reloaded from disk, false otherwise
					
					\note		The given path, in most circumstances, should be the same as given to the load() method!
				*/
				virtual bool reload(
					const std::string& path_,
					const crypt::aes::Key& key_,
					const crypt::aes::InitVector& iv_
				);

				/**
					\brief		Gets the value of the parent application serial number element
					\author		GW
					\date		09/2013

					\returns	The value previously set for the app's serial number, or "" if never set
				*/
				virtual std::string appSerial() const;

				/**
					\brief		Sets the value of the parent application serial number element
					\author		GW
					\date		09/2013

					\param		serial_		Serial number to set
					\returns	true if the value is set successfully, false otherwise

					\note		For Illustrator, the app serial number can be acquired with the
								hdi::core::Illustrator::serialNumber() method.
					\note		For Photoshop, the app serial number can be acquired with the
								hdi::core::Photoshop::serialNumber() method.
					\note		This method is useful to see if a user has registered their plugin with one license of
								an app, but then attempted to use the same registration with a different parent app
								license. Enforcement of this sort of registration policy is not automatic.
				*/
				virtual bool setAppSerial(const std::string& serial_);

				/**
					\brief		Gets the value of the user name element
					\author		GW
					\date		09/2013

					\returns	The value previously set for user's name, as UTF-8, or "" if never set
				*/
				virtual std::string name() const;

				/**
					\brief		Sets the value of the user name element
					\author		GW
					\date		09/2013

					\param		name_	User name to set, as UTF-8
					\returns	true if the value is set successfully, false otherwise
				*/
				virtual bool setName(const std::string& name_);

				/**
					\brief		Gets the value of the company name element
					\author		GW
					\date		09/2013

					\returns	The value previously set for company's name, as UTF-8, or "" if never set
				*/
				virtual std::string company() const;

				/**
					\brief		Sets the value of the company name element
					\author		GW
					\date		09/2013

					\param		company_	Company name to set, as UTF-8
					\returns	true if the value is set successfully, false otherwise
				*/
				virtual bool setCompany(const std::string& company_);

				/**
					\brief		Gets the value of the product (plugin) serial
					\author		GW
					\date		09/2013

					\returns	The value previously set for the product serial, or "" if never set
				*/
				virtual std::string pluginSerial() const;

				/**
					\brief		Sets the value of the product (plugin) serial number
					\author		GW
					\date		09/2013

					\param		serial_		Serial number to set
					\returns	true if the value is set successfully, false otherwise
				*/
				virtual bool setPluginSerial(const std::string& serial_);

				/**
					\brief		Gets the value of the product (plugin) upgrade serial
					\author		GW
					\date		06/2016

					\returns	The value previously set for the product upgrade serial, or "" if never set
				*/
				virtual std::string pluginUpgradeSerial() const;

				/**
					\brief		Sets the value of the product (plugin) upgrade serial
					\author		GW
					\date		06/2016

					\param		serial_		Serial number to set
					\returns	true if the value is set successfully, false otherwise
				*/
				virtual bool setPluginUpgradeSerial(const std::string& serial_);

				/**
					\brief		Gets whether the product (plugin) is in demo mode
					\author		GW
					\date		09/2013

					\returns	true if in demo mode, false otherwise
				*/
				virtual bool demoMode() const;

				/**
					\brief	Sets whether the product (plugin) is in demo mode
					\author	GW
					\date	09/2013

					\param	isDemo_		true for demo, false otherwise

					\note	This value is not stored persistently across launches
				*/
				virtual void setDemoMode(const bool isDemo_);

				/**
					\brief		Gets whether the product (plugin) is in demo mode (persistently!)
					\author		GW
					\date		01/2016

					\returns	true if demo mode, false otherwise
				*/
				virtual bool demoModePersistent() const;

				/**
					\brief	Sets whether the product (plugin) is in demo mode (persistently!)
					\author	GW
					\date	01/2016

					\param	isDemo_		true for demo, false otherwise
				*/
				virtual void setDemoModePersistent(const bool isDemo_);

				/**
					\brief		Gets the number version for the registration data
					\author		GW
					\date		09/2013

					\returns	Stored version number, or 0 if none
				*/
				virtual int32_t versionNumber() const;

				/**
					\brief		Sets the number version for the registration data
					\author		GW
					\date		09/2013

					\param		version_	New number version value
					\returns	true if the number is successfully updated, or false otherwise
				*/
				virtual bool setVersionNumber(const int32_t version_);

				/**
					\brief		Gets the version string for the registration data
					\author		GW
					\date		09/2013

					\returns	Stored version string, or "" if none
				*/
				virtual std::string versionString() const;

				/**
					\brief		Sets the version string for the registration data
					\author		GW
					\date		09/2013

					\param		version_	New string version value
					\returns	true if the string is successfully updated, or false otherwise
				*/
				virtual bool setVersionString(const std::string& version_);

				/**
					\brief		Gets the dictionary for storing user data
					\author		GW
					\date		10/2016

					\returns	Dictionary object for adding custom user data to the underlying registration data
				*/
				virtual std::unique_ptr<DictionaryPref> userData() const;


			private:
				/**
					\brief	Private implementation data
				*/
				void* __data;
		};
		
		typedef std::unique_ptr<Registration> RegistrationUP;
		typedef std::shared_ptr<Registration> RegistrationSP;
		typedef std::weak_ptr<Registration> RegistrationWP;
	}
}

#endif
// __HDI_CORE_REGISTRATION__
