/**
	\file
	\brief		Header file for handling vector graphic IDs
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_SVG_IDS__
#define __HDI_CORE_SVG_IDS__

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Stores both light and dark SVG resource IDs, such that app UI brightness changes can be properly
					handled
		*/
		class SVGIDs
		{
			public:
				/**
					\brief		Default image ID to indicate that no image has been set

					\warning	Do not use 0 as your own SVG image ID!
				*/
				static const int16_t noneID = 0;
			
				/**
					\brief	ID of the SVG resource that should be used when the UI is light in color (and Adobe expects
							the native resolution to be 2x)
				*/
				int16_t lightID_2x;
			
				/**
					\brief	ID of the SVG resource that should be used when the UI is dark in color (and Adobe expects
							the native resolution to be 2x)
				*/
				int16_t darkID_2x;
			
				/**
					\brief	Constructs a new SVGIDs object with default values (i.e. noneID)
					\author	GW
					\date	10/2018
				*/
				SVGIDs();
			
				/**
					\brief	Constructs a new SVGIDs object with the same values as an existing object
					\author	GW
					\date	10/2018
				 
					\param	svg_	SVGIDs object to copy values from
				*/
				SVGIDs(const SVGIDs& svg_);
			
				/**
					\brief	Constructs a new SVGIDs object from light and dark image IDs
					\author	GW
					\date	10/2018
				 
					\param	light2x_	Light UI resource ID at 2x native resolution
					\param	dark2x_		Dark UI resource ID at 2x native resolution
				 
					\note	SVG resource IDs from 0-5000 are reserved for hdi_core.
				*/
				SVGIDs(const int16_t light2x_, const int16_t dark2x_);
			
				/**
					\brief	Destructs a SVGIDs object
					\author	GW
					\date	10/2018
				*/
				virtual ~SVGIDs();
			
				/**
					\brief		Overloaded assignment operator to copy values from one SVGIDs object to another
					\author		GW
					\date		10/2018
				 
					\param		rhs_	Righthand side of the assignment
					\returns	The lefthand side of the assignment, but with its values updated to match rhs_
				*/
				SVGIDs& operator=(const SVGIDs& rhs_);
			
				/**
					\brief		Overloaded equality operator to compare two SVGIDs objects with one another
					\author		GW
					\date		10/2018
				 
					\param		rhs_	Righthand side of the comparison
					\returns	true if the lefthand side and righthand side of the comparison have the same image
								IDs
				*/
				bool operator==(const SVGIDs& rhs_) const;
			
				/**
					\brief		Overloaded inequality operator to compare two SVGIDs objects with one another
					\author		GW
					\date		10/2018
				 
					\param		rhs_	Righthand side of the comparison
					\returns	true if the lefthand side and righthand side of the comparison have the differing
								image IDs
				*/
				bool operator!=(const SVGIDs& rhs_) const;
		};
		
		typedef std::unique_ptr<SVGIDs> SVGIDsUP;
		typedef std::shared_ptr<SVGIDs> SVGIDsSP;
		typedef std::weak_ptr<SVGIDs> SVGIDsWP;
		
		

		/**
			\brief	Allows for image rollover support
		*/
		class SVGWithRolloverIDs : public SVGIDs
		{
			public:
			
				/**
					\brief	ID of the SVG resource that should be used for rollover when the UI is light in color (and
							Adobe expects the native resolution to be 2x)
				*/
				int16_t lightOverID_2x;
			
				/**
					\brief	ID of the SVG resource that should be used for rollover when the UI is dark in color (and
							Adobe expects the native resolution to be 2x)
				*/
				int16_t darkOverID_2x;
			
				/**
					\brief	Default SVGWithRolloverIDs constructor, setting all ID values to noneID
					\author	GW
					\date	10/2018
				*/
				SVGWithRolloverIDs();
			
				/**
					\brief	SVGWithRolloverIDs copy constructor, using values from an existing object
					\author	GW
					\date	10/2018
				 
					\param	ids_	Existing SVGWithRolloverIDs object to copy values from
				*/
				SVGWithRolloverIDs(const SVGWithRolloverIDs& ids_);
			
				/**
					\brief	Constructs a SVGWithRolloverIDs object from light and dark image IDs (without any rollovers)
					\author	GW
					\date	10/2018
				 
					\param	light2x_	Light UI resource ID at 2x native resolution
					\param	dark2x_		Dark UI resource ID at 2x native resolution
				 
					\note	SVG resource IDs from 0-5000 are reserved for hdi_core.
				*/
				SVGWithRolloverIDs(const int16_t light2x_, const int16_t dark2x_);
			
				/**
					\brief	Constructs a SVGWithRolloverIDs object from light-UI, light-UI rollover, dark-UI, and
							dark-UI resource IDs
					\author	GW
					\date	10/2018
				 
					\param	light2x_		Light UI resource ID at 2x native resolution
					\param	lightOver2x_	Light UI rollover resource ID at 2x native resolution
					\param	dark2x_			Dark UI resource ID at 2x native resolution
					\param	darkOver2x_		Dark UI rollover resource ID at 2x native resolution
				 
					\note	SVG resource IDs from 0-5000 are reserved for hdi_core.
				*/
				SVGWithRolloverIDs(
					const int16_t light2x_,
					const int16_t lightOver2x_,
					const int16_t dark2x_,
					const int16_t darkOver2x_
				);
			
				/**
					\brief	SVGWithRolloverIDs destructor
					\author	GW
					\date	10/2018
				*/
				virtual ~SVGWithRolloverIDs();
			
				/**
					\brief		Overloaded assignment operator to copy values from one SVGWithRolloverIDs object to
								another
					\author		GW
					\date		10/2018
				 
					\param		rhs_	Righthand side of the assignment
					\returns	The lefthand side of the assignment, but with its values updated to match rhs_
				*/
				SVGWithRolloverIDs& operator=(const SVGWithRolloverIDs& rhs_);
			
				/**
					\brief		Overloaded equality operator to compare two SVGWithRolloverIDs objects with one another
					\author		GW
					\date		10/2018
				 
					\param		rhs_	Righthand side of the comparison
					\returns	true if the lefthand side and righthand side of the comparison have the same image
								IDs
				*/
				bool operator==(const SVGWithRolloverIDs& rhs_) const;
			
				/**
					\brief		Overloaded inequality operator to compare two SVGWithRolloverIDs objects with one another
					\author		GW
					\date		10/2018
				 
					\param		rhs_	Righthand side of the comparison
					\returns	true if the lefthand side and righthand side of the comparison have the differing
								image IDs
				*/
				bool operator!=(const SVGWithRolloverIDs& rhs_) const;
		};
		
		typedef std::unique_ptr<SVGWithRolloverIDs> SVGWithRolloverIDsUP;
		typedef std::shared_ptr<SVGWithRolloverIDs> SVGWithRolloverIDsSP;
		typedef std::weak_ptr<SVGWithRolloverIDs> SVGWithRolloverIDsWP;
	}
}

#endif
// __HDI_CORE_SVG_IDS__
