/**
	\file
	\brief		Header file for path segment point manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_SEG_POINT__
#define __HDI_CORE_SEG_POINT__

#include "hdicorePoint.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Describes a segment point of a bezier path, typically in an Illustrator document view
		*/
		class SegPoint
		{
			public:
				/**
					\brief	Anchor point
				*/
				Point p;

				/**
					\brief	Segment "in" control point
				*/
				Point in;

				/**
					\brief	Segment "out" control point
				*/
				Point out;

				/**
					\brief	Constructs a SegPoint object with the p, in, and out points at (0,0)
					\author	GW
					\date	07/2014
				*/
				SegPoint();

				/**
					\brief	SegPoint copy constructor
					\author	GW
					\date	07/2014

					\param	segPt_	Existing SegPoint object to copy values from
				*/
				SegPoint(const SegPoint& segPt_);

				/**
					\brief	Constructs a SegPoint from a given point for the anchor and both control points
					\author	GW
					\date	07/2014

					\param	pt_			Anchor point, control points
				*/
				SegPoint(const Point& pt_);

				/**
					\brief	Constructs a SegPoint from given points
					\author	GW
					\date	07/2014

					\param	p_			Anchor point
					\param	in_			"In" control point
					\param	out_		"Out" control point
				*/
				SegPoint(
					const Point& p_,
					const Point& in_,
					const Point& out_
				);

				/**
					\brief	SegPoint destructor
					\author	GW
					\date	07/2014
				*/
				virtual ~SegPoint();

				/**
					\brief		Sets a SegPoint object from another SegPoint object
					\author		GW
					\date		07/2014

					\param		rhs_	Righthand side of the = operator; the SegPoint to pull values from
					\returns	The target SegPoint object, with its values updated
				*/
				virtual SegPoint& operator=(const SegPoint& rhs_);
				
				/**
					\brief		Tests whether a given SegPoint object is the same as another
					\author		GW
					\date		07/2014

					\param		rhs_	SegPoint to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same segment point, false otherwise
				*/
				virtual bool operator==(const SegPoint& rhs_) const;
				
				/**
					\brief		Tests whether a given SegPoint object is not the same as another
					\author		GW
					\date		07/2014

					\param		rhs_	SegPoint to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different segment points, false otherwise
				*/
				virtual bool operator!=(const SegPoint& rhs_) const;

				/**
					\brief		Gets whether the path formed by two segment points is curved
					\author		GW
					\date		07/2014

					\param		segPt2_		Second segment point in the path
					\returns	true if the path formed by the two points is curved, false otherwise
				*/
				virtual bool curved(const SegPoint& segPt2_) const;
		};
		
		typedef std::unique_ptr<SegPoint> SegPointUP;
		typedef std::shared_ptr<SegPoint> SegPointSP;
		typedef std::weak_ptr<SegPoint> SegPointWP;
	}
}

#endif
// __HDI_CORE_SEG_POINT__
