/**
	\file
	\brief		Header file for describing 2D dimensions (width and height)
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_SIZE__
#define __HDI_CORE_SIZE__

#include "hdicoreAngle.h"

namespace hdi
{
	namespace geo
	{
		class Size;
	}

	namespace core
	{
		/**
			\brief	Contains a width and height for a rectangular shape
		*/
		class Size
		{
			public:
				/**
					\brief	Width of the item
				*/
				double width;

				/**
					brief	Height of the item
				*/
				double height;

				/**
					\brief	Default Size constructor; sets width and height to 0.0
					\author	GW
					\date	08/2013
				*/
				Size();

				/**
					\brief	Size constructor, taking width and height arguments
					\author	GW
					\date	08/2013

					\param	width_		Width of the item
					\param	height_		Height of the item
				*/
				Size(const double width_, const double height_);

				/**
					\brief	Size copy constructor
					\author	GW
					\date	08/2013

					\param	size_	Existing Size object to copy
				*/
				Size(const Size& size_);
				
				/**
					\brief	Size destructor
					\author	GW
					\date	08/2013
				*/
				virtual ~Size();

				/**
					\brief		Sets a Size object from another Size object
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the = operator; Size object to pull values from
					\returns	The target Size object, but with its value updated
				*/
				virtual Size& operator=(const Size& rhs_);
				
				/**
					\brief		Tests whether a given Size object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Size to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same size, false otherwise
				*/
				virtual bool operator==(const Size& rhs_) const;
				
				/**
					\brief		Tests whether a given Size object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Size to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different sizes, false otherwise
				*/
				virtual bool operator!=(const Size& rhs_) const;

				/**
					\brief		Allows two sizes to have their width and height values added using the + operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the + operator
					\returns	A copy of the target Size, with all width and height values summed together
				*/
				virtual Size operator+(const Size& rhs_) const;

				/**
					\brief		Allows two sizes to have their width and height values added and assigned using the +=
								operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the += operator
					\returns	The lefthand side of the += operator, with all width and height values summed together
				*/
				virtual Size& operator+=(const Size& rhs_);

				/**
					\brief		Allows two sizes to have their width and height values subtracted using the - operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the - operator
					\returns	A copy of the target Size, with rhs_ width and height values subtracted from this width
								and height
				*/
				virtual Size operator-(const Size& rhs_) const;

				/**
					\brief		Allows two sizes to have their width and height values subtracted and assigned using the
								-= operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the -= operator
					\returns	The lefthand side of the -= operator, with rhs_ width and height values subtracted from
								this width and height
				*/
				virtual Size& operator-=(const Size& rhs_);

				/**
					\brief		Gets the angle from the top-left to the bottom-right of the "box" made by the Size object
					\author		GW
					\date		08/2013

					\returns	The angle from the top-left to the bottom-right
				*/
				virtual Angle diagonalAngle() const;

				/**
					\brief		Gets the length of the hypotenuse formed from the top-left to the bottom-right of the
								"box" made by the width and height
					\author		GW
					\date		08/2013

					\returns	The length from the top-left to the bottom-right
				*/
				virtual double diagonalLength() const;


			private:
				friend geo::Size* __accessImpl(const Size&);
				friend Size __accessCtor(const geo::Size&);

				/**
					\brief	Private implementation object
				*/
				geo::Size* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				Size(const geo::Size&);
		};
		
		typedef std::unique_ptr<Size> SizeUP;
		typedef std::shared_ptr<Size> SizeSP;
		typedef std::weak_ptr<Size> SizeWP;
		
		extern geo::Size* __accessImpl(const Size&);
		extern Size __accessCtor(const geo::Size&);
	}
}

#endif
// __HDI_CORE_SIZE__
