/**
	\file
	\brief		Header file for size preference data storage
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_SIZE_PREF__
#define __HDI_CORE_SIZE_PREF__

#include "hdicoreSize.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Allows for storage of 2D sizes
		*/
		class SizePref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty SizePref object
					\author	GW
					\date	09/2013
					
					\note	To test if a SizePref object is empty, call isEmpty() on it
					\note	Empty SizePref objects do not relate to any actual preference data; they are designed to be
							"receivers" of some other SizePref object via the overloaded assignment operator. Empty
							SizePref objects are useless until such time (though it is safe to call any of their methods)
				*/
				SizePref();
			
				/**
					\brief	Constructs a new SizePref object from an existing SizePref object (copy constructor)
					\author	GW
					\date	09/2013

					\param	s_	Existing SizePref object
				*/
				SizePref(const SizePref& s_);

				/**
					\brief	Constructs a SizePref object, with new data, to be added to a container at a later
							time
					\author	GW
					\date	09/2013

					\param	value_	Initial size value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				SizePref(const Size& value_, const std::string& name_ = "");

				/**
					\brief	Destructs a SizePref object
					\author	GW
					\date	09/2013
				*/
				virtual ~SizePref();

				/**
					\brief		Allows one SizePref object to be assigned from another
					\author		GW
					\date		09/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target SizePref object, but with its value updated to match that of rhs_
				*/
				virtual SizePref& operator=(const SizePref& rhs_);
				
				/**
					\brief		Convenience method to clone an SizePref object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new SizePref object
					
					\note		If you subclass SizePref, you MUST overload this method yourself! If you don't and/or 
								your clone() method does not return an instance of your SizePref subclass, you will
								experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned SizePref object.
				*/
				virtual SizePref* clone() const;

				/**
					\brief		Gets the size for this object
					\author		GW
					\date		09/2013

					\returns	The Size object being described by this wrapper
				*/
				virtual Size size() const;

				/**
					\brief		Sets the size for this object
					\author		GW
					\date		09/2013

					\param		s_	New size value
					\returns	true if the size is successfully updated, or false otherwise
				*/
				virtual bool setSize(const Size& s_);
		};
		
		typedef std::unique_ptr<Size> SizeUP;
		typedef std::shared_ptr<Size> SizeSP;
		typedef std::weak_ptr<Size> SizeWP;
	}
}

#endif
// __HDI_CORE_SIZE_PREF__
