/**
	\file
	\brief		Header file for plugin unicode string manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_STRINGS__
#define __HDI_CORE_STRINGS__

#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		namespace strings
		{
			typedef std::vector<std::string> StringVector;
			
			/**
				\brief		Gets the number of times a substring occurs in a given string
				\author		GW
				\date		08/2013

				\param		str_		Primary string, as UTF-8, from which the count of substr_ is acquired
				\param		substr_		Potential substring of str_, as UTF-8
				\returns	The number of times substr_ occurs in str_
			*/
			uint32_t substrCount(const std::string& str_, const std::string& substr_);
			
			/**
				\brief		Tokenizes a string by a given substring delimiter
				\author		GW
				\date		08/2013

				\param		str_		String to tokenize, as UTF-8
				\param		delim_		Delimiter to use for tokenization, as UTF-8
				\param		elements__	String vector for the delimited results, as UTF-8
				\returns	true if the string could be split, false otherwise (only fails if str_ doesn't contain delim_)
			*/
			bool split(const std::string& str_, const std::string& delim_, StringVector& elements__);

			/**
				\brief		Joins the given string elements into a single string with a given delimiter
				\author		GW
				\date		07/2025

				\param		elements_	Strings to join, as UTF-8
				\param		delim_		Delimiter to use for joining, as UTF-8
				\returns	The given elements in elements_ joined into a single string with delim_
			*/
			std::string join(const StringVector& elements_, const std::string& delim_);

			/**
				\brief		Trims the whitespace from both ends of a string, returning a duplicate
				\author		GW
				\date		08/2013

				\param		str_	String to trim, as UTF-8
				\returns	Value of str_ argument but with prefixing and suffixing whitespace removed, as UTF-8
			*/
			std::string trim(const std::string& str_);

			/**
				\brief		Uppercases an entire string, returning a duplicate
				\author		GW
				\date		08/2013

				\param		str_	String to uppercase, as UTF-8
				\returns	Value of str_ argument but with alphabetic characters uppercased, as UTF-8
			*/
			std::string uppercase(const std::string& str_);

			/**
				\brief		Lowercases an entire string, returning a duplicate
				\author		GW
				\date		08/2013

				\param		str_	String to lowercase, as UTF-8
				\returns	Value of str_ argument but with alphabetic characters lowercased, as UTF-8
			*/
			std::string lowercase(const std::string& str_);

			/**
				\brief		Replaces all instances of a search string, in a subject string, with a given replacement
				\author		GW
				\date		08/2013

				\param		subject_	String in which the search_ substring will be replaced with replace_, as UTF-8
				\param		search_		Substring to search for and ultimately replace, as UTF-8
				\param		replace_	Replacement string for search_, as UTF-8
				\returns	Value of subject_ argument as UTF-8, but with all instances of search_ argument value
							replaced with the value of the replace_ argument
			*/
			std::string replaceAll(const std::string& subject_, const std::string& search_, const std::string& replace_);
			
			/**
				\brief		Converts the given UTF-8 string to a UTF-16 string
				\author		GW
				\date		10/2015
				
				\param		str_	UTF-8 string to convert
				\returns	Resultant UTF-16 string
			*/
			UTF16String utf8ToUTF16(const UTF8String& str_);
			
			/**
				\brief		Converts the given UTF-8 string to a UTF-32 string
				\author		GW
				\date		10/2015
				
				\param		str_	UTF-8 string to convert
				\returns	Resultant UTF-32 string
			*/
			UTF32String utf8ToUTF32(const UTF8String& str_);
			
			/**
				\brief		Converts the given UTF-16 string to a UTF-8 string
				\author		GW
				\date		10/2015
				
				\param		str_	UTF-16 string to convert
				\returns	Resultant UTF-8 string
			*/
			UTF8String utf16ToUTF8(const UTF16String& str_);
			
			/**
				\brief		Converts the given UTF-16 string to a UTF-32 string
				\author		GW
				\date		10/2015
				
				\param		str_	UTF-16 string to convert
				\returns	Resultant UTF-32 string
			*/
			UTF32String utf16ToUTF32(const UTF16String& str_);
			
			/**
				\brief		Converts the given UTF-32 string to a UTF-8 string
				\author		GW
				\date		10/2015
				
				\param		str_	UTF-32 string to convert
				\returns	Resultant UTF-8 string
			*/
			UTF8String utf32ToUTF8(const UTF32String& str_);
			
			/**
				\brief		Converts the given UTF-32 string to a UTF-16 string
				\author		GW
				\date		10/2015
				
				\param		str_	UTF-32 string to convert
				\returns	Resultant UTF-16 string
			*/
			UTF16String utf32ToUTF16(const UTF32String& str_);
		}
	}
}

#endif
// __HDI_CORE_STRINGS__
