/**
	\file
	\brief		Header file for stroke style map preference data storage
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_STROKE_STYLE_MAP_PREF__
#define __HDI_CORE_STROKE_STYLE_MAP_PREF__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicorePathStyle.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace aip
	{
		class PersistentDictionary;
	}

	namespace core
	{
		/**
			\brief	Allows for storage of stroke style maps
		*/
		class StrokeStyleMapPref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty StrokeStyleMapPref object
					\author	GW
					\date	11/2015
					
					\note	To test if a StrokeStyleMapPref object is empty, call isEmpty() on it
					\note	Empty StrokeStyleMapPref objects do not relate to any actual preference data; they are designed
							to be "receivers" of some other StrokeStyleMapPref object via the overloaded assignment operator.
							Empty StrokeStyleMapPref objects are useless until such time (though it is safe to call any of
							their methods)
				*/
				StrokeStyleMapPref();
			
				/**
					\brief	Constructs a new StrokeStyleMapPref object from an existing StrokeStyleMapPref object (copy constructor)
					\author	GW
					\date	11/2015

					\param	ss_		Existing StrokeStyleMapPref object
				*/
				StrokeStyleMapPref(const StrokeStyleMapPref& ss_);

				/**
					\brief	Constructs a StrokeStyleMapPref object, with new data, to be added to a container at a later time
					\author	GW
					\date	11/2015

					\param	value_	Initial stroke style map value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				StrokeStyleMapPref(const StrokeStyle::Map& value_, const std::string& name_ = "");

				/**
					\brief	Destructs a StrokeStyleMapPref object
					\author	GW
					\date	11/2015
				*/
				virtual ~StrokeStyleMapPref();

				/**
					\brief		Allows one StrokeStyleMapPref object to be assigned from another
					\author		GW
					\date		11/2015

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target StrokeStyleMapPref object, but with its value updated to match that of rhs_
				*/
				virtual StrokeStyleMapPref& operator=(const StrokeStyleMapPref& rhs_);
				
				/**
					\brief		Convenience method to clone an StrokeStyleMapPref object on the heap
					\author		GW
					\date		11/2015

					\returns	A pointer to the new StrokeStyleMapPref object
					
					\note		If you subclass StrokeStyleMapPref, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your StrokeStyleMapPref subclass, you will
								experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned StrokeStyleMapPref object.
				*/
				virtual StrokeStyleMapPref* clone() const;

				/**
					\brief		Gets the type of persistent data
					\author		GW
					\date		11/2015

					\returns	StrokeStyleMapType

					\note		This is useful for knowing how to cast a PrefData object to its proper subclass
				*/
				virtual Type dataType() const;

				/**
					\brief		Gets the stroke style map value for this object
					\author		GW
					\date		11/2015

					\returns	The stroke style map value being described by this wrapper
				*/
				virtual StrokeStyle::Map strokeStyleMap() const;

				/**
					\brief		Sets the stroke style map value for this object
					\author		GW
					\date		11/2015

					\param		value_	New stroke style map value
					\returns	true if the stroke style map is successfully updated, or false otherwise
				*/
				virtual bool setStrokeStyleMap(const StrokeStyle::Map& value_);


			private:
				friend class PathStyleMapPref;

				/**
					\brief	Internal use only
					\author	GW
					\date	11/2015
				*/
				StrokeStyleMapPref(aip::PersistentDictionary*&);
		};
		
		typedef std::unique_ptr<StrokeStyleMapPref> StrokeStyleMapPrefUP;
		typedef std::shared_ptr<StrokeStyleMapPref> StrokeStyleMapPrefSP;
		typedef std::weak_ptr<StrokeStyleMapPref> StrokeStyleMapPrefWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_STROKE_STYLE_MAP_PREF__
