/**
	\file
	\brief		Header file for general Illustrator text art manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_TEXT_ART__
#define __HDI_CORE_TEXT_ART__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreTypes.h"
	
namespace hdi
{
	namespace core
	{
		class Art;
		class TextLine;
		class TextRange;
		class TextStory;
	
		/**
			\brief	Allows for text frame manipulation, selection, metadata access, etc.
		*/
		class TextArt
		{
			public:
				typedef std::vector< std::shared_ptr<TextLine> > TextLineVector;
			
				enum Type
				{
					UnknownType		= 0,
					PointTextType	= 10,
					InPathTextType	= 20,
					OnPathTextType	= 30
				};
				
				enum LineOrientation
				{
					UnknownLineOrientation	= 0,
					HorizLineOrientation	= 10,
					VertL2RLineOrientation	= 20,	// Not used by Illustrator
					VertR2LLineOrientation	= 30
				};

				/**
					\brief	Destructs a TextArt object
					\author	GW
					\date	08/2013
				*/
				~TextArt();

				/**
					\brief		Gets the ATE text frame ref around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	ATE::ITextFrame for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				std::unique_ptr<ATE::ITextFrame> ateTextFrame() const;
	
				/**
					\brief		Retrieves the parent story of the target text frame
					\author		GW
					\date		12/2013
					
					\returns	The parent story, or NULL if error
				*/
				std::unique_ptr<TextStory> parentStory() const;
				
				/**
					\brief		Retrieves the text range of the target frame
					\author		GW
					\date		12/2013
					
					\returns	The text range, or NULL if error
				*/
				std::unique_ptr<TextRange> textRange() const;

				/**
					\brief		Retrieves a text-line vector with which you can access the text lines of the target text
								frame
					\author		GW
					\date		12/2013
					
					\returns	A vector of TextLine object pointers
				*/
				TextLineVector textLines() const;

				/**
					\brief		Retrieves the type of the target text frame
					\author		GW
					\date		12/2013
					
					\returns	The text frame type
				*/
				Type frameType() const;

				/**
					\brief		Reports whether the target frame is selected
					\author		GW
					\date		12/2013
					
					\returns	true if selected, false otherwise
				*/
				bool selected() const;

				/**
					\brief		Retrieves the line orientation of the target text frame
					\author		GW
					\date		12/2013
					
					\returns	The line orientation
				*/
				LineOrientation lineOrientation() const;
				
				/**
					\brief	Sets the line orientation value for the target text frame
					\author	GW
					\date	12/2013
					
					\param	orientation_	New line orientation
				*/
				void setLineOrientation(const LineOrientation orientation_);

				/**
					\brief		Retrieves the number of rows for the target text frame
					\author		GW
					\date		12/2013
					
					\returns	Number of frame rows
				*/
				int32_t rowCount() const;
								
				/**
					\brief	Sets the number of rows for the target text frame
					\author	GW
					\date	12/2013
					
					\param	count_	New row count
				*/
				void setRowCount(const int32_t count_);

				/**
					\brief		Retrieves the number of columns for the target text frame
					\author		GW
					\date		12/2013
					
					\returns	Number of frame columns
				*/
				int32_t columnCount() const;
								
				/**
					\brief	Sets the number of columns for the target text frame
					\author	GW
					\date	12/2013
					
					\param	count_	New column count
				*/
				void setColumnCount(const int32_t count_);

				/**
					\brief		Reports whether the text range is arranged in row-major order
					\author		GW
					\date		12/2013
					
					\returns	true if in row-major order, false otherwise
				*/
				bool rowMajorOrder() const;
								
				/**
					\brief	Turns row-major order on or off for the target text frame
					\author	GW
					\date	12/2013
					
					\param	isMajor_	true to enable row-major order, false otherwise
				*/
				void setRowMajorOrder(const bool isMajor_);
				
				/**
					\brief		Retrieves the row gutter value for the target text frame
					\author		GW
					\date		12/2013
					
					\returns	The frame's row gutter value
				*/
				double rowGutter() const;
								
				/**
					\brief	Sets the row gutter value for the target text frame
					\author	GW
					\date	12/2013
					
					\param	gutter_		New row gutter value
				*/
				void setRowGutter(const double gutter_);

				/**
					\brief		Retrieves the column gutter value for the target text frame
					\author		GW
					\date		12/2013
					
					\returns	The frame's column gutter value
				*/
				double columnGutter() const;
								
				/**
					\brief	Sets the column gutter value for the target text frame
					\author	GW
					\date	12/2013
					
					\param	gutter_		New column gutter value
				*/
				void setColumnGutter(const double gutter_);

				/**
					\brief		Retrieves the line spacing value for the target text frame
					\author		GW
					\date		12/2013
					
					\returns	The frame's line spacing value
				*/
				double spacing() const;
								
				/**
					\brief	Sets the line spacing value for the target text frame
					\author	GW
					\date	12/2013
					
					\param	spacing_	New line spacing value
				*/
				void setSpacing(const double spacing_);

				/**
					\brief		Reports whether optical alignment is on for the target text frame
					\author		GW
					\date		12/2013
					
					\returns	true if optical alignment is active, false otherwise
				*/
				bool opticalAlignment() const;
				
				/**
					\brief	Turns optical alignment on or off for the target text frame
					\author	GW
					\date	12/2013
					
					\param	active_		true to activate optical alignment, false otherwise
				*/
				void setOpticalAlignment(const bool active_);


			private:
				// Only these classes can construct a new TextArt object
				friend class Art;
				friend class TextLine;
				friend class TextRange;
				friend class TextStory;
				
				/**
					\brief	Art to which the target text frame belongs
				*/
				void* __data;
				
				/**
					\brief	Whether to cleanup internal data
				*/
				bool __cleanup;
				
				/**
					\brief	Empty TextArt objects cannot be constructed
					\author	GW
					\date	08/2013
				*/
				TextArt();

				/**
					\brief	Constructs a new TextArt object from a given Art object
					\author	GW
					\date	08/2013

					\param	art_	Pointer for the existing art we're wrapping around
				*/
				TextArt(void* const art_);
			
				/**
					\brief	TextArt objects cannot be copied
					\author	GW
					\date	08/2013
				*/
				TextArt(const TextArt&);
			
				/**
					\brief	TextArt objects cannot be copied
					\author	GW
					\date	08/2013
				*/
				TextArt& operator=(const TextArt&);
		};
		
		typedef std::unique_ptr<TextArt> TextArtUP;
		typedef std::shared_ptr<TextArt> TextArtSP;
		typedef std::weak_ptr<TextArt> TextArtWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_ART__
