/**
	\file
	\brief		Header file for general Illustrator text character feature manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_TEXT_CHAR_FEATURES__
#define __HDI_CORE_TEXT_CHAR_FEATURES__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicorePathStyle.h"
#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextCharFeatures;
	}

	namespace core
	{
		class Angle;
		class ArtColor;
		class Font;
	
		/**
			\brief	Allows for easy manipulation of Illustrator text character features
		*/
		class TextCharFeatures
		{
			public:
				typedef std::vector<double> DoubleVector;
			
				enum Language
				{
					UnknownLanguage				= 0,
					AlbanianLanguage			= 10,
					ArabicLanguage				= 20,
					BelarussianLanguage			= 30,
					BengaliLanguage				= 40,
					BokmalNorwegianLanguage		= 50,
					BulgarianLanguage			= 60,
					BurmeseLanguage				= 65,
					CatalanLanguage				= 70,
					ChineseLanguage				= 80,
					CroatianLanguage			= 90,
					CzechLanguage				= 100,
					DanishLanguage				= 110,
					DutchLanguage				= 120,
					EnglishLanguage				= 130,
					EnglishBritishLanguage		= 140,
					EnglishCanadianLanguage		= 150,
					EstonianLanguage			= 160,
					FarsiLanguage				= 170,
					FinnishLanguage				= 180,
					FrenchLanguage				= 190,
					FrenchCanadianLanguage		= 200,
					GermanLanguage				= 210,
					GermanSwissLanguage			= 220,
					GreekLanguage				= 230,
					GujaratiLanguage			= 240,
					HebrewLanguage				= 250,
					HindiLanguage				= 260,
					HungarianLanguage			= 270,
					IcelandicLanguage			= 280,
					IndonesianLanguage			= 285,
					ItalianLanguage				= 290,
					JapaneseLanguage			= 300,
					KannadaLanguage				= 310,
					KhmerLanguage				= 315,
					LaoLanguage					= 320,
					LatvianLanguage				= 330,
					LithuanianLanguage			= 340,
					MalayalamLanguage			= 350,
					MarathiLanguage				= 360,
					NynorskNorwegianLanguage	= 370,
					OriyaLanguage				= 380,
					PolishLanguage				= 390,
					PortugueseLanguage			= 400,
					PortugueseBrazilianLanguage	= 410,
					PunjabiLanguage				= 420,
					RomanianLanguage			= 430,
					RussianLanguage				= 440,
					SerbianLanguage				= 450,
					SinhaleseLanguage			= 455,
					SlovakianLanguage			= 460,
					SlovenianLanguage			= 470,
					SpanishLanguage				= 480,
					SwedishLanguage				= 490,
					TamilLanguage				= 500,
					TeluguLanguage				= 510,
					ThaiLanguage				= 520,
					TurkishLanguage				= 530,
					UkranianLanguage			= 540,
					VietnameseLanguage			= 550
				};
			
				/**
					\brief	Creates a new TextCharFeatures object
					\author	GW
					\date	08/2014
					
					\note	New TextCharFeatures objects do not relate to any text characters on the Illustrator
							artboard; they are designed to be "receivers" of some other TextCharFeatures object via the
							overloaded assignment operator or assigned to some other existent text art.
				*/
				TextCharFeatures();
			
				/**
					\brief	TextCharFeatures copy constructor
					\author	GW
					\date	08/2014
					
					\param	cf_	TextCharFeatures object to copy values from
				*/
				TextCharFeatures(const TextCharFeatures& cf_);

				/**
					\brief	Destructs a TextCharFeatures object
					\author	GW
					\date	08/2014
				*/
				virtual ~TextCharFeatures();
			
				/**
					\brief		Overloaded assignment operator for TextCharFeatures objects
					\author		GW
					\date		08/2014

					\param		rhs_	Existing TextCharFeatures object to copy values from
					\returns	The target TextCharFeatures object, but with its values updated to match that of rhs_
				*/
				virtual TextCharFeatures& operator=(const TextCharFeatures& rhs_);
				
				/**
					\brief		Tests whether a given TextCharFeatures object is the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextCharFeatures to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text char features, false otherwise
				*/
				virtual bool operator==(const TextCharFeatures& rhs_) const;
				
				/**
					\brief		Tests whether a given TextCharFeatures object is not the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextCharFeatures to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text char features, false otherwise
				*/
				virtual bool operator!=(const TextCharFeatures& rhs_) const;

				/**
					\brief		Gets the ATE char features ref around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	ATE::ICharFeatures for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::ICharFeatures> ateCharFeatures() const;

				/**
					\brief		Gets whether the target TextCharFeatures object is empty
					\author		GW
					\date		08/2014
					
					\returns	true if the target TextCharFeatures object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Duplicates the target object
					\author		GW
					\date		08/2014
					
					\returns	A duplicate of the target object, or NULL for error
				*/
				virtual std::unique_ptr<TextCharFeatures> duplicate() const;
				
				/**
					\brief		Retrieves the font attribute of the character
					\author		GW
					\date		08/2014
					
					\param		font__	Return-by-reference for the font
					\returns	true if the attribute could be acquired and font__ was set, false otherwise
				*/
				virtual bool font(Font& font__) const;
				
				/**
					\brief		Retrieves the font size attribute of the character
					\author		GW
					\date		08/2014
					
					\param		size__	Return-by-reference for the font size
					\returns	true if the attribute could be acquired and size__ was set, false otherwise
				*/
				virtual bool fontSize(double& size__) const;
				
				/**
					\brief		Retrieves the horizontal scale attribute of the character
					\author		GW
					\date		08/2014
					
					\param		scale__	Return-by-reference for the horizontal scale
					\returns	true if the attribute could be acquired and scale__ was set, false otherwise
				*/
				virtual bool horizontalScale(double& scale__) const;
				
				/**
					\brief		Retrieves the vertical scale attribute of the character
					\author		GW
					\date		08/2014
					
					\param		scale__	Return-by-reference for the vertical scale
					\returns	true if the attribute could be acquired and scale__ was set, false otherwise
				*/
				virtual bool verticalScale(double& scale__) const;
				
				/**
					\brief		Retrieves the automatic leading attribute of the character
					\author		GW
					\date		08/2014
					
					\param		leading__	Return-by-reference for the automatic leading
					\returns	true if the attribute could be acquired and leading__ was set, false otherwise
				*/
				virtual bool autoLeading(bool& leading__) const;
				
				/**
					\brief		Retrieves the leading attribute of the character
					\author		GW
					\date		08/2014
					
					\param		leading__	Return-by-reference for the leading attribute
					\returns	true if the attribute could be acquired and leading__ was set, false otherwise
				*/
				virtual bool leading(double& leading__) const;
				
				/**
					\brief		Retrieves the tracking attribute of the character
					\author		GW
					\date		08/2014
					
					\param		tracking__	Return-by-reference for the tracking attribute
					\returns	true if the attribute could be acquired and tracking__ was set, false otherwise
				*/
				virtual bool tracking(int32_t& tracking__) const;
				
				/**
					\brief		Retrieves the baseline shift attribute of the character
					\author		GW
					\date		08/2014
					
					\param		shift__	Return-by-reference for the baseline shift
					\returns	true if the attribute could be acquired and shift__ was set, false otherwise
				*/
				virtual bool baselineShift(double& shift__) const;
				
				/**
					\brief		Retrieves the character rotation attribute of the character
					\author		GW
					\date		08/2014
					
					\param		rotate__	Return-by-reference for the character rotation
					\returns	true if the attribute could be acquired and rotate__ was set, false otherwise
				*/
				virtual bool characterRotation(Angle& rotate__) const;
				
				/**
					\brief		Retrieves the automatic kerning type attribute of the character
					\author		GW
					\date		08/2014
					
					\param		kerning__	Return-by-reference for the automatic kerning
					\returns	true if the attribute could be acquired and kerning__ was set, false otherwise

					\note		The actual kerning value applies to character pairs rather than individual characters,
								so is available through TextCharStyle.
				*/
				virtual bool autoKernType(AutoKernType& kerning__) const;
				
				/**
					\brief		Retrieves the font caps option attribute of the character
					\author		GW
					\date		08/2014
					
					\param		caps__	Return-by-reference for the font caps option
					\returns	true if the attribute could be acquired and caps__ was set, false otherwise
				*/
				virtual bool fontCapsOption(FontCapsType& caps__) const;
				
				/**
					\brief		Retrieves the font baseline option attribute of the character
					\author		GW
					\date		08/2014
					
					\param		baseline__	Return-by-reference for the font baseline option
					\returns	true if the attribute could be acquired and baseline__ was set, false otherwise
				*/
				virtual bool fontBaselineOption(FontBaseline& baseline__) const;
				
				/**
					\brief		Retrieves the font OpenType position option attribute of the character
					\author		GW
					\date		08/2014
					
					\param		pos__	Return-by-reference for the font OpenType position
					\returns	true if the attribute could be acquired and pos__ was set, false otherwise
				*/
				virtual bool fontOpenTypePositionOption(FontOpenTypePosition& pos__) const;
				
				/**
					\brief		Retrieves the strikethrough position attribute of the character
					\author		GW
					\date		08/2014
					
					\param		pos__	Return-by-reference for the strikethrough position
					\returns	true if the attribute could be acquired and pos__ was set, false otherwise
				*/
				virtual bool strikethroughPosition(StrikethroughPosition& pos__) const;
				
				/**
					\brief		Retrieves the underline position attribute of the character
					\author		GW
					\date		08/2014
					
					\param		pos__	Return-by-reference for the underline position
					\returns	true if the attribute could be acquired and pos__ was set, false otherwise
				*/
				virtual bool underlinePosition(UnderlinePosition& pos__) const;
				
				/**
					\brief		Retrieves the underline offset attribute of the character
					\author		GW
					\date		08/2014
					
					\param		offset__	Return-by-reference for the underline offset
					\returns	true if the attribute could be acquired and offset__ was set, false otherwise
				*/
				virtual bool underlineOffset(double& offset__) const;
				
				/**
					\brief		Retrieves the OpenType ligature attribute of the character
					\author		GW
					\date		08/2014
					
					\param		lig__	Return-by-reference for the OpenType ligature
					\returns	true if the attribute could be acquired and lig__ was set, false otherwise
				*/
				virtual bool ligature(bool& lig__) const;
				
				/**
					\brief		Retrieves the OpenType discretionary ligatures attribute of the character
					\author		GW
					\date		08/2014
					
					\param		lig__	Return-by-reference for the OpenType discretionary ligatures
					\returns	true if the attribute could be acquired and lig__ was set, false otherwise
				*/
				virtual bool discretionaryLigatures(bool& lig__) const;
				
				/**
					\brief		Retrieves the OpenType context ligatures attribute of the character
					\author		GW
					\date		08/2014
					
					\param		lig__	Return-by-reference for the OpenType context ligatures
					\returns	true if the attribute could be acquired and lig__ was set, false otherwise
				*/
				virtual bool contextualLigatures(bool& lig__) const;
				
				/**
					\brief		Retrieves the OpenType alternate ligatures attribute of the character
					\author		GW
					\date		08/2014
					
					\param		lig__	Return-by-reference for the OpenType alternate ligatures
					\returns	true if the attribute could be acquired and lig__ was set, false otherwise
				*/
				virtual bool alternateLigatures(bool& lig__) const;
				
				/**
					\brief		Retrieves the OpenType old-style attribute of the character
					\author		GW
					\date		08/2014
					
					\param		old__	Return-by-reference for the OpenType old-style
					\returns	true if the attribute could be acquired and old__ was set, false otherwise
				*/
				virtual bool oldStyle(bool& old__) const;
				
				/**
					\brief		Retrieves the OpenType fractions attribute of the character
					\author		GW
					\date		08/2014
					
					\param		frac__	Return-by-reference for the OpenType fractions
					\returns	true if the attribute could be acquired and frac__ was set, false otherwise
				*/
				virtual bool fractions(bool& frac__) const;
				
				/**
					\brief		Retrieves the OpenType ordinals attribute of the character
					\author		GW
					\date		08/2014
					
					\param		ord__	Return-by-reference for the OpenType ordinals
					\returns	true if the attribute could be acquired and ord__ was set, false otherwise
				*/
				virtual bool ordinals(bool& ord__) const;
				
				/**
					\brief		Retrieves the OpenType swash attribute of the character
					\author		GW
					\date		08/2014
					
					\param		swash__	Return-by-reference for the OpenType swash
					\returns	true if the attribute could be acquired and swash__ was set, false otherwise
				*/
				virtual bool swash(bool& swash__) const;
				
				/**
					\brief		Retrieves the OpenType titling attribute of the character
					\author		GW
					\date		08/2014
					
					\param		titling__	Return-by-reference for the OpenType titling
					\returns	true if the attribute could be acquired and titling__ was set, false otherwise
				*/
				virtual bool titling(bool& titling__) const;
				
				/**
					\brief		Retrieves the OpenType connection forms attribute of the character
					\author		GW
					\date		08/2014
					
					\param		connForms__	Return-by-reference for the OpenType connection forms
					\returns	true if the attribute could be acquired and connForms__ was set, false otherwise
				*/
				virtual bool connectionForms(bool& connForms__) const;
				
				/**
					\brief		Retrieves the OpenType stylistic alternates attribute of the character
					\author		GW
					\date		08/2014
					
					\param		styleAlt__	Return-by-reference for the OpenType stylistic alternates
					\returns	true if the attribute could be acquired and styleAlt__ was set, false otherwise
				*/
				virtual bool stylisticAlternates(bool& styleAlt__) const;

				/**
					\brief		Retrieves the OpenType stylistic sets attribute of the character
					\author		GW
					\date		07/2017
					
					\param		styleSets__	Return-by-reference for the OpenType stylistic sets
					\returns	true if the attribute could be acquired and styleSets__ was set, false otherwise
				*/
				bool stylisticSets(int32_t& styleSets__) const;
				
				/**
					\brief		Retrieves the OpenType ornaments attribute of the character
					\author		GW
					\date		08/2014
					
					\param		orn__	Return-by-reference for the OpenType ornaments
					\returns	true if the attribute could be acquired and orn__ was set, false otherwise
				*/
				virtual bool ornaments(bool& orn__) const;
				
				/**
					\brief		Retrieves the OpenType figure style attribute of the character
					\author		GW
					\date		08/2014
					
					\param		style__	Return-by-reference for the OpenType figure style
					\returns	true if the attribute could be acquired and style__ was set, false otherwise
				*/
				virtual bool figureStyle(FigureStyle& style__) const;
				
				/**
					\brief		Retrieves the no-break attribute of the character
					\author		GW
					\date		08/2014
					
					\param		noBr__	Return-by-reference for the no-break
					\returns	true if the attribute could be acquired and noBr__ was set, false otherwise
				*/
				virtual bool noBreak(bool& noBr__) const;
				
				/**
					\brief		Retrieves the fill color attribute of the character
					\author		GW
					\date		08/2014
					
					\param		color__	Return-by-reference for the fill color
					\returns	true if the attribute could be acquired and color__ was set, false otherwise
				*/
				virtual bool fillColor(ArtColor& color__) const;
				
				/**
					\brief		Retrieves the fill attribute of the character
					\author		GW
					\date		08/2014
					
					\param		fill__	Return-by-reference for the fill
					\returns	true if the attribute could be acquired and fill__ was set, false otherwise
				*/
				virtual bool fill(bool& fill__) const;

				/**
					\brief		Retrieves the fill visibility attribute of the character
					\author		GW
					\date		08/2014
					
					\param		fillVis__	Return-by-reference for the fill visibility
					\returns	true if the attribute could be acquired and fillVis__ was set, false otherwise
				*/
				virtual bool fillVisible(bool& fillVis__) const;
				
				/**
					\brief		Retrieves the stroke color attribute of the character
					\author		GW
					\date		08/2014
					
					\param		color__	Return-by-reference for the stroke color
					\returns	true if the attribute could be acquired and color__ was set, false otherwise
				*/
				virtual bool strokeColor(ArtColor& color__) const;
				
				/**
					\brief		Retrieves the stroke attribute of the character
					\author		GW
					\date		08/2014
					
					\param		stroke__	Return-by-reference for the stroke
					\returns	true if the attribute could be acquired and stroke__ was set, false otherwise
				*/
				virtual bool stroke(bool& stroke__) const;

				/**
					\brief		Retrieves the stroke visibility attribute of the character
					\author		GW
					\date		08/2014
					
					\param		strokeVis__	Return-by-reference for the stroke visibility
					\returns	true if the attribute could be acquired and strokeVis__ was set, false otherwise
				*/
				virtual bool strokeVisible(bool& strokeVis__) const;
				
				/**
					\brief		Retrieves the fill-first attribute of the character
					\author		GW
					\date		08/2014
					
					\param		fillFirst__	Return-by-reference for the fill-first
					\returns	true if the attribute could be acquired and fillFirst__ was set, false otherwise
				*/
				virtual bool fillFirst(bool& fillFirst__) const;
				
				/**
					\brief		Retrieves the fill overprint attribute of the character
					\author		GW
					\date		08/2014
					
					\param		overprint__	Return-by-reference for the fill overprint
					\returns	true if the attribute could be acquired and overprint__ was set, false otherwise
				*/
				virtual bool fillOverPrint(bool& overprint__) const;
				
				/**
					\brief		Retrieves the stroke overprint attribute of the character
					\author		GW
					\date		08/2014
					
					\param		overprint__	Return-by-reference for the stroke overprint
					\returns	true if the attribute could be acquired and overprint__ was set, false otherwise
				*/
				virtual bool strokeOverPrint(bool& overprint__) const;

				/**
					\brief		Retrieves the fill background color attribute of the character
					\author		GW
					\date		08/2014
					
					\param		color__	Return-by-reference for the fill background
					\returns	true if the attribute could be acquired and color__ was set, false otherwise
				*/
				virtual bool fillBackgroundColor(ArtColor& color__) const;
				
				/**
					\brief		Retrieves the fill background attribute of the character
					\author		GW
					\date		08/2014
					
					\param		fillBG__	Return-by-reference for the fill background
					\returns	true if the attribute could be acquired and fillBG__ was set, false otherwise
				*/
				virtual bool fillBackground(bool& fillBG__) const;
				
				/**
					\brief		Retrieves the line cap attribute of the character
					\author		GW
					\date		08/2014
					
					\param		cap__	Return-by-reference for the line cap
					\returns	true if the attribute could be acquired and cap__ was set, false otherwise
				*/
				virtual bool lineCap(StrokeStyle::CapStyle& cap__) const;
				
				/**
					\brief		Retrieves the line join attribute of the character
					\author		GW
					\date		08/2014
					
					\param		join__	Return-by-reference for the line join
					\returns	true if the attribute could be acquired and join__ was set, false otherwise
				*/
				virtual bool lineJoin(StrokeStyle::JoinStyle& join__) const;
				
				/**
					\brief		Retrieves the line width attribute of the character
					\author		GW
					\date		08/2014
					
					\param		width__	Return-by-reference for the line width
					\returns	true if the attribute could be acquired and width__ was set, false otherwise
				*/
				virtual bool lineWidth(double& width__) const;
				
				/**
					\brief		Retrieves the miter limit attribute of the character
					\author		GW
					\date		08/2014
					
					\param		limit__	Return-by-reference for the miter limit
					\returns	true if the attribute could be acquired and limit__ was set, false otherwise
				*/
				virtual bool miterLimit(double& limit__) const;
				
				/**
					\brief		Retrieves the line dash offset attribute of the character
					\author		GW
					\date		08/2014
					
					\param		offset__	Return-by-reference for the line dash offset
					\returns	true if the attribute could be acquired and offset__ was set, false otherwise
				*/
				virtual bool lineDashOffset(double& offset__) const;
				
				/**
					\brief		Retrieves the line dash array attribute of the character
					\author		GW
					\date		08/2014
					
					\param		dists__	Return-by-reference for the line dash array (as a vector)
					\returns	true if the attribute could be acquired and dists__ was set, false otherwise
				*/
				virtual bool lineDashArray(DoubleVector& dists__) const;

				/**
					\brief		Retrieves the language attribute of the character
					\author		GW
					\date		08/2014
					
					\param		lang__	Return-by-reference for the language
					\returns	true if the attribute could be acquired and lang__ was set, false otherwise
				*/
				virtual bool language(Language& lang__) const;
				
				/**
					\brief	Sets the local font attribute for the character
					\author	GW
					\date	08/2014

					\param	font_	The new font object
				*/
				virtual void setFont(const Font& font_);
				
				/**
					\brief	Sets the local font size attribute for the character
					\author	GW
					\date	08/2014

					\param	size_	The new value, in the range [0.1,1296.0]
				*/
				virtual void setFontSize(const double size_);
				
				/**
					\brief	Sets the local horizontal scale attribute for the character
					\author	GW
					\date	08/2014

					\param	scale_	The new scaling factor, in the range [0.1,100.0], where 1.0 means 100%
				*/
				virtual void setHorizontalScale(const double scale_);
				
				/**
					\brief	Sets the local vertical scale attribute for the character
					\author	GW
					\date	08/2014

					\param	scale_	The new scaling factor, in the range [0.1,100.0], where 1.0 means 100%
				*/
				virtual void setVerticalScale(const double scale_);
				
				/**
					\brief	Sets the local automatic leading attribute for the character
					\author	GW
					\date	08/2014

					\param	leading_	true to turn automatic leading on, false to turn it off
				*/
				virtual void setAutoLeading(const bool leading_);
				
				/**
					\brief	Sets the local leading attribute for the character
					\author	GW
					\date	08/2014

					\param	leading_	The new leading value in document points
				*/
				virtual void setLeading(const double leading_);
				
				/**
					\brief	Sets the local tracking attribute for the character
					\author	GW
					\date	08/2014

					\param	tracking_	The new tracking value
				*/
				virtual void setTracking(const int32_t tracking_);
				
				/**
					\brief	Sets the local baseline shift attribute for the character
					\author	GW
					\date	08/2014

					\param	shift_	The new baseline shift value in document points
				*/
				virtual void setBaselineShift(const double shift_);
				
				/**
					\brief	Sets the local character rotation attribute for the character
					\author	GW
					\date	08/2014

					\param	rotation_	The new rotation value
				*/
				virtual void setCharacterRotation(const Angle& rotation_);
				
				/**
					\brief	Sets the local automatic kerning type attribute for the character style
					\author	GW
					\date	08/2014

					\param	kerning_	The new automatic kerning type
				*/
				virtual void setAutoKernType(const AutoKernType kerning_);
				
				/**
					\brief	Sets the local font caps option attribute for the character style
					\author	GW
					\date	08/2014

					\param	caps_	The new font caps option
				*/
				virtual void setFontCapsOption(const FontCapsType caps_);
				
				/**
					\brief	Sets the local font baseline option attribute for the character style
					\author	GW
					\date	08/2014

					\param	baseline_	The new font baseline option
				*/
				virtual void setFontBaselineOption(const FontBaseline baseline_);
				
				/**
					\brief	Sets the local font OpenType position option attribute for the character style
					\author	GW
					\date	08/2014

					\param	pos_	The new font OpenType position option
				*/
				virtual void setFontOpenTypePositionOption(const FontOpenTypePosition pos_);
				
				/**
					\brief	Sets the local strikethrough position  attribute for the character style
					\author	GW
					\date	08/2014

					\param	pos_	The new strikethrough position
				*/
				virtual void setStrikethroughPosition(const StrikethroughPosition pos_);
				
				/**
					\brief	Sets the local underline position  attribute for the character style
					\author	GW
					\date	08/2014

					\param	pos_	The new underline position
				*/
				virtual void setUnderlinePosition(const UnderlinePosition pos_);
				
				/**
					\brief	Sets the local underline offset  attribute for the character style
					\author	GW
					\date	08/2014

					\param	offset_	The new underline offset value, in document points
				*/
				virtual void setUnderlineOffset(const double offset_);
				
				/**
					\brief	 Sets the local OpenType ligature attribute for the character style
					\author	GW
					\date	08/2014

					\param	lig_	The new local OpenType ligature attribute
				*/
				virtual void setLigature(const bool lig_);
				
				/**
					\brief	Sets the local OpenType discretionary ligatures attribute for the character style
					\author	GW
					\date	08/2014

					\param	lig_	true to turn discretionary ligatures on, false to turn it off
				*/
				virtual void setDiscretionaryLigatures(const bool lig_);
				
				/**
					\brief	Sets the local OpenType contextual ligatures attribute for the character style
					\author	GW
					\date	08/2014

					\param	lig_	true to turn contextual ligatures on, false to turn it off
				*/
				virtual void setContextualLigatures(const bool lig_);
				
				/**
					\brief	Sets the local OpenType alternate ligatures attribute for the character style
					\author	GW
					\date	08/2014

					\param	lig_	true to turn alternate ligatures on, false to turn it off
				*/
				virtual void setAlternateLigatures(const bool lig_);
				
				/**
					\brief	Sets the local OpenType old-style attribute for the character style
					\author	GW
					\date	08/2014

					\param	old_	true to turn old-style on, false to turn it off
				*/
				virtual void setOldStyle(const bool old_);
				
				/**
					\brief	Sets the local OpenType fractions attribute for the character style
					\author	GW
					\date	08/2014

					\param	frac_	true to turn fractions on, false to turn it off
				*/
				virtual void setFractions(const bool frac_);
				
				/**
					\brief	Sets the local OpenType ordinals attribute for the character style
					\author	GW
					\date	08/2014

					\param	ord_	true to turn ordinals on, false to turn it off
				*/
				virtual void setOrdinals(const bool ord_);
				
				/**
					\brief	Sets the local OpenType swash attribute for the character style
					\author	GW
					\date	08/2014

					\param	swash_	true to turn swash on, false to turn it off
				*/
				virtual void setSwash(const bool swash_);
				
				/**
					\brief	Sets the local OpenType titling attribute for the character style
					\author	GW
					\date	08/2014

					\param	titling_	true to turn titling on, false to turn it off
				*/
				virtual void setTitling(const bool titling_);
				
				/**
					\brief	Sets the local OpenType connection forms attribute for the character style
					\author	GW
					\date	08/2014

					\param	connForms_	true to turn connection forms on, false to turn it off
				*/
				virtual void setConnectionForms(const bool connForms_);
				
				/**
					\brief	Sets the local OpenType stylistic alternates attribute for the character style
					\author	GW
					\date	08/2014

					\param	styleAlt_	true to turn stylistic alternates on, false to turn it off
				*/
				virtual void setStylisticAlternates(const bool styleAlt_);

				/**
					\brief	Sets the local OpenType stylistic sets attribute for the character style
					\author	GW
					\date	07/2017

					\param	styleSets_	New stylistic sets value
				*/
				void setStylisticSets(const int32_t styleSets_);
				
				/**
					\brief	Sets the local OpenType ornaments attribute for the character style
					\author	GW
					\date	08/2014

					\param	orn_	true to turn ornaments on, false to turn it off
				*/
				virtual void setOrnaments(const bool orn_);
				
				/**
					\brief	Sets the local OpenType figure style attribute for the character style
					\author	GW
					\date	08/2014

					\param	style_	The new figure style
				*/
				virtual void setFigureStyle(const FigureStyle style_);
				
				/**
					\brief	Sets the local no-break attribute of the character
					\author	GW
					\date	08/2014

					\param	noBr_	true to turn no-break on, false to turn it off
				*/
				virtual void setNoBreak(const bool noBr_);
				
				/**
					\brief	Sets the local fill color attribute of the character
					\author	GW
					\date	08/2014

					\param	color_	The new fill color value
				*/
				virtual void setFillColor(const ArtColor& color_);
				
				/**
					\brief	Sets the local fill attribute of the character
					\author	GW
					\date	08/2014

					\param	fill_	true to turn fill on, false to turn it off
				*/
				virtual void setFill(const bool fill_);

				/**
					\brief	Sets the local fill visibility attribute of the character
					\author	GW
					\date	08/2014

					\param	fillVis_	true to turn fill visibility on, false to turn it off
				*/
				virtual void setFillVisible(const bool fillVis_);
				
				/**
					\brief	Sets the local fill overprint attribute of the character
					\author	GW
					\date	08/2014

					\param	overprint_	true to turn fill overprint on, false to turn it off
				*/
				virtual void setFillOverPrint(const bool overprint_);
				
				/**
					\brief	Sets the local stroke color attribute of the character
					\author	GW
					\date	08/2014

					\param	color_	The new stroke color value
				*/
				virtual void setStrokeColor(const ArtColor& color_);
				
				/**
					\brief	Sets the local stroke attribute of the character
					\author	GW
					\date	08/2014

					\param	stroke_	true to turn stroke on, false to turn it off
				*/
				virtual void setStroke(const bool stroke_);

				/**
					\brief	Sets the local stroke visibility attribute of the character
					\author	GW
					\date	08/2014

					\param	strokeVis_	true to turn stroke visibility on, false to turn it off
				*/
				virtual void setStrokeVisible(const bool strokeVis_);
				
				/**
					\brief	Sets the local stroke overprint attribute of the character
					\author	GW
					\date	08/2014

					\param	overprint_	true to turn stroke overprint on, false to turn it off
				*/
				virtual void setStrokeOverPrint(const bool overprint_);
				
				/**
					\brief	Sets the local fill-first attribute of the character
					\author	GW
					\date	08/2014

					\param	fillFirst_	true to turn fill-first on, false to turn it off
				*/
				virtual void setFillFirst(const bool fillFirst_);

				/**
					\brief	Sets the local fill background color attribute of the character
					\author	GW
					\date	08/2014

					\param	color_	The new fill color value
				*/
				virtual void setFillBackgroundColor(const ArtColor& color_);
				
				/**
					\brief	Sets the local fill background attribute of the character
					\author	GW
					\date	08/2014

					\param	fillBG_	true to turn fill on, false to turn it off
				*/
				virtual void setFillBackground(const bool fillBG_);
				
				/**
					\brief	Sets the local line cap attribute of the character
					\author	GW
					\date	08/2014

					\param	cap_	The new line cap type
				*/
				virtual void setLineCap(const StrokeStyle::CapStyle cap_);
				
				/**
					\brief	Sets the local line join attribute of the character
					\author	GW
					\date	08/2014

					\param	join_	The new line join type
				*/
				virtual void setLineJoin(const StrokeStyle::JoinStyle join_);
				
				/**
					\brief	Sets the local line width attribute of the character
					\author	GW
					\date	08/2014

					\param	width_	The new line width value in document points
				*/
				virtual void setLineWidth(const double width_);
				
				/**
					\brief	Sets the local miter limit attribute of the character
					\author	GW
					\date	08/2014

					\param	limit_	The new miter limit value in document points
				*/
				virtual void setMiterLimit(const double limit_);
				
				/**
					\brief	Sets the local line dash offset attribute of the character
					\author	GW
					\date	08/2014

					\param	offset_	The new line dash offset value in document points
				*/
				virtual void setLineDashOffset(const double offset_);
				
				/**
					\brief	Sets the local line dash array attribute of the character
					\author	GW
					\date	08/2014

					\param	dists_	The new line dash array (as a vector)
				*/
				virtual void setLineDashArray(const DoubleVector& dists_);

				/**
					\brief	Sets the language attribute of the character
					\author	GW
					\date	08/2014
					
					\param	lang_	New language for the character
				*/
				virtual void setLanguage(const Language lang_);
				
				/**
					\brief	Removes the local font attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearFont();
				
				/**
					\brief	Removes the local font size attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearFontSize();
				
				/**
					\brief	Removes the local horizontal scale attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearHorizontalScale();
				
				/**
					\brief	Removes the local vertical scale attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearVerticalScale();
				
				/**
					\brief	Removes the local automatic leading attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearAutoLeading();
				
				/**
					\brief	Removes the local leading attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearLeading();
				
				/**
					\brief	Removes the local tracking attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearTracking();
				
				/**
					\brief	Removes the local baseline shift attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearBaselineShift();
				
				/**
					\brief	Removes the local character rotation attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearCharacterRotation();
				
				/**
					\brief	Removes the local automatic kerning attribute value of this character style. This is not
							available for individual characters.
					\author	GW
					\date	08/2014
				*/
				virtual void clearAutoKernType();
				
				/**
					\brief	Removes the local font caps option attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearFontCapsOption();
				
				/**
					\brief	Removes the local font baseline option attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearFontBaselineOption();
				
				/**
					\brief	Removes the local font OpenType position option attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearFontOpenTypePositionOption();
				
				/**
					\brief	Removes the local strikethrough position attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearStrikethroughPosition();
				
				/**
					\brief	Removes the local underline position attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearUnderlinePosition();
				
				/**
					\brief	Removes the local underline offset attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearUnderlineOffset();
				
				/**
					\brief	Removes the local OpenType ligature attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearLigature();
				
				/**
					\brief	Removes the local OpenType discretionary ligature attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearDiscretionaryLigatures();
				
				/**
					\brief	Removes the local OpenType contextual ligature attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearContextualLigatures();
				
				/**
					\brief	Removes the local OpenType alternate ligature attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearAlternateLigatures();
				
				/**
					\brief	Removes the local OpenType old-style attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearOldStyle();
				
				/**
					\brief	Removes the local OpenType fractions attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearFractions();
				
				/**
					\brief	Removes the local OpenType ordinals attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearOrdinals();
				
				/**
					\brief	Removes the local OpenType swash attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearSwash();
				
				/**
					\brief	Removes the local OpenType titling attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearTitling();
				
				/**
					\brief	Removes the local OpenType connection forms attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearConnectionForms();
				
				/**
					\brief	Removes the local OpenType stylistic alternates attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearStylisticAlternates();

				/**
					\brief	Removes the local OpenType stylistic sets attribute value of this character
					\author	GW
					\date	07/2017
				*/
				void clearStylisticSets();
				
				/**
					\brief	Removes the local OpenType ornaments attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearOrnaments();
				
				/**
					\brief	Removes the local OpenType figure style attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearFigureStyle();
				
				/**
					\brief	Removes the local no-break attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearNoBreak();
				
				/**
					\brief	Removes the local fill color attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearFillColor();
				
				/**
					\brief	Removes the local stroke color attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearStrokeColor();
				
				/**
					\brief	Removes the local fill attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearFill();
			
				/**
					\brief	Removes the local fill visibility attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearFillVisible();
				
				/**
					\brief	Removes the local stroke attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearStroke();
			
				/**
					\brief	Removes the local stroke visibility attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearStrokeVisible();
				
				/**
					\brief	Removes the local fill-first attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearFillFirst();
				
				/**
					\brief	Removes the local fill overprint attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearFillOverPrint();
				
				/**
					\brief	Removes the local stroke overprint attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearStrokeOverPrint();
			
				/**
					\brief	Removes the local fill background color attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearFillBackgroundColor();
				
				/**
					\brief	Removes the local fill background attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearFillBackground();
				
				/**
					\brief	Removes the local line cap attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearLineCap();
				
				/**
					\brief	Removes the local line join attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearLineJoin();
				
				/**
					\brief	Removes the local line width attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearLineWidth();
				
				/**
					\brief	Removes the local miter limit attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearMiterLimit();
				
				/**
					\brief	Removes the local line dash offset attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearLineDashOffset();
				
				/**
					\brief	Removes the local line dash array attribute value of this character
					\author	GW
					\date	08/2014
				*/
				virtual void clearLineDashArray();

				/**
					\brief	Removes the language attribute of the character
					\author	GW
					\date	08/2014
				*/
				virtual void clearLanguage();


			private:
				friend aip::TextCharFeatures* __accessImpl(const TextCharFeatures&);
				friend TextCharFeatures __accessCtor(const aip::TextCharFeatures&);

				/**
					\brief	Private implementation object
				*/
				aip::TextCharFeatures* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				TextCharFeatures(const aip::TextCharFeatures&);
		};
		
		typedef std::unique_ptr<TextCharFeatures> TextCharFeaturesUP;
		typedef std::shared_ptr<TextCharFeatures> TextCharFeaturesSP;
		typedef std::weak_ptr<TextCharFeatures> TextCharFeaturesWP;
		
		extern aip::TextCharFeatures* __accessImpl(const TextCharFeatures&);
		extern TextCharFeatures __accessCtor(const aip::TextCharFeatures&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_CHAR_FEATURES__
