/**
	\file
	\brief		Header file for Illustrator text character inspection
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_TEXT_CHAR_INSPECTOR__
#define __HDI_CORE_TEXT_CHAR_INSPECTOR__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreArtColor.h"
#include "hdicorePathStyle.h"
#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextCharInspector;
	}

	namespace core
	{
		class Font;
	
		/**
			\brief	Allows for easy inspection of Illustrator text characters
		*/
		class TextCharInspector
		{
			public:
				typedef std::vector<ArtColor> ArtColorVector;
				typedef std::vector<bool> BoolVector;
				typedef std::vector<double> DoubleVector;
				typedef std::vector< std::shared_ptr<Font> > FontVector;
				typedef std::vector<int32_t> IntVector;
				
				typedef std::vector<StrokeStyle::CapStyle> CapStyleVector;
				typedef std::vector<FigureStyle> FigureStyleVector;
				typedef std::vector<FontCapsType> FontCapsTypeVector;
				typedef std::vector<FontBaseline> FontBaselineVector;
				typedef std::vector<FontOpenTypePosition> FontOpenTypePositionVector;
				typedef std::vector<StrokeStyle::JoinStyle> JoinStyleVector;
				typedef std::vector<StrikethroughPosition> StrikethroughPositionVector;
				typedef std::vector<UnderlinePosition> UnderlinePositionVector;
			
				enum CollapseDirection
				{
					UnknownCollapseDir	= 0,
					CollapseEndDir		= 10,
					CollapseStartDir	= 20
				};

				/**
					\brief	Creates a new TextCharInspector object
					\author	GW
					\date	12/2013
					
					\note	New TextCharInspector objects do not relate to any text characters on the Illustrator
							artboard; they are designed to be "receivers" of some other TextCharInspector object via the
							overloaded assignment operator or assigned to some other existent text art.
				*/
				TextCharInspector();
			
				/**
					\brief	TextCharInspector copy constructor
					\author	GW
					\date	12/2013
					
					\param	ci_	TextCharInspector object to copy values from
				*/
				TextCharInspector(const TextCharInspector& ci_);

				/**
					\brief	Destructs a TextCharInspector object
					\author	GW
					\date	12/2013
				*/
				virtual ~TextCharInspector();
			
				/**
					\brief		Overloaded assignment operator for TextCharInspector objects
					\author		GW
					\date		12/2013
					
					\param		rhs_	Existing TextCharInspector object to copy values from
					\returns	The target TextCharInspector object, but with its values updated to match that of rhs_
				*/
				virtual TextCharInspector& operator=(const TextCharInspector& rhs_);
				
				/**
					\brief		Tests whether a given TextCharInspector object is the same as another
					\author		GW
					\date		12/2013

					\param		rhs_	TextCharInspector to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text char inspector, false otherwise
				*/
				virtual bool operator==(const TextCharInspector& rhs_) const;
				
				/**
					\brief		Tests whether a given TextCharInspector object is not the same as another
					\author		GW
					\date		12/2013

					\param		rhs_	TextCharInspector to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text char inspectors, false otherwise
				*/
				virtual bool operator!=(const TextCharInspector& rhs_) const;

				/**
					\brief		Gets the ATE char inspector ref around which the target object is wrapped
					\author		GW
					\date		12/2013
					
					\returns	ATE::ICharInspector for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::ICharInspector> ateCharInspector() const;

				/**
					\brief		Gets whether the target TextCharInspector object is empty
					\author		GW
					\date		12/2013
					
					\returns	true if the target TextCharInspector object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Retrieves the fonts from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of fonts for each character being inspected
				*/
				virtual FontVector font() const;

				/**
					\brief		Retrieves the font size values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of font sizes for each character being inspected
				*/
				virtual DoubleVector fontSize() const;
				
				/**
					\brief		Retrieves the horizontal scale values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of horizontal scales for each character being inspected
				*/
				virtual DoubleVector horizontalScale() const;
				
				/**
					\brief		Retrieves the vertical scale values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of vertical scales for each character being inspected
				*/
				virtual DoubleVector verticalScale() const;
				
				/**
					\brief		Retrieves the synthetic bold values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of synthetic bold settings for each character being inspected
				*/
				virtual BoolVector syntheticBold() const;
				
				/**
					\brief		Retrieves the synthetic italic values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of synthetic italic settings for each character being inspected
				*/
				virtual BoolVector syntheticItalic() const;
				
				/**
					\brief		Retrieves the auto-leading values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of auto-leading settings for each character being inspected
				*/
				virtual BoolVector autoLeading() const;
				
				/**
					\brief		Retrieves the leading values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of leading values for each character being inspected
				*/
				virtual DoubleVector leading() const;
				
				/**
					\brief		Retrieves the tracking values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of tracking values for each character being inspected
				*/
				virtual IntVector tracking() const;
				
				/**
					\brief		Retrieves the baseline shift values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of baseline shifts for each character being inspected
				*/
				virtual DoubleVector baselineShift() const;
				
				/**
					\brief		Retrieves the char rotation values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of char rotations for each character being inspected
				*/
				virtual DoubleVector characterRotation() const;
				
				/**
					\brief		Retrieves the font cap values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of font caps for each character being inspected
				*/
				virtual FontCapsTypeVector fontCaps() const;
				
				/**
					\brief		Retrieves the font baseline values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of font baselines for each character being inspected
				*/
				virtual FontBaselineVector fontBaseline() const;
				
				/**
					\brief		Retrieves the font position values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of font positions for each character being inspected
				*/
				virtual FontOpenTypePositionVector fontOpenTypePosition() const;
				
				/**
					\brief		Retrieves the strikethrough position values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of strikethrough positions for each character being inspected
				*/
				virtual StrikethroughPositionVector strikethroughPosition() const;
				
				/**
					\brief		Retrieves the underline position values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of underline positions for each character being inspected
				*/
				virtual UnderlinePositionVector underlinePosition() const;
				
				/**
					\brief		Retrieves the underline offset values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of underline offsets for each character being inspected
				*/
				virtual DoubleVector underlineOffset() const;
				
				/**
					\brief		Retrieves the ligature values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of ligature settings for each character being inspected
				*/
				virtual BoolVector ligature() const;
				
				/**
					\brief		Retrieves the discretionary ligature values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of discretionary ligature settings for each character being inspected
				*/
				virtual BoolVector discretionaryLigatures() const;
				
				/**
					\brief		Retrieves the contextual ligature values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of contextual ligature settings for each character being inspected
				*/
				virtual BoolVector contextualLigatures() const;
				
				/**
					\brief		Retrieves the alternate ligature values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of alternate ligature settings for each character being inspected
				*/
				virtual BoolVector alternateLigatures() const;
				
				/**
					\brief		Retrieves the old-style values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of old-style settings for each character being inspected
				*/
				virtual BoolVector oldStyle() const;
				
				/**
					\brief		Retrieves the fraction values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of fraction settings for each character being inspected
				*/
				virtual BoolVector fractions() const;
				
				/**
					\brief		Retrieves the ordinal values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of ordinal settings for each character being inspected
				*/
				virtual BoolVector ordinals() const;
				
				/**
					\brief		Retrieves the swash values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of swash settings for each character being inspected
				*/
				virtual BoolVector swash() const;
				
				/**
					\brief		Retrieves the titling values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of titling settings for each character being inspected
				*/
				virtual BoolVector titling() const;
				
				/**
					\brief		Retrieves the connection form values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of connection form settings for each character being inspected
				*/
				virtual BoolVector connectionForms() const;
				
				/**
					\brief		Retrieves the style alternative values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of style alternative settings for each character being inspected
				*/
				virtual BoolVector stylisticAlternates() const;
			
				/**
					\brief		Retrieves the style sets values from the character set
					\author		GW
					\date		07/2017
					
					\returns	A vector of style sets values for each character being inspected
				*/
				IntVector stylisticSets() const;
				
				/**
					\brief		Retrieves the ornament values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of ornaments for each character being inspected
				*/
				virtual BoolVector ornaments() const;
				
				/**
					\brief		Retrieves the figure style values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of figure styles for each character being inspected
				*/
				virtual FigureStyleVector figureStyle() const;
				
				/**
					\brief		Retrieves the no-break values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of no-break settings for each character being inspected
				*/
				virtual BoolVector noBreak() const;
				
				/**
					\brief		Retrieves the fill values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of fill settings for each character being inspected
				*/
				virtual BoolVector fill() const;
				
				/**
					\brief		Retrieves the fill visibility values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of fill visibility settings for each character being inspected
				*/
				virtual BoolVector fillVisible() const;
				
				/**
					\brief		Retrieves the fill color values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of fill colors for each character being inspected
				*/
				virtual ArtColorVector fillColor() const;
				
				/**
					\brief		Retrieves the fill-first values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of fill-first settings for each character being inspected
				*/
				virtual BoolVector fillFirst() const;
				
				/**
					\brief		Retrieves the fill overprint values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of fill overprint settings for each character being inspected
				*/
				virtual BoolVector fillOverPrint() const;
			
				/**
					\brief		Retrieves the background fill values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of background fills for each character being inspected
				*/
				virtual BoolVector fillBackground() const;

				/**
					\brief		Retrieves the background fill color values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of background fill colors for each character being inspected
				*/
				virtual ArtColorVector fillBackgroundColor() const;
				
				/**
					\brief		Retrieves the stroke values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of stroke settings for each character being inspected
				*/
				virtual BoolVector stroke() const;
				
				/**
					\brief		Retrieves the stroke visibility values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of stroke visibility settings for each character being inspected
				*/
				virtual BoolVector strokeVisible() const;
				
				/**
					\brief		Retrieves the stroke color values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of stroke colors for each character being inspected
				*/
				virtual ArtColorVector strokeColor() const;
				
				/**
					\brief		Retrieves the stroke overprint values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of stroke overprint settings for each character being inspected
				*/
				virtual BoolVector strokeOverPrint() const;
				
				/**
					\brief		Retrieves the line cap values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of line caps for each character being inspected
				*/
				virtual CapStyleVector lineCap() const;
				
				/**
					\brief		Retrieves the line join values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of line joins for each character being inspected
				*/
				virtual JoinStyleVector lineJoin() const;
				
				/**
					\brief		Retrieves the line width values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of line widths for each character being inspected
				*/
				virtual DoubleVector lineWidth() const;
				
				/**
					\brief		Retrieves the miter limit values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of miter limits for each character being inspected
				*/
				virtual DoubleVector miterLimit() const;
				
				/**
					\brief		Retrieves the line dash offset values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of line dash offsets for each character being inspected
				*/
				virtual DoubleVector lineDashOffset() const;
				
				/**
					\brief		Retrieves the line dash values from the character set
					\author		GW
					\date		12/2013
					
					\returns	A vector of vectors of line dashes for each character being inspected
				*/
				virtual std::vector<DoubleVector> lineDashArray() const;


			private:
				friend aip::TextCharInspector* __accessImpl(const TextCharInspector&);
				friend TextCharInspector __accessCtor(const aip::TextCharInspector&);

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	12/2013
				*/
				void* __impl() const;

				/**
					\brief	Internal use only
					\author	GW
					\date	12/2013
				*/
				TextCharInspector(const aip::TextCharInspector&);
		};
		
		typedef std::unique_ptr<TextCharInspector> TextCharInspectorUP;
		typedef std::shared_ptr<TextCharInspector> TextCharInspectorSP;
		typedef std::weak_ptr<TextCharInspector> TextCharInspectorWP;
		
		extern aip::TextCharInspector* __accessImpl(const TextCharInspector&);
		extern TextCharInspector __accessCtor(const aip::TextCharInspector&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_CHAR_INSPECTOR__
