/**
	\file
	\brief		Header file for general Illustrator character styles manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_TEXT_CHAR_STYLE__
#define __HDI_CORE_TEXT_CHAR_STYLE__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextCharStyle;
	}
	
	namespace core
	{
		class TextCharFeatures;
	
		/**
			\brief	Encapsulates character style functionality from the Illustrator SDK
		*/
		class TextCharStyle
		{
			public:
				/**
					\brief	Creates a new TextCharStyle object
					\author	GW
					\date	08/2014
					
					\note	New TextCharStyle objects do not relate to any text char style on the Illustrator artboard;
							they are designed to be "receivers" of some other TextCharStyle object via the overloaded
							assignment operator or assigned to some other existent text art.
				*/
				TextCharStyle();
			
				/**
					\brief	TextCharStyle copy constructor
					\author	GW
					\date	08/2014
					
					\param	cs_	TextCharStyle object to copy values from
				*/
				TextCharStyle(const TextCharStyle& cs_);

				/**
					\brief	Destructs a TextCharStyle object
					\author	GW
					\date	08/2014
				*/
				virtual ~TextCharStyle();
			
				/**
					\brief		Overloaded assignment operator for TextCharStyle objects
					\author		GW
					\date		08/2014
					
					\param		rhs_	Existing TextCharStyle object to copy values from
					\returns	The target TextCharStyle object, but with its values updated to match that of rhs_
				*/
				virtual TextCharStyle& operator=(const TextCharStyle& rhs_);
				
				/**
					\brief		Tests whether a given TextCharStyle object is the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextCharStyle to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text char style, false otherwise
				*/
				virtual bool operator==(const TextCharStyle& rhs_) const;
				
				/**
					\brief		Tests whether a given TextCharStyle object is not the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextCharStyle to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text char styles, false otherwise
				*/
				virtual bool operator!=(const TextCharStyle& rhs_) const;

				/**
					\brief		Gets the ATE char style ref around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	ATE::ICharStyleRef for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::ICharStyle> ateCharStyle() const;

				/**
					\brief		Gets whether the target TextCharStyle object is empty
					\author		GW
					\date		08/2014
					
					\returns	true if the target TextCharStyle object is empty, false otherwise
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Gets the name of the char style
					\author		GW
					\date		08/2014
					
					\returns	The char style name, as UTF-8
				*/
				virtual std::string name() const;

				/**
					\brief		Sets the name of the char style
					\author		GW
					\date		08/2014
					
					\param		name_	New char style name, as UTF-8
					\returns	true if the name could be set, false otherwise
				*/
				virtual bool setName(const std::string& name_);

				/**
					\brief		Gets the parent style of the target char style object
					\author		GW
					\date		08/2014
					
					\returns	The TextCharStyle object for the parent, or NULL for none
				*/
				virtual std::unique_ptr<TextCharStyle> parent() const;

				/**
					\brief		Sets the parent style of the target char style object
					\author		GW
					\date		08/2014
					
					\param		parent_		New TextCharStyle parent
					\returns	true if the parent could be set, false otherwise
				*/
				virtual bool setParent(const TextCharStyle& parent_);

				/**
					\brief		Gets whether the target object has a parent
					\author		GW
					\date		08/2014
					
					\returns	true if a parent exists, false otherwise
				*/
				virtual bool hasParent() const;

				/**
					\brief		Gets the features defined for the target style
					\author		GW
					\date		08/2014
					
					\returns	The TextCharFeatures object for the style, or NULL for error
				*/
				virtual std::unique_ptr<TextCharFeatures> features() const;

				/**
					\brief	Sets the features defined for the target style
					\author	GW
					\date	08/2014
					
					\param	feat_	New TextCharFeatures object for the style
				*/
				virtual void setFeatures(const TextCharFeatures& feat_);

				/**
					\brief	Replaces existing features in the target style, or adds them anew if they were not already
							present (values that are unassigned in the replacement set remain unchanged in the target 
							style).
					\author	GW
					\date	08/2014
					
					\param	feat_	New/replacement TextCharFeatures object for the style
				*/
				virtual void replaceOrAddFeatures(const TextCharFeatures& feat_);


			private:
				friend aip::TextCharStyle* __accessImpl(const TextCharStyle&);
				friend TextCharStyle __accessCtor(const aip::TextCharStyle&);

				/**
					\brief	Private implementation object
				*/
				aip::TextCharStyle* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				TextCharStyle(const aip::TextCharStyle&);
		};
		
		typedef std::unique_ptr<TextCharStyle> TextCharStyleUP;
		typedef std::shared_ptr<TextCharStyle> TextCharStyleSP;
		typedef std::weak_ptr<TextCharStyle> TextCharStyleWP;
		
		extern aip::TextCharStyle* __accessImpl(const TextCharStyle&);
		extern TextCharStyle __accessCtor(const aip::TextCharStyle&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_CHAR_STYLE__
