/**
	\file
	\brief		Header file for plugin user interface text field widgets
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_TEXTFIELD__
#define __HDI_CORE_TEXTFIELD__

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		class Label;

		/**
			\brief	Text field widget, which allows the user to enter arbitrary text values
		*/
		class TextField : public Widget
		{
			public:
				friend class Label;
			
				/**
					\brief	Text alignment options
				*/
				enum Alignment
				{
					UnknownAlignment	= 0,
					LeftAlignment		= 10,
					CenterAlignment		= 20,
					RightAlignment		= 30
				};

				/**
					\brief	Constructs an empty TextField object
					\author	GW
					\date	09/2013
					
					\note	To test if a TextField object is empty, call isEmpty() on it
					\note	Empty TextField objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other TextField object via the overloaded assignment operator. Empty
							TextField objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				TextField();

				/**
					\brief	Constructs a new TextField object from an existing TextField object (copy constructor)
					\author	GW
					\date	09/2013

					\param	field_	Existing TextField object
				*/
				TextField(const TextField& field_);

				/**
					\brief	Constructs an editable text field (single line only)
					\author	GW
					\date	09/2013

					\param	loc_	Position of the field, in 1x resolution coordinates
					\param	text_	Default text in the field, as UTF-8
					\param	width_	Width of the field, at 1x resolution 
				*/
				TextField(const Point& loc_, const std::string& text_, const double width_);
				
				/**
					\brief	TextField destructor
					\author	GW
					\date	09/2013
				*/
				virtual ~TextField();

				/**
					\brief		Allows one TextField object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target TextField object, but with its value updated to match that of rhs_
				*/
				virtual TextField& operator=(const TextField& rhs_);
				
				/**
					\brief		Convenience method to clone a TextField object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new TextField object
					
					\note		If you subclass TextField, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your TextField subclass, you will
								experience "object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned TextField object.
				*/
				virtual TextField* clone() const;
				
				/**
					\brief		Convenience method to duplicate a TextField object, creating a new and identical UI
								element to the target (but not belonging to the same containing window)
					\author		GW
					\date		11/2013
					
					\returns	A pointer to the new TextField object (and new UI element)
					
					\warning	The caller must manage the memory for the returned TextField object.
				*/
				virtual TextField* duplicate() const;

				/**
					\brief		Gets the current alignment of the text field
					\author		GW
					\date		09/2013

					\returns	Current text alignment
				*/
				virtual Alignment alignment() const;

				/**
					\brief	Sets the current alignment of the text field
					\author	GW
					\date	09/2013

					\param	alignment_	New text alignment
				*/
				virtual void setAlignment(const Alignment alignment_);
			
				/**
					\brief		Gets whether the text is bold
					\author		GW
					\date		09/2013

					\returns	true if the text is bold, false otherwise
				*/
				virtual bool bold() const;

				/**
					\brief	Sets the font weight
					\author	GW
					\date	09/2013

					\param	bold_	true for bold, false otherwise
				*/
				virtual void setBold(const bool bold_);

				/**
					\brief		Gets whether the text is italicized
					\author		GW
					\date		09/2013

					\returns	true if the text is italicized, false otherwise
				*/
				virtual bool italic() const;

				/**
					\brief	Sets the font style
					\author	GW
					\date	09/2013

					\param	italic_		true for italics, false otherwise
				*/
				virtual void setItalic(const bool italic_);
				
				/**
					\brief		If the target text field has been associated with a label, this gets said label
					\author		GW
					\date		04/2014
					
					\returns	The label associated with the text field, or NULL for none
				*/
				virtual std::unique_ptr<Label> label() const;
				
				/**
					\brief	Sets the label that, when clicked, will focus the target text field
					\author	GW
					\date	04/2014
					
					\param	label_	New label for focusing the target text field
					
					\note	If the label was previously linked with a text view, this will remove said link.
				*/
				virtual void setLabel(Label& label_);
				
				/**
					\brief	Removes the label that, when clicked, would have focused the target text field
					\author	GW
					\date	04/2014
				*/
				virtual void removeLabel();

				/**
					\brief		Gets the selection range of the text in the field, if any, in UTF-8
					\author		GW
					\date		03/2014
				 
					\param		pos__	Return-by-reference for the starting position of the selection range
					\param		len__	Return-by-reference for the length of the selection range
					\returns	true if the field is focused and the selection range could be acquired, false otherwise
				 
				 	\warning	This method utilizes UTF-8 byte ranges, which can be useful in certain circumstances. If
				 				you require character ranges instead, use the UTF-32 version of this method.
				 
					\note		If len__ is zero then no text is selected, but pos__ indicates the position of the
								insertion point (text caret).
				*/
				virtual bool selectionRange(uint32_t& pos__, uint32_t& len__) const;
			
				/**
					\brief		Sets the selection range of the text in the field, in UTF-8 (and focuses it)
					\author		GW
					\date		03/2014
				 
					\param		pos_	The starting position of the selection range
					\param		len_	The length of the selection range
				 
				 	\warning	This method utilizes UTF-8 byte ranges, which can be useful in certain circumstances. If
				 				you require character ranges instead, use the UTF-32 version of this method.
				 
					\note		If len_ is zero then no text will be selected, but the insertion point (text caret) will be
								moved to the pos_ value.
				*/
				virtual void setSelectionRange(const uint32_t pos_, const uint32_t len_);

				/**
					\brief		Gets the selection range of the text in the field, if any, in UTF-32
					\author		GW
					\date		03/2014
				 
					\param		pos__	Return-by-reference for the starting position of the selection range
					\param		len__	Return-by-reference for the length of the selection range
					\returns	true if the field is focused and the selection range could be acquired, false otherwise
				 
				 	\warning	This method utilizes UTF-32 character ranges, which can be useful in certain circumstances.
				 				If you require byte ranges instead, use the UTF-8 version of this method.
				 
					\note		If len__ is zero then no text is selected, but pos__ indicates the position of the
								insertion point (text caret).
				*/
				virtual bool selectionRangeUTF32(uint32_t& pos__, uint32_t& len__) const;
			
				/**
					\brief		Sets the selection range of the text in the field, in UTF-32 (and focuses it)
					\author		GW
					\date		03/2014
				 
					\param		pos_	The starting position of the selection range
					\param		len_	The length of the selection range
				 
				 	\warning	This method utilizes UTF-32 character ranges, which can be useful in certain circumstances.
				 				If you require byte ranges instead, use the UTF-8 version of this method.
				 
					\note		If len_ is zero then no text will be selected, but the insertion point (text caret) will be
								moved to the pos_ value.
				*/
				virtual void setSelectionRangeUTF32(const uint32_t pos_, const uint32_t len_);

				/**
					\brief		Gets the last character typed into the field
					\author		GW
					\date		09/2013

					\returns	Last character typed, or "" if none

					\note		The return value of this method is only valid when called from inside a value-changing
								or value-changed callback (and it's very useful to determine if the user pressed e.g. the
								enter or return key)
				*/
				virtual std::string lastCharTyped() const;

				/**
					\brief		Gets the last set of modifier keys used when typing into the field
					\author		GW
					\date		05/2023

					\returns	Last set of modifier keys, or NoModifierKey for none

					\note		The return value of this method is only valid when called from inside a value-changing
								or value-changed callback.
					\note		Some modifiers are difficult/impossible to detect here, such as ControlModifierKey on
								Windows. The reason is that such modifiers are usually captured by other parts of the UI
								event loop before the widget is even notified of the key strokes (picture e.g. ctrl+a on
								the Windows to "select all").
				*/
				virtual ModifierKey lastModifierKeys() const;

				/**
					\brief	Simulates the value of the text field changing
					\author	GW
					\date	09/2013
				*/
				virtual void valueChanging();
				
				/**
					\brief		Gets the value changing callback for the text field
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered value changing callback
				*/
				virtual Callback* const valueChangingCallback() const;
				
				/**
					\brief	Sets the value changing callback
					\author	GW
					\date	09/2013

					\param	callback_	New callback for when the text field's value is changing
				*/
				virtual void setValueChangingCallback(const Callback& callback_);

				/**
					\brief	Simulates the text field having its value changed
					\author	GW
					\date	09/2013
				*/
				virtual void valueChanged();
				
				/**
					\brief		Gets the value changed callback for the text field
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered value changed callback
				*/
				virtual Callback* const valueChangedCallback() const;
				
				/**
					\brief	Sets the value changed callback
					\author	GW
					\date	09/2013

					\param	callback_	New callback for when the text field's value has changed
				*/
				virtual void setValueChangedCallback(const Callback& callback_);
			
				/**
					\brief		Gets whether dropping support for this particular widget is enabled
					\author		GW
					\date		09/2018
				 
					\returns	true if drag-and-drop destination support is enabled, false otherwise
				*/
				virtual bool droppable() const;
			
				/**
					\brief	Sets whether dropping support for this particular widget is enabled
					\author	GW
					\date	09/2018
				 
					\param	enable_		true to enable drag-and-drop destination support, false otherwise
				*/
				virtual void setDroppable(const bool enable_);
			
				/**
					\brief		Gets the current drop callback for the widget
					\author		GW
					\date		09/2018

					\returns	A pointer to the currently registered drop callback
				*/
				virtual Callback* const dropCallback() const;
			
				/**
					\brief	Sets the drop callback for the widget
					\author	GW
					\date	09/2018

					\param	callback_	New callback for when the widget receives a drop
				*/
				virtual void setDropCallback(const Callback& callback_);
			
				/**
					\brief		Gets the data that was passed when the widget received a drop
					\author		GW
					\date		09/2018
				 
					\returns	The dropped data
				 
					\warning	The result of this method is only valid when it is called from within a drop callback.
				*/
				virtual std::string droppedData() const;
			
				/**
					\brief		Gets the point at which the drop occurred, in the widget's coordinate system
					\author		GW
					\date		09/2018
				 
					\returns	The point at which the drop occurred
				 
					\warning	The result of this method is only valid when it is called from within a drop callback.
				*/
				virtual Point droppedPoint() const;
		};
		
		typedef std::unique_ptr<TextField> TextFieldUP;
		typedef std::shared_ptr<TextField> TextFieldSP;
		typedef std::weak_ptr<TextField> TextFieldWP;
	}
}

#endif
// __HDI_CORE_TEXTFIELD__
