/**
	\file
	\brief		Header file for general Illustrator paragraph manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_TEXT_PARA__
#define __HDI_CORE_TEXT_PARA__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextPara;
	}
	
	namespace core
	{
		class TextRange;
		class TextRangeSet;
		class TextStory;
		class TextWord;
	
		/**
			\brief	Encapsulates text paragraph functionality from the Illustrator SDK
		*/
		class TextPara
		{
			public:
				typedef std::vector< std::shared_ptr<TextWord> > TextWordVector;
			
				/**
					\brief	Creates a new TextPara object
					\author	GW
					\date	08/2014
					
					\note	New TextPara objects do not relate to any text paragraph on the Illustrator artboard;
							they are designed to be "receivers" of some other TextPara object via the overloaded
							assignment operator or assigned to some other existent text art.
				*/
				TextPara();
			
				/**
					\brief	TextPara copy constructor
					\author	GW
					\date	08/2014
					
					\param	p_	TextPara object to copy values from
				*/
				TextPara(const TextPara& p_);

				/**
					\brief	Destructs a TextPara object
					\author	GW
					\date	08/2014
				*/
				virtual ~TextPara();
			
				/**
					\brief		Overloaded assignment operator for TextPara objects
					\author		GW
					\date		08/2014
					
					\param		rhs_	Existing TextPara object to copy values from
					\returns	The target TextPara object, but with its values updated to match that of rhs_
				*/
				virtual TextPara& operator=(const TextPara& rhs_);
				
				/**
					\brief		Tests whether a given TextPara object is the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextPara to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text paragraph, false otherwise
				*/
				virtual bool operator==(const TextPara& rhs_) const;
				
				/**
					\brief		Tests whether a given TextPara object is not the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextPara to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text paragraphs, false otherwise
				*/
				virtual bool operator!=(const TextPara& rhs_) const;

				/**
					\brief		Gets the ATE paragraph ref around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	ATE::IParagraphRef for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::IParagraph> ateParagraph() const;

				/**
					\brief		Gets whether the target TextPara object is empty
					\author		GW
					\date		08/2014
					
					\returns	true if the target TextPara object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Retrieves the length of this paragraph
					\author		GW
					\date		08/2014
					
					\returns	The number of characters in this paragraph, or -1 for error
				*/
				virtual int32_t length() const;

				/**
					\brief		Retrieves the story that contains this paragraph
					\author		GW
					\date		08/2014
					
					\returns	The story object
				*/
				virtual std::unique_ptr<TextStory> story() const;

				/**
					\brief		Retrieves the text range corresponding to this paragraph
					\author		GW
					\date		08/2014
					
					\returns	The text range object
				*/
				virtual std::unique_ptr<TextRange> textRange() const;

				/**
					\brief		Retrieves the set of text ranges corresponding to this paragraph
					\author		GW
					\date		08/2014
					
					\returns	The text ranges object
				*/
				virtual std::unique_ptr<TextRangeSet> textRangeSet() const;

				/**
					\brief		Retrieves the text selection in this paragraph
					\author		GW
					\date		08/2014
					
					\returns	The text range set object
				*/
				virtual std::unique_ptr<TextRangeSet> selections() const;

				/**
					\brief		Retrieves a vector of the words in this paragraph
					\author		GW
					\date		08/2014
					
					\returns	The words vector object
				*/
				virtual TextWordVector words() const;

				/**
					\brief		Gets the value (contents) of the text range
					\author		GW
					\date		08/2014
					
					\param		le_		Indicates the desired line ending for the output string
					\returns	A string for the value of the text art, as UTF-8
					
					\note		Illustrator forces text art to utilize the "\r" character for line endings; if this
								behavior is unexpected or undesired for your application (as most systems use either
								"\n" or "\r\n" these days), you can specify what you'd prefer with the optional le_
								argument.
				*/
				virtual std::string contents(const std::string& le_ = "\n");


			private:
				friend aip::TextPara* __accessImpl(const TextPara&);
				friend TextPara __accessCtor(const aip::TextPara&);

				/**
					\brief	Private implementation object
				*/
				aip::TextPara* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				TextPara(const aip::TextPara&);
		};
		
		typedef std::unique_ptr<TextPara> TextParaUP;
		typedef std::shared_ptr<TextPara> TextParaSP;
		typedef std::weak_ptr<TextPara> TextParaWP;
		
		extern aip::TextPara* __accessImpl(const TextPara&);
		extern TextPara __accessCtor(const aip::TextPara&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_PARA__
