/**
	\file
	\brief		Header file for general Illustrator paragraph style set manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_TEXT_PARA_STYLE_SET__
#define __HDI_CORE_TEXT_PARA_STYLE_SET__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextParaStyleSet;
	}
	
	namespace core
	{
		class TextParaStyle;
		class TextRange;
	
		/**
			\brief	Encapsulates paragraph style set functionality from the Illustrator SDK
		*/
		class TextParaStyleSet
		{
			public:
				/**
					\brief	Allows for iteration through the children of a paragraph style set, a la std::vector iteration

					\code
						// Here's an example, iterating through the children of someSet:
						for(hdi::core::TextParaStyleSet::Iterator iter = someSet.begin(); !iter.atEnd(); ++iter)
						{
							// do stuff to iter.style()
						}
					\endcode
				*/
				class Iterator
				{
					public:
						/**
							\brief	Constructs an Iterator object from an existing Iterator object (copy constructor)
							\author	GW
							\date	08/2014

							\param	iter_	Existing Iterator object
						*/
						Iterator(const Iterator& iter_);
					
						/**
							\brief	Destructs an Iterator object
							\author	GW
							\date	08/2014
						*/
						~Iterator();
						
						/**
							\brief		Assigns one Iterator object to another
							\author		GW
							\date		08/2014

							\param		iter_	Existing Iterator object to copy values from
							\returns	The target Iterator object, but with its values updated to match that of the
										rhs_ argument
						*/
						Iterator& operator=(const Iterator& iter_);
						
						/**
							\brief		Tests whether a given Iterator object is the same as another
							\author		GW
							\date		08/2014

							\param		iter_	Iterator to compare against (righthand side of equality operator)
							\returns	true for the target and rhs_ having the same state, false otherwise
						*/
						bool operator==(const Iterator& iter_) const;
						
						/**
							\brief		Tests whether a given Iterator object is not the same as another
							\author		GW
							\date		08/2014

							\param		iter_	Iterator to compare against (righthand side of inequality operator)
							\returns	true for the target and rhs_ having differing states, false otherwise
						*/
						bool operator!=(const Iterator& iter_) const;

						/**
							\brief		Prefix increment operator for forward iteration
							\author		GW
							\date		08/2014
							
							\returns	A reference to the iterator after its position was changed
						*/
						Iterator& operator++();

						/**
							\brief		Postfix increment operator for forward iteration
							\author		GW
							\date		08/2014
							
							\returns	A copy of the original iterator before its position was changed
						*/
						Iterator operator++(int);

						/**
							\brief		Prefix decrement operator for reverse iteration
							\author		GW
							\date		08/2014
							
							\returns	A reference to the iterator after its position was changed
						*/
						Iterator& operator--();

						/**
							\brief		Postfix decrement operator for reverse iteration
							\author		GW
							\date		08/2014
							
							\returns	A copy of the original iterator before its position was changed
						*/
						Iterator operator--(int);

						/**
							\brief		Determines whether iteration should stop; works well as the conditional in a
										for() loop
							\author		GW
							\date		08/2014
							
							\returns	true if at the end of the list of paragraph styles, false otherwise
						*/
						bool atEnd() const;

						/**
							\brief		Gets the current paragraph style pointed to by the iterator
							\author		GW
							\date		08/2014

							\returns	The current style, or an empty TextParaStyle object if at the end (see default
										TextParaStyle constructor for more information)
						*/
						std::unique_ptr<TextParaStyle> style() const;


					private:
						// Only TextParaStyleSet can construct a new Iterator object
						friend class TextParaStyleSet;

						/**
							\brief	Private implementation data
						*/
						void* __data;
				
						/**
							\brief	Empty Iterator objects cannot be constructed
							\author	GW
							\date	08/2014
						*/
						Iterator();
					
						/**
							\brief	Creates a new iterator for a TextParaStyleSet
							\author	GW
							\date	08/2014

							\param	set_	Set to be iterated through
						*/
						explicit Iterator(const TextParaStyleSet& set_);
				};
		
				typedef std::unique_ptr<Iterator> IteratorUP;
				typedef std::shared_ptr<Iterator> IteratorSP;
				typedef std::weak_ptr<Iterator> IteratorWP;
				
				/**
					\brief	Creates a new TextParaStyleSet object
					\author	GW
					\date	08/2014
					
					\note	New TextParaStyleSet objects do not relate to any text paragraph style set on the
							Illustrator artboard; they are designed to be "receivers" of some other TextParaStyleSet
							object via the overloaded assignment operator or assigned to some other existent text art.
				*/
				TextParaStyleSet();
			
				/**
					\brief	TextParaStyleSet copy constructor
					\author	GW
					\date	08/2014
					
					\param	pss_	TextParaStyleSet object to copy values from
				*/
				TextParaStyleSet(const TextParaStyleSet& pss_);

				/**
					\brief	Destructs a TextParaStyleSet object
					\author	GW
					\date	08/2014
				*/
				virtual ~TextParaStyleSet();
			
				/**
					\brief		Overloaded assignment operator for TextParaStyleSet objects
					\author		GW
					\date		08/2014

					\param		rhs_	Existing TextParaStyleSet object to copy values from
					\returns	The target TextParaStyleSet object, but with its values updated to match that of rhs_
				*/
				virtual TextParaStyleSet& operator=(const TextParaStyleSet& rhs_);
				
				/**
					\brief		Tests whether a given TextParaStyleSet object is the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextParaStyleSet to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text paragraph style set, false otherwise
				*/
				virtual bool operator==(const TextParaStyleSet& rhs_) const;
				
				/**
					\brief		Tests whether a given TextParaStyleSet object is not the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextParaStyleSet to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text paragraph style sets, false otherwise
				*/
				virtual bool operator!=(const TextParaStyleSet& rhs_) const;

				/**
					\brief		Gets the ATE paragraph style set ref around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	ATE::IParaStylesRef for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::IParaStyles> ateParaStyleSet() const;

				/**
					\brief		Gets whether the target TextParaStyleSet object is empty
					\author		GW
					\date		08/2014
					
					\returns	true if the target TextParaStyleSet object is empty, false otherwise
					
					\note		A TextParaStyleSet object being empty is not the same as it having a size of 0.
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Gets the count of the contained paragraph styles
					\author		GW
					\date		08/2014
					
					\returns	The count of the contained paragraph style, or -1 if isEmpty() returns true
				*/
				virtual int32_t size() const;

				/**
					\brief		Gets an iterator for the set
					\author		GW
					\date		08/2014
					
					\returns	An Iterator object, for iterating through the target's children
					
					\code
						// Here's an example, iterating through the children of someSet:
						for(hdi::core::TextParaStyleSet::Iterator iter = someSet.begin(); !iter.atEnd(); ++iter)
						{
							// do stuff to iter.style()
						}
					\endcode
				*/
				virtual Iterator begin() const;

				/**
					\brief	Shows all paragraph styles
					\author	GW
					\date	08/2014
				*/
				virtual void showAllStyles() const;

				/**
					\brief	Shows only styles to which there are not references in text
					\author	GW
					\date	08/2014
				*/
				virtual void showOnlyUnreferencedStyles() const;

				/**
					\brief	Shows only styles that are used in the given text range
					\author	GW
					\date	08/2014
					
					\param	range_	The text range in question
				*/
				virtual void showOnlyStylesReferencedIn(const TextRange& range_);

				/**
					\brief	Reorders styles in this set by moving a style to a given position
					\author	GW
					\date	08/2014
					
					\param	style_	Style to move
					\param	pos_	New index for style_
				*/
				virtual void moveStyleTo(const TextParaStyle& style_, const int32_t pos_);


			private:
				friend aip::TextParaStyleSet* __accessImpl(const TextParaStyleSet&);
				friend TextParaStyleSet __accessCtor(const aip::TextParaStyleSet&);

				/**
					\brief	Private implementation object
				*/
				aip::TextParaStyleSet* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				TextParaStyleSet(const aip::TextParaStyleSet&);
		};
		
		typedef std::unique_ptr<TextParaStyleSet> TextParaStyleSetUP;
		typedef std::shared_ptr<TextParaStyleSet> TextParaStyleSetSP;
		typedef std::weak_ptr<TextParaStyleSet> TextParaStyleSetWP;
		
		extern aip::TextParaStyleSet* __accessImpl(const TextParaStyleSet&);
		extern TextParaStyleSet __accessCtor(const aip::TextParaStyleSet&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_PARA_STYLE_SET__
