/**
	\file
	\brief		Header file for plugin user interface text view widgets
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_TEXTVIEW__
#define __HDI_CORE_TEXTVIEW__

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		class Label;

		/**
			\brief	Text view widget, which allows the user to enter a large amount of arbitrary text (also useful for
					displaying large amounts of text in a read-only fashion)
		*/
		class TextView : public Widget
		{
			public:
				friend class Label;
				
				/**
					\brief	Text alignment options
				*/
				enum Alignment
				{
					UnknownAlignment	= 0,
					LeftAlignment		= 10,
					CenterAlignment		= 20,
					RightAlignment		= 30
				};

				/**
					\brief	Constructs an empty TextView object
					\author	GW
					\date	07/2014
					
					\note	To test if a TextView object is empty, call isEmpty() on it
					\note	Empty TextView objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other TextView object via the overloaded assignment operator. Empty
							TextView objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				TextView();

				/**
					\brief	Constructs a new TextView object from an existing TextView object (copy constructor)
					\author	GW
					\date	07/2014

					\param	view_	Existing TextView object
				*/
				TextView(const TextView& view_);

				/**
					\brief	Constructs a text view, which can be editable or read-only
					\author	GW
					\date	07/2014

					\param	frame_	Frame for the text view, in 1x resolution coordinates
					\param	text_	Default text in the view, as UTF-8
					\param	edit_	Whether the text view is editable
				*/
				TextView(const Rect& frame_, const std::string& text_, const bool edit_ = true);
				
				/**
					\brief	TextView destructor
					\author	GW
					\date	07/2014
				*/
				virtual ~TextView();

				/**
					\brief		Allows one TextView object to be assigned from another
					\author		GW
					\date		07/2014

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target TextView object, but with its value updated to match that of rhs_
				*/
				virtual TextView& operator=(const TextView& rhs_);
				
				/**
					\brief		Convenience method to clone a TextView object on the heap
					\author		GW
					\date		07/2014

					\returns	A pointer to the new TextView object
					
					\note		If you subclass TextView, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your TextView subclass, you will
								experience "object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned TextView object.
				*/
				virtual TextView* clone() const;
				
				/**
					\brief		Convenience method to duplicate a TextView object, creating a new and identical UI
								element to the target (but not belonging to the same containing window)
					\author		GW
					\date		07/2014
					
					\returns	A pointer to the new TextView object (and new UI element)
					
					\warning	The caller must manage the memory for the returned TextView object.
				*/
				virtual TextView* duplicate() const;
				
				/**
					\brief		Gets whether the text view is read-only
					\author		GW
					\date		07/2014
					
					\returns	true for editable or false for read-only
				*/
				virtual bool editable() const;

				/**
					\brief		Gets the current alignment of the text view
					\author		GW
					\date		07/2014

					\returns	Current text alignment
				*/
				virtual Alignment alignment() const;

				/**
					\brief	Sets the current alignment of the text view
					\author	GW
					\date	07/2014

					\param	alignment_	New text alignment
				*/
				virtual void setAlignment(const Alignment alignment_);
			
				/**
					\brief		Gets whether the text is bold
					\author		GW
					\date		07/2014

					\returns	true if the text is bold, false otherwise
				*/
				virtual bool bold() const;

				/**
					\brief	Sets the font weight
					\author	GW
					\date	07/2014

					\param	bold_	true for bold, false otherwise
				*/
				virtual void setBold(const bool bold_);

				/**
					\brief		Gets whether the text is italicized
					\author		GW
					\date		07/2014

					\returns	true if the text is italicized, false otherwise
				*/
				virtual bool italic() const;

				/**
					\brief	Sets the font style
					\author	GW
					\date	07/2014

					\param	italic_		true for italics, false otherwise
				*/
				virtual void setItalic(const bool italic_);
				
				/**
					\brief		If the target text view has been associated with a label, this gets said label
					\author		GW
					\date		07/2014
					
					\returns	The label associated with the text view, or NULL for none
				*/
				virtual std::unique_ptr<Label> label() const;
				
				/**
					\brief	Sets the label that, when clicked, will focus the target text view
					\author	GW
					\date	07/2014
					
					\param	label_	New label for focusing the target text view
					
					\note	If the label was previously linked with a text field, this will remove said link.
				*/
				virtual void setLabel(Label& label_);
				
				/**
					\brief	Removes the label that, when clicked, would have focused the target text view
					\author	GW
					\date	07/2014
				*/
				virtual void removeLabel();

				/**
					\brief		Gets the selection range of the text in the view, if any, in UTF-8
					\author		GW
					\date		03/2014
				 
					\param		pos__	Return-by-reference for the starting position of the selection range
					\param		len__	Return-by-reference for the length of the selection range
					\returns	true if the view is focused and the selection range could be acquired, false otherwise
				 
				 	\warning	This method utilizes UTF-8 byte ranges, which can be useful in certain circumstances. If
				 				you require character ranges instead, use the UTF-32 version of this method.
				 
					\note		If len__ is zero then no text is selected, but pos__ indicates the position of the
								insertion point (text caret).
				*/
				virtual bool selectionRange(uint32_t& pos__, uint32_t& len__) const;
			
				/**
					\brief		Sets the selection range of the text in the view, in UTF-8 (and focuses it)
					\author		GW
					\date		03/2014
				 
					\param		pos_	The starting position of the selection range
					\param		len_	The length of the selection range
				 
				 	\warning	This method utilizes UTF-8 byte ranges, which can be useful in certain circumstances. If
				 				you require character ranges instead, use the UTF-32 version of this method.
				 
					\note		If len_ is zero then no text will be selected, but the insertion point (text caret) will be
								moved to the pos_ value.
				*/
				virtual void setSelectionRange(const uint32_t pos_, const uint32_t len_);

				/**
					\brief		Gets the selection range of the text in the view, if any, in UTF-32
					\author		GW
					\date		03/2014
				 
					\param		pos__	Return-by-reference for the starting position of the selection range
					\param		len__	Return-by-reference for the length of the selection range
					\returns	true if the view is focused and the selection range could be acquired, false otherwise
				 
				 	\warning	This method utilizes UTF-32 character ranges, which can be useful in certain circumstances.
				 				If you require byte ranges instead, use the UTF-8 version of this method.
				 
					\note		If len__ is zero then no text is selected, but pos__ indicates the position of the
								insertion point (text caret).
				*/
				virtual bool selectionRangeUTF32(uint32_t& pos__, uint32_t& len__) const;
			
				/**
					\brief		Sets the selection range of the text in the view, in UTF-32 (and focuses it)
					\author		GW
					\date		03/2014
				 
					\param		pos_	The starting position of the selection range
					\param		len_	The length of the selection range
				 
				 	\warning	This method utilizes UTF-32 character ranges, which can be useful in certain circumstances.
				 				If you require byte ranges instead, use the UTF-8 version of this method.
				 
					\note		If len_ is zero then no text will be selected, but the insertion point (text caret) will be
								moved to the pos_ value.
				*/
				virtual void setSelectionRangeUTF32(const uint32_t pos_, const uint32_t len_);

				/**
					\brief		Gets the last character typed into the view
					\author		GW
					\date		07/2014

					\returns	Last character typed, or "" if none

					\note		The return value of this method is only valid when called from inside a value-changing
								or value-changed callback.
				*/
				virtual std::string lastCharTyped() const;

				/**
					\brief	Simulates the value of the text view changing
					\author	GW
					\date	07/2014
				*/
				virtual void valueChanging();
				
				/**
					\brief		Gets the value changing callback for the text view
					\author		GW
					\date		07/2014

					\returns	A pointer to the currently registered value changing callback
				*/
				virtual Callback* const valueChangingCallback() const;
				
				/**
					\brief	Sets the value changing callback
					\author	GW
					\date	07/2014

					\param	callback_	New callback for when the text view's value is changing
				*/
				virtual void setValueChangingCallback(const Callback& callback_);

				/**
					\brief	Simulates the text view having its value changed
					\author	GW
					\date	07/2014
				*/
				virtual void valueChanged();
				
				/**
					\brief		Gets the value changed callback for the text view
					\author		GW
					\date		07/2014

					\returns	A pointer to the currently registered value changed callback
				*/
				virtual Callback* const valueChangedCallback() const;
				
				/**
					\brief	Sets the value changed callback
					\author	GW
					\date	07/2014

					\param	callback_	New callback for when the text view's value has changed
				*/
				virtual void setValueChangedCallback(const Callback& callback_);
		};
		
		typedef std::unique_ptr<TextView> TextViewUP;
		typedef std::shared_ptr<TextView> TextViewSP;
		typedef std::weak_ptr<TextView> TextViewWP;
	}
}

#endif
// __HDI_CORE_TEXTFIELD__
