/**
	\file
	\brief		Header file for general Illustrator word manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_TEXT_WORD__
#define __HDI_CORE_TEXT_WORD__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextWord;
	}
	
	namespace core
	{
		class TextRange;
	
		/**
			\brief	Encapsulates text word functionality from the Illustrator SDK
		*/
		class TextWord
		{
			public:
				/**
					\brief	Creates a new TextWord object
					\author	GW
					\date	08/2014
					
					\note	New TextWord objects do not relate to any text word on the Illustrator artboard; they are
							designed to be "receivers" of some other TextWord object via the overloaded assignment
							operator or assigned to some other existent text art.
				*/
				TextWord();
			
				/**
					\brief	TextWord copy constructor
					\author	GW
					\date	08/2014
					
					\param	w_	TextWord object to copy values from
				*/
				TextWord(const TextWord& w_);

				/**
					\brief	Destructs a TextWord object
					\author	GW
					\date	08/2014
				*/
				virtual ~TextWord();
			
				/**
					\brief		Overloaded assignment operator for TextWord objects
					\author		GW
					\date		08/2014
					
					\param		rhs_	Existing TextWord object to copy values from
					\returns	The target TextWord object, but with its values updated to match that of rhs_
				*/
				virtual TextWord& operator=(const TextWord& rhs_);
				
				/**
					\brief		Tests whether a given TextWord object is the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextWord to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text word, false otherwise
				*/
				virtual bool operator==(const TextWord& rhs_) const;
				
				/**
					\brief		Tests whether a given TextWord object is not the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextWord to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text words, false otherwise
				*/
				virtual bool operator!=(const TextWord& rhs_) const;

				/**
					\brief		Gets the ATE word ref around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	ATE::IWordsIteratorRef for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::IWordsIterator> ateWord() const;

				/**
					\brief		Gets whether the target TextWord object is empty
					\author		GW
					\date		08/2014
					
					\returns	true if the target TextWord object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Retrieves the text range for the current word, including trailing characters
					\author		GW
					\date		08/2014

					\returns	The text range of the target word
				*/
				std::unique_ptr<TextRange> range() const;

				/**
					\brief		Retrieves the number of trailing spaces for the current word. Trailing spaces are those
								after the word, regardless of the direction of iteration.
					\author		GW
					\date		08/2014

					\returns	The number of trailing spaces
				*/
				int32_t trailingSpaceCount() const;
				
				/**
					\brief		Retrieves the total number of trailing characters for the current word, including
								spaces, other white-space characters, and punctuation characters. Trailing characters
								are those after the word, regardless of the direction of iteration.
					\author		GW
					\date		08/2014

					\returns	The number of trailing characters
				*/
				int32_t trailingCount() const;
				
				/**
					\brief		Retrieves the number of trailing punctuation characters for the current word. Trailing
								characters are those after the word, regardless of the direction of iteration.
					\author		GW
					\date		08/2014

					\returns	The number of trailing punctuation characters
				*/
				int32_t trailingTerminatingPunctuationCount() const;


			private:
				friend aip::TextWord* __accessImpl(const TextWord&);
				friend TextWord __accessCtor(const aip::TextWord&);

				/**
					\brief	Private implementation object
				*/
				aip::TextWord* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				TextWord(const aip::TextWord&);
		};
		
		typedef std::unique_ptr<TextWord> TextWordUP;
		typedef std::shared_ptr<TextWord> TextWordSP;
		typedef std::weak_ptr<TextWord> TextWordWP;
		
		extern aip::TextWord* __accessImpl(const TextWord&);
		extern TextWord __accessCtor(const aip::TextWord&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_WORD__
