/**
	\file
	\brief		Header file for handling a time of day
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_TIME__
#define __HDI_CORE_TIME__

#include "hdicoreDuration.h"

namespace hdi
{
	namespace plat
	{
		class Time;
	}
	
	namespace core
	{
		/**
			\brief	Allows for acquisition and manipulation of a time (of day)
		*/
		class Time
		{
			public:
				/**
					\brief	Constructs a Time object with a default value (midnight, 0:00)
					\author	GW
					\date	11/2014
				*/
				Time();
				
				/**
					\brief	Constructs a Time object from another existing Time object
					\author	GW
					\date	11/2014
					
					\param	t_	Existing object to copy values from
				*/
				Time(const Time& t_);
				
				/**
					\brief	Constructs a Time object with an offset from midnight in seconds
					\author	GW
					\date	11/2014
					
					\param	secsSinceMN_	Number of seconds since midnight, in the range [0,86400]
					
					\note	Values below or above the expected range will be forced to the minimum or maximum value,
							respectively.
				*/
				explicit Time(const int32_t secsSinceMN_);
				
				
				/**
					\brief	Constructs a Time object with an offset from midnight in seconds
					\author	GW
					\date	11/2014
					
					\param	secsSinceMN_	Number of seconds since midnight, in the range [0.0,86400.0]
					
					\note	Values below or above the expected range will be forced to the minimum or maximum value,
							respectively.
				*/
				explicit Time(const double secsSinceMN_);
				
				/**
					\brief	Constructs a Time object from a string representation of time
					\author	GW
					\date	11/2014
					
					\param	time_	String representation of a time of day, in 24-hour format
					
					\note	The expected format is e.g. "14:34:56.789". The hours must be in 24-hour format, and a
							variety of delimeters can be used (i.e. ':', '-', '.', and ','). Strings that do not follow
							this format have the same result as using the Time() ctor.
				*/
				Time(const std::string& time_);
				
				/**
					\brief	Constructs a Time object from a given hour, minute, second, and microsecond
					\author	GW
					\date	11/2014
					
					\param	hour_	Hour of the day, in the range [0,23]
					\param	min_	Minute of the given hour, in the range [0,59]
					\param	sec_	Second of the given minute, in the range [0,59]
					\param	us_		Microsecond of the given second, in the range [0,999999]
					
					\note	Values below or above the expected range will be forced to the minimum or maximum value,
							respectively.
				*/
				Time(const int16_t hour_, const int16_t min_, const int16_t sec_ = 0, const int32_t us_ = 0);
				
				/**
					\brief		Constructs a Time object from the current system time
					\author		GW
					\date		04/2015
					
					\param		us_		Whether to use microsecond resolution
					\returns	A Time object for the current UTC time
				*/
				static Time Current(const bool us_ = false);
				
				/**
					\brief	Destructs a Time object
					\author	GW
					\date	11/2014
				*/
				virtual ~Time();
				
				/**
					\brief		Overloaded assignment operator to copy values from one Time object to another
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator; Time object to copy values from
					\returns	The target Time object, but with values updated to match that of rhs_
				*/
				virtual Time& operator=(const Time& rhs_);
				
				/**
					\brief		Overloaded equality operator to determine if two Time objects are equal
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator; Time object to compare
					\returns	true if the two objects represent the same time of day
				*/
				virtual bool operator==(const Time& rhs_) const;
				
				/**
					\brief		Overloaded inequality operator to determine if two Time objects differ
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator; Time object to compare
					\returns	true if the two objects represent different times of day
				*/
				virtual bool operator!=(const Time& rhs_) const;
				
				/**
					\brief		Overloaded less-than operator to determine if one Time object is earlier in the day than
								the other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; Time object that might be later in the day than
										the target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a time earlier than rhs_
				*/
				virtual bool operator<(const Time& rhs_) const;
				
				/**
					\brief		Overloaded less-than-or-equal operator to determine if one Time object is earlier in the
								day than, or equal to, the other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; Time object that might be later in the day than,
										or equal to, the target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a time earlier than, or equal to, rhs_
				*/
				virtual bool operator<=(const Time& rhs_) const;
				
				/**
					\brief		Overloaded greater-than operator to determine if one Time object is later in the day
								than the other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; Time object that might be earlier in the day
										than the target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a time later than rhs_
				*/
				virtual bool operator>(const Time& rhs_) const;
				
				/**
					\brief		Overloaded greater-than-or-equal operator to determine if one Time object is later in
								the day than, or equal to, the other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; Time object that might be earlier in the day
										than, or equal to, the target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a time later than, or equal to, rhs_
				*/
				virtual bool operator>=(const Time& rhs_) const;
				
				/**
					\brief		Overloaded subtraction operator to determine the duration between two Time objects
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Time object to "substract" from the lefthand side
					\returns	The duration between the two times of day
				*/
				virtual Duration operator-(const Time& rhs_) const;
				
				/**
					\brief		Overloaded substraction operator to determine the resultant time before the target
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to substract from the lefthand
										side
					\returns	The time of day that would result from subtracting the duration from the target
				*/
				virtual Time operator-(const Duration& rhs_) const;
				
				/**
					\brief		Overloaded subtraction-assignment operator to determine and assign the resultant time
								before the initial target
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to substract from the lefthand
										side
					\returns	The target Time object, but with its values updated to reflect the substraction of the
								given duration
				*/
				virtual Time& operator-=(const Duration& rhs_);
				
				/**
					\brief		Overloaded addition operator to determine the resultant time after the target
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to add to the lefthand side
					\returns	The time of day that would result from adding the duration to the target
				*/
				virtual Time operator+(const Duration& rhs_) const;
				
				/**
					\brief		Overloaded addition-assignment operator to determine and assign the resultant time
								after the initial target
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to add to the lefthand side
					\returns	The target Time object, but with its values updated to reflect the addition of the given
								duration
				*/
				virtual Time& operator+=(const Duration& rhs_);
				
				/**
					\brief		Formats the target Time object into a string representation
					\author		GW
					\date		11/2014
					
					\param		format_		Input string representing the format for the output string
					\returns	An output string for the target object's time of day, formatted as indicated
					
					\note		The following characters are supported as part of the format_ argument: "H" for 24-hour
								format of an hour with leading zeros, "h" for 12-hour format of an hour with leading
								zeros, "G" for 24-hour format of an hour without leading zeros, "g" for 12-hour format
								of an hour without leading zeros, "i" for minutes with leading zeros, "s" for seconds
								with leading zeros, "u" for microseconds, "a" for lowercase Ante meridiem and Post
								meridiem, and "A" for uppercase Ante meridiem and Post meridiem.
					\note		The special characters above will be expanded as described, and any other characters
								will simply be inserted in the output string. A special character can be preceeded by a
								'\\' (backslash) character to prevent expansion (the backslash will not be inserted in
								the output string). If a backslash character is desired in the output string, preceed it
								with another backslash character.
				*/
				virtual std::string format(const std::string& format_ = "H:i:s") const;
				
				/**
					\brief		Gets the hour represented by the target
					\author		GW
					\date		11/2014
					
					\returns	The hour, in 24-hour format
				*/
				virtual int16_t hour() const;
								
				/**
					\brief	Sets the hour that the target should represent
					\author	GW
					\date	11/2014
					
					\param	hour_	Hour of day, in 24-hour format, in range [0,23]
					
					\note	Values below or above the expected range will be forced to the minimum or maximum value,
							respectively.
				*/
				virtual void setHour(const int16_t hour_);
				
				/**
					\brief		Gets the minute of the hour represented by the target
					\author		GW
					\date		11/2014
					
					\returns	The minute of the hour
				*/
				virtual int16_t minute() const;
								
				/**
					\brief	Sets the minute of the hour represented by the target
					\author	GW
					\date	11/2014
					
					\param	min_	Minute of the hour, in range [0,59]
					
					\note	Values below or above the expected range will be forced to the minimum or maximum value,
							respectively.
				*/
				virtual void setMinute(const int16_t min_);
				
				/**
					\brief		Gets the second of the minute represented by the target
					\author		GW
					\date		11/2014
					
					\returns	The second of the minute
				*/
				virtual int16_t second() const;
								
				/**
					\brief	Sets the second of the minute represented by the target
					\author	GW
					\date	11/2014
					
					\param	sec_	Second of the minute, in range [0,59]
					
					\note	Values below or above the expected range will be forced to the minimum or maximum value,
							respectively.
				*/
				virtual void setSecond(const int16_t sec_);
				
				/**
					\brief		Gets the microsecond of the second represented by the target
					\author		GW
					\date		11/2014
					
					\returns	The microsecond of the second
				*/
				virtual int32_t microsecond() const;
								
				/**
					\brief	Sets the microsecond of the second represented by the target
					\author	GW
					\date	11/2014
					
					\param	us_		Microsecond of the second, in range [0,999999]
					
					\note	Values below or above the expected range will be forced to the minimum or maximum value,
							respectively.
				*/
				virtual void setMicrosecond(const int32_t us_);
				
				/**
					\brief		Gets the number of seconds, since midnight, represented by the target
					\author		GW
					\date		11/2014
					
					\returns	The number of seconds since midnight
				*/
				virtual double secondsSinceMidnight() const;
				
				/**
					\brief	Sets the number of seconds, since midnight, represented by the target
					\author	GW
					\date	11/2014
					
					\param	secsSinceMN_	Seconds since midnight, in range [0.0,86400.0]
					
					\note	Values below or above the expected range will be forced to the minimum or maximum value,
							respectively.
				*/
				virtual void setSecondsSinceMidnight(const double secsSinceMN_);


			private:
				friend class DateTime;
				
				/**
					\brief	Stores the internal time object
				*/
				plat::Time* __impl;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	11/2014
				*/
				Time(const plat::Time&);
		};
		
		typedef std::unique_ptr<Time> TimeUP;
		typedef std::shared_ptr<Time> TimeSP;
		typedef std::weak_ptr<Time> TimeWP;
	}
}

/**
	\brief		Adds a duration to a time of day
	\author		GW
	\date		11/2014
	
	\param		lhs_	Duration to use to offset the time
	\param		rhs_	Time to be offset by a duration
	\returns	A new Time object for the result of the offset
*/
hdi::core::Time operator+(const hdi::core::Duration& lhs_, const hdi::core::Time& rhs_);

#endif
// __HDI_CORE_TIME__
