/**
	\file
	\brief		Header file for describing a 2D transformation matrix
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_TRANSFORM_MATRIX__
#define __HDI_CORE_TRANSFORM_MATRIX__

#include <memory>

namespace hdi
{
	#if defined(HDI_CORE_AIP_MODE)
		namespace aip
		{
			class TransformMatrix;
		}
	#elif defined(HDI_CORE_PSP_MODE)
		namespace geo
		{
			class TransformMatrix;
		}
	#endif

	namespace core
	{
		class Angle;
	
		/**
			\brief	Allows for 2-dimensional transformations in a single application via matrix math
		*/
		class TransformMatrix
		{
			public:
				/**
					\brief	Row 0, column 0 of the transformation matrix
				*/
				double a;

				/**
					\brief	Row 0, column 1 of the transformation matrix
				*/
				double b;

				/**
					\brief	Row 1, column 0 of the transformation matrix
				*/
				double c;

				/**
					\brief	Row 1, column 1 of the transformation matrix
				*/
				double d;

				/**
					\brief	Stores the x-axis translation (offset) value
				*/
				double tx;

				/**
					\brief	Stores the y-axis translation (offset) value
				*/
				double ty;

				/**
					\brief	Constructs a TransformMatrix and sets its entries to that of an identity matrix
					\author	GW
					\date	08/2013
				*/
				TransformMatrix();

				/**
					\brief	Constructs a TransformMatrix and sets its entries to the arguments provided
					\author	GW
					\date	08/2013

					\param	a_		Value for entry at row 0, column 0
					\param	b_		Value for entry at row 0, column 1
					\param	c_		Value for entry at row 1, column 0
					\param	d_		Value for entry at row 1, column 1
					\param	tx_		x-axis translation (offset) value
					\param	ty_		y-axis translation (offset) value
				*/
				TransformMatrix(
					const double a_,
					const double b_,
					const double c_,
					const double d_,
					const double tx_,
					const double ty_
				);

				/**
					\brief	TransformMatrix copy constructor
					\author	GW
					\date	08/2013

					\param	m_	Existing TransformMatrix to copy
				*/
				TransformMatrix(const TransformMatrix& m_);

				/**
					\brief	TransformMatrix destructor
					\author	GW
					\date	08/2013
				*/
				virtual ~TransformMatrix();

				/**
					\brief	Sets a TransformMatrix object's entries to that of an identity matrix
					\author	GW
					\date	08/2013
				*/
				virtual void setIdentity();

				/**
					\brief		Gets whether a TransformMatrix currently has its entries set to that of an identity matrix
					\author		GW
					\date		08/2013

					\returns	true if the matrix values are those of an identity matrix, or false otherwise
				*/
				virtual bool isIdentity() const;

				/**
					\brief	Sets the translation values of a matrix
					\author	GW
					\date	08/2013

					\param	tx_		Amount to offset along the x-axis
					\param	ty_		Amount to offset along the y-axis

					\note	Negative values go toward 0.0 if the object to be transformed has a positive x and/or y
							value; positive values go away from 0.0
				*/
				virtual void setTranslate(const double tx_, const double ty_);

				/**
					\brief	Sets the rotation values, about the origin, of a matrix
					\author	GW
					\date	08/2013

					\param	angle_	Amount to rotate about the origin
				*/
				virtual void setRotate(const Angle& angle_);

				/**
					\brief	Sets the reflection values, about the origin, of a matrix
					\author	GW
					\date	12/2024

					\param	angle_	Hyperplane angle about which to reflect
				*/
				virtual void setReflect(const Angle& angle_);

				/**
					\brief	Sets the scale values, from the origin, of a matrix
					\author	GW
					\date	08/2013

					\param	sx_		Amount to scale in the x-direction (1.0 being the current size)
					\param	sy_		Amount to scale in the y-direction (1.0 being the current size)
				*/
				virtual void setScale(const double sx_, const double sy_);

				/**
					\brief	Concatenates translation values to an existing matrix
					\author	GW
					\date	08/2013

					\param	tx_		Amount to offset along the x-axis
					\param	ty_		Amount to offset along the y-axis

					\note	Negative values go toward 0.0 if the object to be transformed has a positive x and/or y
							value; positive values go away from 0.0
				*/
				virtual void concatTranslate(const double tx_, const double ty_);

				/**
					\brief	Concatenates rotation values to an existing matrix
					\author	GW
					\date	08/2013

					\param	angle_	Amount to rotate about the origin
				*/
				virtual void concatRotate(const Angle& angle_);

				/**
					\brief	Concatenates reflection values to an existing matrix
					\author	GW
					\date	12/2024

					\param	angle_	Hyperplane angle about which to reflect
				*/
				virtual void concatReflect(const Angle& angle_);

				/**
					\brief	Concatenates scale values to an existing matrix
					\author	GW
					\date	08/2013

					\param	sx_		Amount to scale in the x-direction (1.0 being the current size)
					\param	sy_		Amount to scale in the y-direction (1.0 being the current size)
				*/
				virtual void concatScale(const double sx_, const double sy_);

				/**
					\brief	Concatenates one matrix with another
					\author	GW
					\date	08/2013

					\param	matrix2_	Second matrix to concatenate with the target matrix
				*/
				virtual void concat(const TransformMatrix& matrix2_);

				/**
					\brief		Gets the determinant of the target matrix
					\author		GW
					\date		08/2013

					\returns	Determinant value of the matrix
				*/
				virtual double determinant() const;

				/**
					\brief		Gets whether the target matrix is degenerate
					\author		GW
					\date		08/2013

					\returns	true for a degenerate matrix (has a determinant of 0), or false otherwise
				*/
				virtual bool degenerate() const;

				/**
					\brief		Inverts a target matrix
					\details	If the inverse of the target matrix were concatenated with the target itself, the result
								would be an identity matrix
					\author		GW
					\date		08/2013

					\param		inverse__	Return-by-reference variable for the inverted matrix
					\returns	true if inversion is possible and was successfully performed, false otherwise

					\note		Inversion cannot be peformed when the target matrix is degenerate. In that case, the
								inverse__ argument will be set to an identity matrix and false is returned.
				*/
				virtual bool invert(TransformMatrix& inverse__) const;

				/**
					\brief		Compares two matrices for equality
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the == operator; the matrix to compare against
					\returns	true if all fields of both matrices are binary equal to each other, or false otherwise
				*/
				virtual bool operator==(const TransformMatrix& rhs_) const;

				/**
					\brief		Compares two matrices for inequality
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the != operator; the matrix to compare against
					\returns	true if any fields of both matrices are unequal to each other, or false otherwise
				*/
				virtual bool operator!=(const TransformMatrix& rhs_) const;

				/**
					\brief		Sets a target matrix to have the same values as another
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the = operator; the matrix to pull values from
					\returns	The target matrix, but with its values equaling those from rhs_
				*/
				virtual TransformMatrix& operator=(const TransformMatrix& rhs_);

				/**
					\brief		Multiplies two matrices together
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the * operator; the matrix to multiply with
					\returns	A new TransformMatrix object containing the values of the fields from this and rhs_
								multiplied together.
				*/
				virtual TransformMatrix operator*(const TransformMatrix& rhs_) const;

				/**
					\brief		Multiplies and assigns (concatenates) one matrix with another
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the *= operator; the matrix to multiply with
					\returns	The target TransformMatrix object, but updated with the values of the fields from this
								and rhs_ multiplied together.
				*/
				virtual TransformMatrix& operator*=(const TransformMatrix& rhs_);


			private:
				#if defined(HDI_CORE_AIP_MODE)
					friend aip::TransformMatrix* __accessImpl(const TransformMatrix&);
					friend TransformMatrix __accessCtor(const aip::TransformMatrix&);

					/**
						\brief	Private implementation object
					*/
					aip::TransformMatrix* __impl;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					TransformMatrix(const aip::TransformMatrix&);
				#elif defined(HDI_CORE_PSP_MODE)
					friend geo::TransformMatrix* __accessImpl(const TransformMatrix&);
					friend TransformMatrix __accessCtor(const geo::TransformMatrix&);

					/**
						\brief	Private implementation object
					*/
					geo::TransformMatrix* __impl;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					TransformMatrix(const geo::TransformMatrix&);
				#endif
		};
		
		typedef std::unique_ptr<TransformMatrix> TransformMatrixUP;
		typedef std::shared_ptr<TransformMatrix> TransformMatrixSP;
		typedef std::weak_ptr<TransformMatrix> TransformMatrixWP;
	
		#if defined(HDI_CORE_AIP_MODE)
			extern aip::TransformMatrix* __accessImpl(const TransformMatrix&);
			extern TransformMatrix __accessCtor(const aip::TransformMatrix&);
		#elif defined(HDI_CORE_PSP_MODE)
			extern geo::TransformMatrix* __accessImpl(const TransformMatrix&);
			extern TransformMatrix __accessCtor(const geo::TransformMatrix&);
		#endif
	}
}

#endif
// __HDI_CORE_TRANSFORM_MATRIX__
