/**
	\file
	\brief		Header file for transformation matrix preference data storage
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_TRANSFORM_MATRIX_PREF__
#define __HDI_CORE_TRANSFORM_MATRIX_PREF__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicorePrefData.h"
#include "hdicoreTransformMatrix.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Allows for storage of transformation matrices
		*/
		class TransformMatrixPref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty TransformMatrixPref object
					\author	GW
					\date	09/2013
					
					\note	To test if a TransformMatrixPref object is empty, call isEmpty() on it
					\note	Empty TransformMatrixPref objects do not relate to any actual preference data; they are
							designed to be "receivers" of some other TransformMatrixPref object via the overloaded
							assignment operator. Empty TransformMatrixPref objects are useless until such time (though
							it is safe to call any of their methods).
				*/
				TransformMatrixPref();
			
				/**
					\brief	Constructs a new TransformMatrixPref object from an existing TransformMatrixPref object (copy
							constructor)
					\author	GW
					\date	09/2013

					\param	tm_	Existing TransformMatrixPref object
				*/
				TransformMatrixPref(const TransformMatrixPref& tm_);

				/**
					\brief	Constructs a TransformMatrixPref object, with new data, to be added to a container at a
							later time
					\author	GW
					\date	09/2013

					\param	value_	Initial matrix value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				TransformMatrixPref(const TransformMatrix& value_, const std::string& name_ = "");

				/**
					\brief	Destructs a TransformMatrixPref object
					\author	GW
					\date	09/2013
				*/
				virtual ~TransformMatrixPref();

				/**
					\brief		Allows one TransformMatrixPref object to be assigned from another
					\author		GW
					\date		09/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target TransformMatrixPref object, but with its value updated to match that of rhs_
				*/
				virtual TransformMatrixPref& operator=(const TransformMatrixPref& rhs_);
				
				/**
					\brief		Convenience method to clone an TransformMatrixPref object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new TransformMatrixPref object
					
					\note		If you subclass TransformMatrixPref, you MUST overload this method yourself! If you
								don't and/or your clone() method does not return an instance of your TransformMatrixPref
								subclass, you will experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned TransformMatrixPref object.
				*/
				virtual TransformMatrixPref* clone() const;

				/**
					\brief		Gets the matrix for this object
					\author		GW
					\date		09/2013

					\returns	The TransformMatrix object being described by this wrapper
				*/
				virtual TransformMatrix transformMatrix() const;

				/**
					\brief		Sets the matrix for this object
					\author		GW
					\date		09/2013

					\param		m_	New matrix value
					\returns	true if the matrix is successfully updated, or false otherwise
				*/
				virtual bool setTransformMatrix(const TransformMatrix& m_);
		};
		
		typedef std::unique_ptr<TransformMatrixPref> TransformMatrixPrefUP;
		typedef std::shared_ptr<TransformMatrixPref> TransformMatrixPrefSP;
		typedef std::weak_ptr<TransformMatrixPref> TransformMatrixPrefWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TRANSFORM_MATRIX_PREF__
