/**
	\file
	\brief		Header file for a wide variety of necessary typedefs, enums, and forwards declarations
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_TYPES__
#define __HDI_CORE_TYPES__

#if defined(MAC_ENV)
	#include <stdint.h>

	@class NSApplication;
	@class NSBundle;
	@class NSFont;
	@class NSGraphicsContext;
	@class NSMenuItem;
	@class NSView;
	@class NSWindow;
#elif defined(WIN_ENV)
	#include <cstdint>

	typedef int16_t mode_t;
	
	#ifndef _INC_WINDOWS
		struct HINSTANCE__;
		typedef struct HINSTANCE__* HINSTANCE;

		struct HWND__;
		typedef struct HWND__* HWND;

		struct HMENU__;
		typedef struct HMENU__* HMENU;

		struct HFONT__;
		typedef struct HFONT__* HFONT;
	#endif
		
	class CDC;
	class CDialog;
	class CWnd;
#endif

#include <memory>
#include <string>

#include "hdicoreMacros.h"



namespace hdi
{
	namespace core
	{
		/**
			\brief	Typedef of unsigned char as byte, because char is guaranteed to be one byte per the spec

			\note	This exists because passing e.g. char* for a binary chunk of data just offers no semantic meaning.
		*/
		typedef unsigned char byte;
		
		/**
			\brief	Typedef of char as UTF8Char
			
			\note	Be aware that one UTF8Char cannot encode all possible UTF-8 code points; UTF-8 is a multibyte
					encoding scheme. As such, if you need to iterate over actual code points, use UTF32Char (see the
					hdicoreStrings.h file).
		*/
		typedef char UTF8Char;
		
		/**
			\brief		Typedef of std::string as UTF8String
			\details	All functions or methods in hdi_core that accept or return a std::string have encoded it as UTF-8.
			
			\note		Be aware that UTF-8 is a multibyte encoding scheme. As such, if you need to iterate over actual
						code points, use UTF32String (see the hdicoreStrings.h file).
		*/
		typedef std::string UTF8String;
		
		/**
			\brief	Typedef of uint16_t as UTF16Char
			
			\note	Be aware that one UTF16Char cannot encode all possible UTF-16 code points; UTF-16 is a multibyte
					encoding scheme. As such, if you need to iterate over actual code points, use UTF32Char (see the
					hdicoreStrings.h file).
		*/
		typedef uint16_t UTF16Char;
		
		/**
			\brief		Typedef of std::basic_string<UTF16Char> as UTF16String
			
			\note		Be aware that UTF-16 is a multibyte encoding scheme. As such, if you need to iterate over actual
						code points, use UTF32String (see the hdicoreStrings.h file).
		*/
		typedef std::basic_string<UTF16Char> UTF16String;
		
		/**
			\brief	Typedef of uint32_t as UTF32Char
		*/
		typedef uint32_t UTF32Char;
		
		/**
			\brief		Typedef of std::basic_string<UTF32Char> as UTF32String
		*/
		typedef std::basic_string<UTF32Char> UTF32String;
		
		#if defined(MAC_ENV)
			typedef NSApplication* PlatformAppPtr;
			typedef NSGraphicsContext* PlatformDrawingContextPtr;
			typedef NSWindow* PlatformFloatingDialogPtr;
			typedef NSFont* PlatformFontPtr;
			typedef NSWindow* PlatformModalDialogPtr;
			typedef NSBundle* PlatformPluginPtr;
			typedef NSView* PlatformWidgetPtr;
		#elif defined(WIN_ENV)
			typedef HINSTANCE PlatformAppPtr;
			typedef CDC* PlatformDrawingContextPtr;
			typedef CDialog* PlatformFloatingDialogPtr;
			typedef HFONT PlatformFontPtr;
			typedef CDialog* PlatformModalDialogPtr;
			typedef HINSTANCE PlatformPluginPtr;
			typedef CWnd* PlatformWidgetPtr;
		#endif
		
		#if defined(WIN_ENV)
			typedef HWND PlatformAppWindowPtr;
		#endif
			
		#if defined(HDI_CORE_AIP_MODE)
			#if defined(MAC_ENV)
				typedef NSWindow* PlatformDocWindowPtr;
			#elif defined(WIN_ENV)
				typedef HWND PlatformDocWindowPtr;
			#endif
		#endif
		// HDI_CORE_AIP_MODE
		
		#if defined(HDI_CORE_PSP_MODE)
			#if defined(MAC_ENV)
				typedef NSMenuItem* PlatformMenuItemPtr;
			#elif defined(WIN_ENV)
				struct WindowsMenuItem
				{
					/**
						\brief	Stores the containing/parent menu
					*/
					HMENU parent;

					/**
						\brief	Stores the ID of the menu item
					*/
					uint16_t menuID;
				};

				typedef const WindowsMenuItem* PlatformMenuItemPtr;
			#endif
		#endif
		// HDI_CORE_PSP_MODE
		
		#if defined(HDI_CORE_AIP_MODE)
			/**
				\brief	Indicates a type of art (when querying an art object for its type)
			*/
			enum ArtType
			{
				ArtTypeUnknown		= 0,
				ArtTypeGroup		= 100,
				ArtTypePath			= 110,
				ArtTypeCompoundPath	= 120,
				ArtTypePlaced		= 200,
				ArtTypeRaster		= 300,
				ArtTypeCustom		= 310,
				ArtTypeMesh			= 320,
				ArtTypeText			= 330,
				ArtTypeSymbol		= 340,
				ArtTypeForeign		= 350,
				ArtTypeTextLegacy	= 360	// From Illustrator 10 and below
			};

			/**
				\brief	Indicates a type of art (when searching the document artwork tree for art objects)
			*/
			enum MatchArtType
			{
				MatchArtTypeUnknown			= 0,
				MatchArtTypeGroup			= 1 << 0,
				MatchArtTypePath			= 1 << 1,
				MatchArtTypeCompoundPath	= 1 << 2,
				MatchArtTypePlaced			= 1 << 3,
				MatchArtTypeRaster			= 1 << 4,
				MatchArtTypeCustom			= 1 << 5,
				MatchArtTypeMesh			= 1 << 6,
				MatchArtTypeText			= 1 << 7,
				MatchArtTypeSymbol			= 1 << 8,
				MatchArtTypeForeign			= 1 << 9,
				MatchArtTypeTextLegacy		= 1 << 10,
				MatchArtTypeAll				= ~0L
			};
			
			/**
				\brief	Indicates a queryable attribute of an art object
			*/
			enum ArtAttribute
			{
				ArtAttrSelected			= 1 << 0,
				ArtAttrFullySelected	= 1 << 1,
				ArtAttrLocked			= 1 << 2,
				ArtAttrHidden			= 1 << 3,
				ArtAttrExpanded			= 1 << 4,
				ArtAttrTargeted			= 1 << 5,
				ArtAttrIsClipMask		= 1 << 6,
				ArtAttrIsTextWrap		= 1 << 7,
				ArtAttrHasSimpleStyle	= 1 << 8,	// Cannot be set, only queried
				ArtAttrHasActiveStyle	= 1 << 9,	// Cannot be set, only queried
				ArtAttrStyleIsDirty		= 1 << 10,
				ArtAttrPartOfCompound	= 1 << 11	// Cannot be set, only queried
			};
			
			/**
				\brief	Indicates desired (or undesired) attributes when searching the document artwork tree for art objects
			*/
			enum MatchArtAttribute
			{
				MatchArtAttrNone					= 0,
				MatchArtAttrSelected				= 1 << 0,
				MatchArtAttrFullySelected			= 1 << 1,
				MatchArtAttrLocked					= 1 << 2,
				MatchArtAttrHidden					= 1 << 3,
				MatchArtAttrExpanded				= 1 << 4,
				MatchArtAttrTargeted				= 1 << 5,
				MatchArtAttrIsClipMask				= 1 << 6,
				MatchArtAttrIsTextWrap				= 1 << 7,
				MatchArtAttrHasSimpleStyle			= 1 << 8,
				MatchArtAttrHasActiveStyle			= 1 << 9,
				MatchArtAttrStyleIsDirty			= 1 << 10,
				MatchArtAttrPartOfCompound			= 1 << 11,
				MatchArtAttrSelectedTopLevelGroups	= 1 << 12,
				MatchArtAttrSelectedLeaves			= 1 << 13,
				MatchArtAttrSelectedTopLevelPainted	= 1 << 14,
				MatchArtAttrArtInDictionaries		= 1 << 15,
				MatchArtAttrArtInGraphs				= 1 << 16,
				MatchArtAttrArtInResultGroups		= 1 << 17,
				MatchArtAttrTextPaths				= 1 << 18,
				MatchArtAttrArtNotInPluginGroups	= 1 << 19
			};
			
			/**
				\brief	Indicates whether a matching art search should include or exclude the specified settings
				
				\note	See MatchArtSpec class
			*/
			enum MatchArtSearch
			{
				MatchArtSearchIncludes	= 10,
				MatchArtSearchExcludes	= 20
			};

			/**
				\brief	Allows a variety of art types and attributes to be specified when searching for art that either
						includes or excludes such settings
			*/
			class MatchArtSpec
			{
				public:
					/**
						\brief	A bitwise OR of MatchArtType enum values to indicate the desired (or undesired) art
					*/
					MatchArtType arts;
					
					/**
						\brief	A bitwise OR of MatchArtAttribute enum values to indicate the desired (or undesired) art
								attributes
					*/
					MatchArtAttribute attributes;

					/**
						\brief	Whether to include or exclude the specified art and attributes when searching for art
					*/
					MatchArtSearch type;
					
					/**
						\brief	Constructs a new MatchArtSpec object with default values
						\author	GW
						\date	08/2013
					*/
					MatchArtSpec();
					
					/**
						\brief	Constructs a new MatchArtSpec object with the specified values
						\author	GW
						\date	08/2013
						
						\param	arts_	Art types to include or exclude
						\param	attrs_	Attributes of the specified art types that should be included or excluded
						\param	type_	Whether the search is including or excluding the art types specified with the given
										attributes
					*/				
					MatchArtSpec(const MatchArtType arts_, const MatchArtAttribute attrs_, const MatchArtSearch type_);
					
					/**
						\brief	Destructs a MatchArtSpec object
						\author	GW
						\date	08/2013
					*/
					virtual ~MatchArtSpec();
			};

			/**
				\brief		Indicates the position of art or layers when during creation or reordering
				\details	PlaceAboveAll and PlaceBelowAll are absolute placements; they do not require any prepositional
							art or layer objects. All other placements require a prepositional art or layer object (to which
							the placement relates). For example, PlaceAbove will reorder a piece of art above another given
							prepositional art object.
			*/
			enum PaintOrder
			{
				PlaceAbove			= 100,
				PlaceBelow			= 110,
				PlaceInsideOnTop	= 120,
				PlaceInsideOnBottom	= 130,
				PlaceAboveAll		= 140,
				PlaceBelowAll		= 150
			};
		
			/**
				\brief	Describes the blending knockout of art
			*/
			enum BlendKnockout
			{
				BlendKnockoutUnknown	= 0,
				BlendKnockoutOff		= 10,
				BlendKnockoutOn			= 20,
				BlendKnockoutInherited	= 100
			};

			/**
				\brief		Describes the types of hit tests that can be performed (AKA a hit request)
				\details	These values control each type of potential hit that can be tested when creating a HitData
							object; any value can be bitwise OR'd with another. Some convenience values are included as well,
							e.g. AnyHitRequestNoGuides

				\note		Guides are technically considered a path by Illustrator, and are not included in
							SegInteriorRequest types. To include guides in a hit test, bitwise OR with the IncludeGuides
							value. To determine if a guide was hit, you will have to check the hit type for SegInteriorHit
							and the art itself using HitData::art()->path()->isGuide()
			*/
			enum HitRequest
			{
				UnknownHitRequest				= 0,
				SegPointHitRequest				= 1 << 0,
				SegInPointHitRequest			= 1 << 1,
				SegOutPointHitRequest			= 1 << 2,
				AnySegControlHitRequest			= (SegPointHitRequest | SegInPointHitRequest | SegOutPointHitRequest),
				SegInteriorHitRequest			= 1 << 3,
				SegPointOrInteriorHitRequest	= (SegPointHitRequest | SegInteriorHitRequest),
				FillHitRequest					= 1 << 4,
				HitRequestIncludesGuides		= 1 << 16,
				AnyHitRequestNoGuides			= (~0L) ^ HitRequestIncludesGuides,
				AnyHitRequest					= ~0L
			};

			/**
				\brief	Indicates the color model of new document
			*/
			enum DocumentColorModel
			{
				ColorModelUnknown	= 0,	// Reserved for future use
				ColorModelGray		= 10,	// Reserved for future use
				ColorModelRGB		= 20,
				ColorModelCMYK		= 30
			};

			/**
				\brief	Indicates how text should be justified for a text art object
			*/
			enum ParagraphJustification
			{
				JustifyLeft		= 100,
				JustifyRight	= 110,
				JustifyCenter	= 120,
				
				JustifyFullLastLineLeft		= 200,
				JustifyFullLastLineRight	= 210,
				JustifyFullLastLineCenter	= 220,
				JustifyFullLastLineFull		= 300
			};

			/**
				\brief	Indicates the orientation of some text
			*/
			enum TextOrientation
			{
				UnknownOrientation		= 0,
				HorizontalOrientation	= 100,
				VerticalOrientation		= 110
			};
			
			/**
				\brief	Indicates the leading of some text
			*/
			enum LeadingType
			{
				UnknownLeadingType	= 0,
				RomanLeadingType	= 10,
				JapaneseLeadingType	= 20
			};
			
			/**
				\brief	Indicates the composer engine of some text
			*/
			enum ComposerEngine
			{
				UnknownComposerEngine	= 0,
				LatinCJKComposerEngine	= 10,	// Original Adobe Type Engine US composer supporting Latin and CJK
				OptycaComposerEngine	= 20,	// Optyca composer provided by WinSoft WRServices, supporting complex
												// scripts like Hebrew, Arabic, Thai, Vietnames, Indic, etc.
				AdornmentComposerEngine	= 30
			};
			
			/**
				\brief	Indicates the figure style of some text
			*/
			enum FigureStyle
			{
				UnknownFigureStyle				= 0,
				DefaultFigureStyle				= 10,
				TabularLiningFigureStyle		= 20,
				ProportionalOldFigureStyle		= 30,
				ProportionalLiningFigureStyle	= 40,
				TabularOldFigureStyle			= 50
			};
			
			/**
				\brief	Indicates the underline position of some text
			*/
			enum UnderlinePosition
			{
				UnknownUnderlinePosition			= 0,
				UnderlineOffPosition				= 10,
				UnderlineOnRightInVerticalPosition	= 20,
				UnderlineOnLeftInVerticalPosition	= 30
			};
			
			/**
				\brief	Indicates the strikethrough position of some text
			*/
			enum StrikethroughPosition
			{
				UnknownStrikethroughPosition	= 0,
				StrikethroughOffPosition		= 10,
				StrikethroughOnXHeightPosition	= 20,
				StrikethroughOnEMBoxPosition	= 30
			};
			
			/**
				\brief	Indicates the position of some OpenType text
			*/
			enum FontOpenTypePosition
			{
				UnknownFontOpenTypePosition		= 0,
				NormalFontOpenTypePosition		= 10,
				SuperscriptFontOpenTypePosition	= 20,
				SubscriptFontOpenTypePosition	= 30,
				NumeratorFontOpenTypePosition	= 40,
				DenominatorFontOpenTypePosition	= 50
			};

			/**
				\brief	Indicates the baseline of some text
			*/
			enum FontBaseline
			{
				UnknownFontBaseline				= 0,
				NormalFontBaseline				= 10,
				FauxedSuperScriptFontBaseline	= 20,
				FauxedSubScriptFontBaseline		= 30
			};

			/**
				\brief	Indicates the capitalization of some text
			*/
			enum FontCapsType
			{
				UnknownFontCapsType		= 0,
				NormalFontCapsType		= 10,
				SmallFontCapsType		= 20,
				AllFontCapsType			= 30,
				AllSmallFontCapsType	= 40
			};
			
			/**
				\brief	Indicates the kerning type of some text
			*/
			enum AutoKernType
			{
				UnknownAutoKernType			= 0,
				NoAutoKernType				= 10,
				MetricAutoKernType			= 20,
				OpticalAutoKernType			= 30,
				MetricRomanOnlyAutoKernType	= 40
			};

			/**
				\brief	Indicates how the case of some text should be changed
			*/
			enum CaseChangeType
			{
				UnknownCaseChange	= 0,
				UppercaseCaseChange	= 10,
				LowercaseCaseChange	= 20,
				TitleCaseChange		= 30,
				SentenceCaseChange	= 40
			};

			/**
				\brief	Describes the orientation of a glyph run
			*/
			enum GlyphOrientation
			{
				UnknownGlyphOrientation				= 0,
				HorizontalGlyphOrientation			= 10,	// Horizontal left to right
				VerticalGlyphOrientation			= 20,	// Standard vertical, straight from the font as vertical glyphs
				HorizontalRotatedGlyphOrientation	= 30,	// Horizontal glyphs rotated 90 degrees to be on side in vertical
				VerticalUprightRomanGlyphOrientation= 40	// Deprecated, but it may still exist in files
			};
		
			/**
				\brief	Indicates the ruler units of a document
			*/
			enum RulerUnits
			{
				UnknownRulerUnits		= 0,
				InchesRulerUnits		= 10,
				CentimetersRulerUnits	= 20,
				MillimetersRulerUnits	= 30,
				PicasRulerUnits			= 40,
				PointsRulerUnits		= 50,
				PixelsRulerUnits		= 60,
				FeetRulerUnits			= 70,
				QRulerUnits				= 1000	// Only available when Illustrator locale is Locale_JPN_JP
			};
		#endif
		// HDI_CORE_AIP_MODE

		#if defined(HDI_CORE_PSP_MODE)
			/**
				\brief	Indicates the color space of a document
			*/
			enum ColorSpace
			{
				UnknownColorSpace			= 0,
				GrayscaleColorSpace			= 10,
				Gray16ColorSpace			= 20,
				Gray32ColorSpace			= 30,
				RGBColorSpace				= 40,
				RGB48ColorSpace				= 50,
				RGB96ColorSpace				= 60,
				CMYKColorSpace				= 70,
				CMYK64ColorSpace			= 80,
				LabColorSpace				= 90,
				Lab48ColorSpace				= 100,
				HSLColorSpace				= 110,
				HSBColorSpace				= 120,
				BitmapColorSpace			= 130,
				IndexedColorSpace			= 140,
				MultichannelColorSpace		= 150,
				DeepMultichannelColorSpace	= 160,
				DuotoneColorSpace			= 170,
				Duotone16ColorSpace			= 180
			};
			
			/**
				\brief	Indicates how the document content should be interpolated when resized
			*/
			enum ImageInterpolation
			{
				UnknownImageInterpolation			= 0,
				NoImageInterpolation				= 10,
				AutomaticImageInterpolation			= 20,
				PreserveDetailsImageInterpolation	= 30,
				BicubicSmootherImageInterpolation	= 40,
				BicubicSharperImageInterpolation	= 50,
				BicubicImageInterpolation			= 60,
				NearestNeighborImageInterpolation	= 70,
				BilinearImageInterpolation			= 80
			};
		
			/**
				\brief	Indicates the type of caller a numeric selector is associated with
			*/
			enum CallerType
			{
				UnknownCallerType		= 0,
				AcquireCallerType		= 10,
				ExportCallerType		= 20,
				ExtensionCallerType		= 30,
				FilterCallerType		= 40,
				FormatCallerType		= 50,
				MeasurementCallerType	= 60,
				PickerCallerType		= 70,
				SelectionCallerType		= 80
			};
		
			/**
				\brief	Indicates the data layout in e.g. FilterMessages
			*/
			enum DataLayout
			{
				UnknownDataLayout			= 0,
				TraditionalDataLayout		= 10,
				DataLayoutRowsColumnsPlanes	= 20,
				DataLayoutRowsPlanesColumns	= 30,
				DataLayoutColumnsRowsPlanes	= 40,
				DataLayoutColumnsPlanesRows	= 50,
				DataLayoutPlanesRowsColumns	= 60,
				DataLayoutPlanesColumnsRows	= 70
			};
		
			/**
				\brief	Used to index filter behavior properties
			*/
			enum FilterCase
			{
				// Unsupported case in Photoshop
				UnsupportedFilterCase						= -10,
				
				// Unknown/error filter case
				UnknownFilterCase							= 0,
				
				// A background layer or a flat image. There is no transparency data or selection.
				FlatImageNoSelectionFilterCase				= 10,
				
				// No transparency data, but a selection may be present. The selection will be presented as mask data.
				FlatImageWithSelectionFilterCase			= 20,
				
				// Image data with an accompanying mask
				FloatingSelectionFilterCase					= 30,
				
				// Layer with transparency editing enabled and no selection. If the editable transparency cases are
				// unsupported, then Photoshop tries the corresponding protected transparency cases. This governs
				// whether the filter is expected to filter the transparency data with the color data.
				EditableTransparencyNoSelectionFilterCase	= 40,
				
				// Layer with transparency editing enabled and a selection. If the editable transparency cases are
				// unsupported, then Photoshop tries the corresponding protected transparency cases. This governs whether
				// the filter is expected to filter the transparency data with the color data.
				EditableTransparencyWithSelectionFilterCase	= 50,
				
				// Layer with transparency editing disabled and no selection.  If the protected transparency case without
				// a selection is disabled, the layer data is treated as a floating selection. Transparency data will be
				// presented via the mask portion of the interface rather than with the input data.
				LockedTransparencyNoSelectionFilterCase		= 60,
				
				// Layer with transparency editing disabled and a selection. If the protected transparency case without
				// a selection is disabled, the layer data is treated as a floating selection. Transparency data will be
				// presented via the mask portion of the interface rather than with the input data.
				LockedTransparencyWithSelectionFilterCase	= 70
			};
		
			/**
				\brief	Used to describe the type of a channel
			*/
			enum ChannelType
			{
				UnspecifiedChannelType			= 0,		// Unspecified channel
				RedChannelType					= 10,		// Red of RGB
				GreenChannelType				= 20,		// Green of RGB
				BlueChannelType					= 30,		// Blue of RGB
				CyanChannelType					= 40,		// Cyan of CMYK
				MagentaChannelType				= 50,		// Magenta of CMYK
				YellowChannelType				= 60,		// Yellow of CMYK
				BlackChannelType				= 70,		// Black of CMYK
				LChannelType					= 80,		// L of LAB
				AChannelType					= 90,		// A of LAB
				BChannelType					= 100,		// B of LAB
				DuotoneChannelType				= 110,		// Duotone
				IndexChannelType				= 120,		// Index
				BitmapChannelType				= 130,		// Bitmap
				ColorSelectedChannelType		= 140,		// Selected color
				ColorProtectedChannelType		= 150,		// Protected color
				TransparencyChannelType			= 160,		// Transparent color
				LayerMaskChannelType			= 170,		// Layer mask (alpha channel). White = transparent, black = mask
				InvertedLayerMaskChannelType	= 180,		// Inverted layer mask (inverted alpha channel)
				SelectionMaskChannelType		= 190,		// Mask/alpha for selection
				SpotColorChannelType			= 200,		// Spot Color
				VectorMaskChannelType			= 210,
				SmartFilterMaskChannelType		= 220
			};
		
			struct MonitorSpecs
			{
				/**
					\brief	The monitor's gamma value or zero if the whole record is invalid
				*/
				double gamma;
				
				/**
					\brief	The chromaticity coordinates of the monitor's phosphors
				*/
				double redX;
				
				/**
					\brief	The chromaticity coordinates of the monitor's phosphors
				*/
				double redY;
				
				/**
					\brief	The chromaticity coordinates of the monitor's phosphors
				*/
				double greenX;
				
				/**
					\brief	The chromaticity coordinates of the monitor's phosphors
				*/
				double greenY;
				
				/**
					\brief	The chromaticity coordinates of the monitor's phosphors
				*/
				double blueX;
				
				/**
					\brief	The chromaticity coordinates of the monitor's phosphors
				*/
				double blueY;
				
				/**
					\brief	The chromaticity coordinates of the monitor's white point
				*/
				double whiteX;
				
				/**
					\brief	The chromaticity coordinates of the monitor's white point
				*/
				double whiteY;
				
				/**
					\brief	The relative amount of ambient light in the room. 0.0 means a relatively dark room, 0.5
							means an average room, and 1.0 means a bright room.
				*/
				double ambient;
			};
		#endif
		// HDI_CORE_PSP_MODE
		
		/**
			\brief	Describes the blending mode of art (used for compositing art objects)
		*/
		enum BlendMode
		{
			UnknownBlendMode		= 0,
			NormalBlendMode			= 10,
			MultiplyBlendMode		= 20,
			ScreenBlendMode			= 30,
			OverlayBlendMode		= 40,
			SoftLightBlendMode		= 50,
			HardLightBlendMode		= 60,
			ColorDodgeBlendMode		= 70,
			ColorBurnBlendMode		= 80,
			DarkenBlendMode			= 90,
			LightenBlendMode		= 100,
			DifferenceBlendMode		= 110,
			ExclusionBlendMode		= 120,
			HueBlendMode			= 130,
			SaturationBlendMode		= 140,
			ColorBlendMode			= 150,
			LuminosityBlendMode		= 160
			
			#if defined(HDI_CORE_PSP_MODE)
				,
				DissolveBlendMode	= 1010,
				BehindBlendMode		= 1020,
				ClearBlendMode		= 1030
			#endif
			
		};
		
		/**
			\brief	Indicates which type of resource is related to a given icon (for passing to Adobe's SDK)
		*/
		enum IconResourceType
		{
			IconResourceTypeUnknown	= 0,
			IconResourceTypePNGI	= 10,	// PNGI resource type
			IconResourceTypeSVG		= 20	// SVG resource type
		};
		
		/**
			\brief	These allow for simpler art transformation/bounds metadata access
			\note	For example, instead of passing a transform point to an Art method, pass a descriptor for which
					point in the art bounds you want to "pin"
		*/
		enum BoundsPinPointType
		{
			// DO NOT CHANGE THE RAW VALUES HERE
			UnknownPinPoint			= 0,
			TopLeftPinPoint			= 1,
			TopCenterPinPoint		= 2,
			TopRightPinPoint		= 3,
			MidLeftPinPoint			= 4,
			MidCenterPinPoint		= 5,
			MidRightPinPoint		= 6,
			BottomLeftPinPoint		= 7,
			BottomCenterPinPoint	= 8,
			BottomRightPinPoint		= 9
		};
		
		/**
			\brief	Describes which points were found when checking for intersections between a rect and a line
		*/
		enum RectLineIntersectionType
		{
			NoLineIntersectionsFound	= 0,
			FirstLineIntersectionFound	= 1,
			BothLineIntersectionsFound	= 2
		};
		
		/**
			\brief	Describes the type of a Window object, allowing for proper casting to its subclass
		*/
		enum WindowType
		{
			UnknownWindowType			= 0,
			ControlBarWindowType		= 10,	// ControlBar class, only available in Illustrator
			ModalDialogWindowType		= 20,	// ModalDialog class
			PanelWindowType				= 30,	// Panel class, only available in Illustrator
			FloatingDialogWindowType	= 40	// FloatingDialog class
		};
		
		/**
			\brief	Used when querying the app for current UI colors
		*/
		enum ThemeComponentType
		{
			WindowBackgroundColorThemeComponent		= 10,
			LabelTextColorThemeComponent			= 20,
			LabelTextDisabledColorThemeComponent	= 30,
			FieldTextColorThemeComponent			= 40,
			FieldTextDisabledColorThemeComponent	= 50,
			FieldBackgroundColorThemeComponent		= 60,
			FocusRingColorThemeComponent			= 70,
			ShadowColorThemeComponent				= 80,
			HighlightColorThemeComponent			= 90
		};

		/**
			\brief	Indicates the locale under which the app is running. Many languages are only listed for one country,
					but some are broken down for regional differences (e.g. French in France versus French in Canada).
		*/
		enum Locale
		{
			Locale_Application	= 0,	// Current application locale (reserved for future use)
			Locale_ARA_AE		= 100,	// Arabic, United Arab Emirates
			Locale_BUL_BG		= 110,	// Bulgarian, Bulgaria
			Locale_CES_CZ		= 120,	// Czech, Czech Republic
			Locale_DAN_DK		= 130,	// Danish, Denmark
			Locale_DEU_CH		= 140,	// German, Switzerland
			Locale_DEU_DE		= 150,	// German, Germany
			Locale_ELL_GR		= 160,	// Greek, Greece
			Locale_ENG_AE		= 170,	// English, United Arab Emirates
			Locale_ENG_CA		= 180,	// English, Canada
			Locale_ENG_GB		= 190,	// English, United Kingdom
			Locale_ENG_IL		= 200,	// English, Israel
			Locale_ENG_US		= 210,	// English, United States
			Locale_FAS_IR		= 220,	// Persian, Islamic Republic of Iran
			Locale_FIN_FI		= 230,	// Finnish, Finland
			Locale_FRA_CA		= 240,	// French, Canada
			Locale_FRA_FR		= 250,	// French, France
			Locale_FRA_MA		= 260,	// French, Morocco
			Locale_HEB_IL		= 270,	// Hebrew, Israel
			Locale_HUN_HU		= 280,	// Hungarian, Hungary
			Locale_ITA_IT		= 290,	// Italian, Italy
			Locale_JPN_JP		= 300,	// Japanese, Japan
			Locale_KOR_KR		= 310,	// Korean, Republic of Korea
			Locale_NLD_NL		= 320,	// Dutch, Netherlands
			Locale_NNO_NO		= 330,	// Nynorsk, Norway
			Locale_NOR_NO		= 340,	// Norwegian, Norway
			Locale_POL_PL		= 350,	// Polish, Poland
			Locale_POR_BR		= 360,	// Portuguese, Brazil
			Locale_POR_PT		= 370,	// Portuguese, Portugal
			Locale_RON_RO		= 380,	// Romanian, Romania
			Locale_RUS_RU		= 390,	// Russian, Russian Federation
			Locale_SPA_ES		= 400,	// Spanish, Spain
			Locale_SPA_MX		= 410,	// Spanish, Mexico
			Locale_SWE_SE		= 420,	// Swedish, Sweden
			Locale_THA_TH		= 430,	// Thai, Thailand
			Locale_TUR_TR		= 440,	// Turkish, Turkey
			Locale_UKR_UA		= 450,	// Ukrainian, Ukraine
			Locale_VIE_VN		= 460,	// Vietnamese, Vietnam
			Locale_ZHO_CN		= 470,	// Chinese (simplified), China
			Locale_ZHO_TW		= 480	// Chinese (traditional), Taiwan
		};

		/**
			\brief		Gets a standard language code followed by a standard country code, in a string, to describe a
						given Locale enum value
			\author		GW
			\date		02/2025
			
			\param		locale_		The locale for which a full standard code is desired
			\returns	A language code followed by a country code, for example "ENG_US" for the Locale_ENG_US value
		*/
		std::string localeToFullStandardCode(const Locale locale_);

		/**
			\brief		Gets a standard language code, in a string, to describe only the language portion of a given
						Locale enum value
			\author		GW
			\date		02/2025
			
			\param		locale_		The locale for which a standard language code is desired
			\returns	A language code, for example "ENG" for the Locale_ENG_US value
		*/
		std::string localeToLanguageCode(const Locale locale_);

		/**
			\brief		Gets a standard country code, in a string, to describe only the country portion of a given
						Locale enum value
			\author		GW
			\date		02/2025
			
			\param		locale_		The locale for which a standard country code is desired
			\returns	A country code, for example "US" for the Locale_ENG_US value
		*/
		std::string localeToCountryCode(const Locale locale_);

		/**
			\brief	Describes various modifier keys
		*/
		enum ModifierKey
		{
			NoModifierKey		= 0,
			ShiftModifierKey	= 1 << 0,
			OptionModifierKey	= 1 << 1,
			AltModifierKey		= OptionModifierKey,
			ControlModifierKey	= 1 << 2,
			CommandModifierKey	= 1 << 3	// Only useful on Mac
		};
		
		/**
			\brief	Describes various mouse buttons
		*/
		enum MouseButton
		{
			NoMouseButton			= 0,
			PrimaryMouseButton		= 1 << 0,	// Usually left button
			SecondaryMouseButton	= 1 << 1,	// Usually right button
			TertiaryMouseButton		= 1 << 2	// Usually middle button
		};
		
		/**
			\brief	Describes common HTTP error codes between platforms, for convenience
		*/
		enum InternetError
		{
			InternetErrorUnknown				= -1,
			InternetErrorNone					= 0,	// No error code is present
			InternetErrorPlatformCode			= 10,	// Check the InternetGET/POST class platformErrorCode() method
			InternetErrorOperationCancelled		= 20,	// Connection was cancelled
			InternetErrorInvalidURL				= 30,	// URL is malformed or otherwise unparseable
			InternetErrorUnsupportedURL			= 40,	// URL format is parseable but not supported
			InternetErrorTimedOut				= 50,	// Connection timed out
			InternetErrorNoNetConnection		= 60,	// No internet connection is present
			InternetErrorHostLookupFailed		= 70,	// DNS lookup for the server indicated in the URL failed
			InternetErrorHostNotFound			= 80,	// No path to the server indicated in the URL
			InternetErrorCannotConnectToHost	= 90,	// The connection to the server failed or was rejected
			InternetErrorConnectionToHostLost	= 100,	// The connection to the server was accepted but lost later on
			InternetErrorTooManyRedirects		= 110,	// Too many HTTP redirects have occurred
			InternetErrorBadRedirectLocation	= 120,	// A bad redirection was attempted (bad URL, HTTPS-to-HTTP, etc.)
			InternetErrorBadServerResponse		= 130,	// The response from the server was malformed or unparseable
			InternetErrorAuthenticationNeeded	= 140,	// HTTP authentication is required, but unsupported by this lib
			InternetErrorCertificateRejected	= 150,	// The certificate was rejected, e.g. because of a CN mismatch
			InternetErrorCertificateInvalid		= 160,	// The cert is invalid, e.g. because of its validity start date
			InternetErrorCertificateExpired		= 170,	// The cert is past its expiration date
			InternetErrorCertificateBadRoot		= 180,	// The CA/root certificate is unknown/untrusted
			InternetErrorFileNotFound			= 190	// The file indicated by the URL could not be found
		};
		
		/**
			\brief	Describes a built-in application menu group
		*/
		enum BuiltinMenuGroup
		{
			UnknownMenuGroup					= 0,
			
			#if defined(HDI_CORE_AIP_MODE)
				AboutMenuGroup						= 10,
				OpenMenuGroup						= 20,
				RecentMenuGroup						= 30,
				CloseMenuGroup						= 40,
				DeviceCentralMenuGroup				= 50,
				SaveMenuGroup						= 60,
				SaveForMenuGroup					= 70,
				ImportMenuGroup						= 80,
				PlaceMenuGroup						= 90,
				ExportMenuGroup						= 100,
				DocumentUtilsMenuGroup				= 110,
				DocumentInterchangeMenuGroup		= 120,
				PrintMenuGroup						= 130,
				SendMenuGroup						= 140,
				AppUtilsMenuGroup					= 150,
				QuitMenuGroup						= 160,
				EditMenuGroup						= 170,
				UndoMenuGroup						= 180,
				PasteUtilsMenuGroup					= 190,
				EditUtilsMenuGroup					= 200,
				EditTextMenuGroup					= 210,
				PresetsMenuGroup					= 220,
				SelectMenuGroup						= 230,
				SelectMenuExternalGroup				= 240,
				SameMenuGroup						= 250,
				SelectObjectMenuGroup				= 260,
				PrefsMenuGroup						= 270,
				SyncMenuGroup						= 280,
				ClipboardMenuGroup					= 290,
				RepeatMenuGroup						= 300,
				ArrangeTransformMenuGroup			= 310,
				ArrangeMoveMenuGroup				= 320,
				ArrangeGroupMenuGroup				= 330,
				ArrangeAttribsMenuGroup				= 340,
				ViewModeMenuGroup					= 350,
				ViewAdornmentsMenuGroup				= 360,
				ViewMenuGroup						= 370,
				ViewUtilsMenuGroup					= 380,
				ViewExtMenuGroup					= 390,
				ViewSnapMenuGroup					= 400,
				ObjectAttribsMenuGroup				= 410,
				ObjectUtilsMenuGroup				= 420,
				ObjectsMenuGroup					= 430,
				ObjectPathsMenuGroup				= 440,
				ObjectPathsPopupMenuGroup			= 450,
				LockMenuGroup						= 460,
				HideMenuGroup						= 470,
				GuidesMenuGroup						= 480,
				MaskMenuGroup						= 490,
				CompoundPathsMenuGroup				= 500,
				ArtboardsGroup						= 510,
				CropMarksMenuGroup					= 520,
				GraphsMenuGroup						= 530,
				BlocksMenuGroup						= 540,
				WrapMenuGroup						= 550,
				TextPathTypeGroup					= 560,
				TypeAttribsMenuGroup				= 570,
				TypePalettesMenuGroup				= 580,
				TypeLayoutMenuGroup					= 590,
				TypeTabsMenuGroup					= TypePalettesMenuGroup,
				TypeUtilsMenuGroup					= 600,
				TypePluginsMenuGroup1				= 610,
				TypePluginsMenuGroup2				= 620,
				TypeAsianOptionsGroup				= 630,
				TypeSizeUtilsMenuGroup				= 640,
				TypeSizeMenuGroup					= 650,
				TypeLeadingUtilsMenuGroup			= 660,
				TypeLeadingMenuGroup				= 670,
				TypeAlignmentMenuGroup				= 680,
				FilterUtilities						= 690,
				EffectsMenuGroup					= 700,
				HelpMenuGroup						= 710,
				HiddenOtherSelectMenuGroup			= 720,
				HiddenOtherTextMenuGroup			= 730,
				HiddenOtherObjectMenuGroup			= 740,
				HiddenOtherPaletteMenuGroup			= 750,
				HiddenOtherMiscMenuGroup			= 760,
				WindowUtilsMenuGroup				= 770,
				ApplicationContainerMenuGroup		= 780,
				ToolPalettesMenuGroup				= 790,
				WindowLibariesMenuGroup				= 800,
				AdobeLabsMenuGroup					= 810,
				ServicesMenuGroup					= 820,
				PaintPalettesMenuGroup				= ToolPalettesMenuGroup,
				ObjectInfoPalettesMenuGroup			= ToolPalettesMenuGroup,
				AttributePalettesMenuGroup			= ToolPalettesMenuGroup,
				OtherPalettesMenuGroup				= ToolPalettesMenuGroup,
				PaintAttributesPaletteMenuGroup		= ToolPalettesMenuGroup,
				SVGPaletteMenuGroup					= ToolPalettesMenuGroup,
				WindowObjectUtilsMenuGroup			= ToolPalettesMenuGroup,
				DocInfoMenuGroup					= 830,
				ObjectRasterMenuGroup				= 840,
				ArrangeTransformMultipleMenuGroup	= 850,
				ATMMenuGroupNearGroup				= ArrangeTransformMenuGroup,
				ObjectPathsPopoutPluginMenuGroup	= 860,
				OPPPMenuGroupNearGroup				= ObjectPathsPopupMenuGroup,
				AIEditColorsSubMenuGroup			= 870,
				AdobeColorHarmonyMenuGroup			= 890,
				AdobeColorFiltersMenuGroup			= 900,
				DocumentSupportMenuGroup			= 910,
				AssetMgmtMenuGroup					= 920,
				WorkgroupMenuGroup					= DocumentSupportMenuGroup,
				ScriptsMenuGroup					= 930,
				WorkspacesMenuGroup					= 940,
				WorkspacesCustomMenuGroup			= 950,
				WorkspacesPresetsMenuGroup			= 960
			#elif defined(HDI_CORE_PSP_MODE)
				PhotoshopMenuGroup					= 10010,	// Mac only
				FileMenuGroup						= 10020,
				EditMenuGroup						= 10030,
				ImageMenuGroup						= 10040,
				LayerMenuGroup						= 10050,
				TypeMenuGroup						= 10060,
				SelectMenuGroup						= 10070,
				FilterMenuGroup						= 10080,
				ThreeDMenuGroup						= 10090,
				ViewMenuGroup						= 10100,
				WindowMenuGroup						= 10110,
				HelpMenuGroup						= 10120
			#endif
		};

		/**
			\brief	Describes the types of plugin messages for which subscriptions are available
		*/
		enum MessageType
		{
			CustomMessageType								= -10,	// Cannot be subscribed to
			UnknownMessageType								= 0,	// Cannot be subscribed to
			
			#if defined(HDI_CORE_AIP_MODE)
				ActionGoMessageType								= 10,
				AnnotationDrawMessageType						= 20,	// Handled automatically; don't subscribe manually
				AnnotationInvalidateMessageType					= 30,	// Handled automatically; don't subscribe manually
				ClipboardGoMessageType							= 40,
				ClipboardCanCopyMessageType						= 50,
				ClipboardCloneMessageType						= 60,
				ClipboardDisposeMessageType						= 70,
				FileFormatGetParamsMessageType					= 80,
				FileFormatSetParamsMessageType					= 81,
				FileFormatGoMessageType							= 90,
				FileFormatCheckMessageType						= 100,
				FileFormatUpdateMessageType						= 110,	// Deprecated
				FilterGetParamsMessageType						= 120,
				FilterGoMessageType								= 130,
				LayerListPushMessageType						= 140,
				LayerListPopMessageType							= 150,
				LayerListDeleteMessageType						= 160,
				LiveEffectEditParamsMessageType					= 170,
				LiveEffectGoMessageType							= 180,
				LiveEffectInterpolateMessageType				= 190,
				LiveEffectInputMessageType						= 200,
				LiveEffectConvertColorSpaceMessageType			= 210,
				LiveEffectScaleParamsMessageType				= 220,
				LiveEffectMergeMessageType						= 230,
				LiveEffectGetSVGFilterMessageType				= 240,
				LiveEffectAdjustColorsMessageType				= 250,
				LiveEffectIsCompatibleMessageType				= 260,
				MenuGoMessageType								= 270,	// Handled automatically; don't subscribe manually
				MenuUpdateMessageType							= 280,	// Handled automatically; don't subscribe manually
				NotifyMessageType								= 290,	// Handled automatically; don't subscribe manually
				ObjectSetDisposeContentsMessageType				= 300,
				ObjectSetMarkUsageMessageType					= 310,
				ObjectSetUpdateInternalMessageType				= 320,
				ObjectSetUpdateExternalMessageType				= 330,
				ObjectSetPurgeMessageType						= 340,
				ObjectSetChangeMessageType						= 350,
				ObjectSetChangeIndirectMessageType				= 360,
				ObjectSetCopyMessageType						= 370,
				ObjectSetInitDoneMessageType					= 380,
				CustomArtNotifyEditsMessageType					= 390,
				CustomArtUpdateViewableArtMessageType			= 400,
				CustomArtCanShowContentsMessageType				= 410,
				CustomArtInterpolateDataMessageType				= 420,
				CustomArtReplaceSymbolMessageType				= 430,
				CustomArtDisallowsArtTypeQueryMessageType		= 440,
				CustomArtPerformCustomHitTestMessageType		= 450,
				CustomArtCollectPaintStylesMessageType			= 460,
				CustomArtApplyPaintStylesMessageType			= 470,
				CustomArtAdjustColorsMessageType				= 480,
				CustomArtShowContentsMessageType				= 490,
				PreferencesInitMessageType						= 500,
				PreferencesOKMessageType						= 510,
				PreferencesCancelMessageType					= 520,
				PreferencesUpdateMessageType					= 530,
				TimerGoMessageType								= 540,	// Handled automatically; don't subscribe manually
				ToolEditMessageType								= 550,	// Handled automatically; don't subscribe manually
				ToolTrackMessageType							= 560,	// Handled automatically; don't subscribe manually
				ToolMouseDownMessageType						= 570,	// Handled automatically; don't subscribe manually
				ToolMouseDragMessageType						= 580,	// Handled automatically; don't subscribe manually
				ToolMouseUpMessageType							= 590,	// Handled automatically; don't subscribe manually
				ToolSelectMessageType							= 600,	// Handled automatically; don't subscribe manually
				ToolReselectMessageType							= 610,	// Handled automatically; don't subscribe manually
				ToolDeselectMessageType							= 620,	// Handled automatically; don't subscribe manually
				ToolDecreaseDiameterMessageType					= 630,	// Handled automatically; don't subscribe manually
				ToolIncreaseDiameterMessageType					= 640,	// Handled automatically; don't subscribe manually
				ToolAlternateSelectionQueryMessageType			= 650,
				ToolAlternateSelectionActivateMessageType		= 660,
				ToolAlternateSelectionDeactivateMessageType		= 670,
				ToolboxStartMessageType							= 680,
				ToolboxEndMessageType							= 690,
				ToolboxStartGroupMessageType					= 700,
				ToolboxEndGroupMessageType						= 710,
				ToolboxStartSetMessageType						= 720,
				ToolboxEndSetMessageType						= 730,
				ToolboxAddToolMessageType						= 740,
				ToolboxAddToolRolloverMessageType				= 750,
				ToolboxToolSelectedMessageType					= 760,
				ToolboxToolAlternateActionSelectedMessageType	= 770,
				ToolboxCycleToolMessageType						= 780,
				ToolboxSoftCycleToolMessageType					= 790,
				TransformAgainMessageType						= 800,
				WorkspaceWriteMessageType						= 810,
				WorkspaceRestoreMessageType						= 820,
				WorkspaceDefaultMessageType						= 830,
				PluginReloadMessageType							= 840,	// Handled automatically; don't subscribe manually
				PluginPurgeCachesMessageType					= 850,
				PluginUnloadMessageType							= 860,	// Handled automatically; don't subscribe manually
				AdaptersStartupMessageType						= 870,
				AdaptersShutdownMessageType						= 880,
				AdaptersDisposeInfoMessageType					= 890,
				AdaptersFlushMessageType						= 900,
				AdaptersRegisterMessageType						= 910,
				AdaptersLoadPluginMessageType					= 920,
				AdaptersReleasePluginMessageType				= 930,
				AdaptersSendMessageMessageType					= 940,
				PropertiesAcquireMessageType					= 950,
				PropertiesReleaseMessageType					= 960
			#elif defined(HDI_CORE_PSP_MODE)
				ActionGoMessageType									= 10010,
			
				AcquireStartMessageType								= 10020,
				AcquireContinueMessageType							= 10030,
				AcquireFinishMessageType							= 10040,
				AcquirePrepareMessageType							= 10050,
				AcquireFinalizeMessageType							= 10060,

				ExportStartMessageType								= 10070,
				ExportContinueMessageType							= 10080,
				ExportFinishMessageType								= 10090,
				ExportPrepareMessageType							= 10100,

				FilterParametersMessageType							= 10110,
				FilterPrepareMessageType							= 10120,
				FilterStartMessageType								= 10130,
				FilterContinueMessageType							= 10140,
				FilterFinishMessageType								= 10150,

				FormatReadPrepareMessageType						= 10160,
				FormatReadStartMessageType							= 10170,
				FormatReadContinueMessageType						= 10180,
				FormatReadFinishMessageType							= 10190,
				FormatOptionsPrepareMessageType						= 10200,
				FormatOptionsStartMessageType						= 10210,
				FormatOptionsContinueMessageType					= 10220,
				FormatOptionsFinishMessageType						= 10230,
				FormatEstimatePrepareMessageType					= 10240,
				FormatEstimateStartMessageType						= 10250,
				FormatEstimateContinueMessageType					= 10260,
				FormatEstimateFinishMessageType						= 10270,
				FormatWritePrepareMessageType						= 10280,
				FormatWriteStartMessageType							= 10290,
				FormatWriteContinueMessageType						= 10300,
				FormatWriteFinishMessageType						= 10310,
				FormatFilterFileMessageType							= 10320,
				FormatGetFilePropertyValueMessageType				= 10330,
				FormatLosslessRotateMessageType						= 10340,
				FormatBulkSettingsMessageType						= 10350,
				FormatXMPReadMessageType							= 10360,
				FormatXMPWriteMessageType							= 10370,
				FormatXMPAppendMessageType							= 10380,
				FormatBulkOpenMessageType							= 10390,
				FormatLoadMessageType								= 10400,
				FormatUnloadMessageType								= 10410,
				FormatPreferencesMessageType						= 10420,
				FormatSettingsCopyMessageType						= 10430,
				FormatSettingsPasteMessageType						= 10440,
				FormatSettingsClearMessageType						= 10450,
				FormatSettingsDefaultMessageType					= 10460,
				FormatSettingsPreviousMessageType					= 10470,
				FormatSettingsPresetListMessageType					= 10480,
				FormatSettingsChecksumMessageType					= 10490,
				FormatReadLayerStartMessageType						= 10500,
				FormatReadLayerContinueMessageType					= 10510,
				FormatReadLayerFinishMessageType					= 10520,
				FormatWriteLayerStartMessageType					= 10530,
				FormatWriteLayerContinueMessageType					= 10540,
				FormatWriteLayerFinishMessageType					= 10550,
				FormatLaunchExternalEditorMessageType				= 10560,

				MeasurementRegisterDataPointDataTypesMessageType	= 10570,
				MeasurementRegisterDataPointsMessageType			= 10580,
				MeasurementPrepareMeasurementsMessageType			= 10590,
				MeasurementRecordMeasurementsMessageType			= 10600,
				MeasurementExportMeasurementMessageType				= 10610,

				PickerPickMessageType								= 10620,

				SelectionExecuteMessageType							= 10630,
				
				PluginReloadMessageType								= 10640,	// Handled automatically; don't subscribe manually
				PluginUnloadMessageType								= 10650		// Handled automatically; don't subscribe manually
			#endif
		};

		#if defined(HDI_CORE_AIP_MODE)
			/**
				\brief	Describes the types of Illustrator notifiers for which subscriptions are available
			*/
			enum NotifierType
			{
				UnknownNotifierType	= 0,

				ArtSelectionChangedNotifierType			= 10,
				ArtObjectsChangedNotifierType			= 11,
				ArtCustomColorChangedNotifierType		= 20,
				ArtPropertiesChangedNotifierType		= 30,
				ArtStyleFocusChangedNotifierType		= 40,
				ArtStyleEditTransparencyNotifierType	= 50,
				ArtGradientChangedNotifierType			= 60,
				ArtPatternChangedNotifierType			= 70,
				ArtSymbolSetChangedNotifierType			= 80,
				ArtSymbolSetDetailedChangeNotifierType	= 90,
				ArtSymbolListChangedNotifierType		= 100,
				ArtSymbolDoubleClickedNotifierType		= 110,

				IsolationModeChangedNotifierType		= 120,
				BeforeIsolationModeChangedNotifierType	= 130,

				CurrentLayerNotifierType		= 140,
				LayerDeletionNotifierType		= 150,
				LayerSetNotifierType			= 160,
				LayerOptionsNotifierType		= 170,
				LayerListChangedNotifierType	= 180,
				
				// The following artboard notifiers are not native to Illustrator. They are provided by this library and
				// perform slightly more slowly than native notifiers (as the library must monitor artboards for these types
				// of changes).
				ArtboardCreatedNotifierType			= 184,
				ArtboardDeletedNotifierType			= 185,
				ArtboardOptionsChangedNotifierType	= 186,
				CurrentArtboardChangedNotifierType	= 187,

				DocumentChangedNotifierType					= 190,
				DocumentAboutToCloseNotifierType			= 200,
				DocumentClosedNotifierType					= 210,
				DocumentOpenedNotifierType					= 220,
				DocumentSavedNotifierType					= 225,
				DocumentNewNotifierType						= 230,
				DocumentClrMdlChangedNotifierType			= 240,
				DocumentProfilesChangedNotifierType			= 250,
				DocumentSpotColorModeChangedNotifierType	= 260,
				DocumentRulerUnitChangedNotifierType		= 270,
				DocumentWritePreprocessNotifierType			= 280,
				DocumentWritePostprocessNotifierType		= 290,
				DocumentWriteOnDiskCompleteNotifierType		= 295,
				DocumentWriteOnDiskFailedNotifierType		= 296,
				DocumentCropAreaModifiedNotifierType		= 300,
				DocumentRulerOriginChangedNotifierType		= 310,
				DocumentBleedsChangedNotifierType			= 320,
				DocumentTransparencyGridNotifierType		= 330,
				DocumentTextResourceChangedNotifierType		= 340,
				FileFormatDocumentOpenedNotifierType		= 350,
				FileFormatLinkUpdateNotifierType			= 360,

				DocumentViewChangedNotifierType					= 370,
				DocumentViewInvalidRectChangedNotifierType		= 380,
				DocumentViewStyleChangedNotifierType			= 390,
				DocumentViewActiveViewChangedNotifierType		= 400,
				DocumentViewTrimViewStateChangedNotifierType	= 405,
				DocumentViewOPPPlateStateChangedNotifierType	= 410,
				DocumentViewEdgesVisibilityChangeNotifierType	= 413,
				ActiveDocumentViewTitleChangedNotifierType		= 416,
				InvalidateByScrollNotifierType					= 420,
				ScreenModeChangedNotifierType					= 425,	// Illustrator 23 only

				CurrentFontChangedNotifierType		= 430,
				CurrentFontSizeChangedNotifierType	= 440,
				FontListChangedNotifierType			= 450,
				BeginImportCompFontNotifierType		= 460,
				EndImportCompFontNotifierType		= 470,

				ToolSelectedNotifierType					= 480,
				ToolDeselectedNotifierType					= 490,
				ToolClearStateNotifierType					= 500,
				ToolTitleChangedNotifierType				= 510,
				ToolTooltipChangedNotifierType				= 520,
				ToolChangedNotifierType						= 540,
				ToolWindowChangedNotifierType				= 550,
				ToolSuspendNotifierType						= 560,
				ToolResumeNotifierType						= 570,
				ToolTabletPointerTypeChangedNotifierType	= 580,
				ToolModifiersChangedNotifierType			= 590,
				ToolSuspendedBySpacebarNotifierType			= 600,
				ToolResumedBySpacebarNotifierType			= 610,
				ToolSuspendedByCmdKeyNotifierType			= 620,
				ToolResumedByCmdKeyNotifierType				= 630,
				UserToolChangedNotifierType					= 640,
				EyedropperDragNotifierType					= 650,

				DrawingModeChangedNotifierType					= 660,
				CurrentCoordinateSystemChangedNotifierType		= 670,
				PerspectiveParamsChangedNotifierType			= 680,
				PerspectiveGridVisibilityChangedNotifierType	= 690,

				UpdatePathStyleNotifierType					= 700,
				PaintStyleFillStrokeChangedNotifierType		= 710,
				PaintStyleGradientStopChangedNotifierType	= 720,
				NamedStyleListChangedNotifierType			= 730,
				BeginImportStylesNotifierType				= 740,
				EndImportStylesNotifierType					= 750,

				ColorCalibrationChangedNotifierType	= 760,
				ReplaceColorNotifierType			= 770,
				SwatchLibraryChangedNotifierType	= 780,

				TagChangedNotifierType		= 790,
				XMLNameChangedNotifierType	= 800,

				AllPluginStartedNotifierType			= 810,
				PluginStoppingNotifierType				= 820,
				CSXSPlugPlugSetupCompleteNotifierType	= 830,

				LaunchProjectBridgeNotifierType		= 840,
				CurrentLanguageChangedNotifierType	= 850,
				MenuChangedNotifierType				= 860,
				PreferenceChangedNotifierType		= 870,
				UIBrightnessChangedNotifierType		= 880,
				MetadataSyncNotifierType			= 890,

				ActionManagerPlayActionEventDoneNotifierType	= 900,
				AlignmentKeyArtChangedNotifierType				= 910,
				ObjectSetChangedNotifierType					= 920,
				SlicingChangedNotifierType						= 930,
				SVGFilterChangedNotifierType					= 940,
				SVGFilterAddedNotifierType						= 950,
				VectorizeUpdateNotifierType						= 960,
			
				AboutIllustratorCommandPreNotifierType	= 1000,
				AboutIllustratorCommandPostNotifierType	= 1010,

				NewCommandPreNotifierType				= 1020,
				NewCommandPostNotifierType				= 1030,
				OpenCommandPreNotifierType				= 1040,
				OpenCommandPostNotifierType				= 1050,
				SaveCommandPreNotifierType				= 1060,
				SaveCommandPostNotifierType				= 1070,
				SaveAsCommandPreNotifierType			= 1080,
				SaveAsCommandPostNotifierType			= 1090,
				SaveACopyAsCommandPreNotifierType		= 1100,
				SaveACopyAsCommandPostNotifierType		= 1110,
				RevertToSavedCommandPreNotifierType		= 1120,
				RevertToSavedCommandPostNotifierType	= 1130,
				CloseCommandPreNotifierType				= 1140,
				CloseCommandPostNotifierType			= 1150,
				QuitCommandPreNotifierType				= 1160,
				QuitCommandPostNotifierType				= 1170,

				PageSetupCommandPreNotifierType		= 1180,
				PageSetupCommandPostNotifierType	= 1190,
				ExportCommandPreNotifierType		= 1200,
				ExportCommandPostNotifierType		= 1210,
				PrintCommandPreNotifierType			= 1220,
				PrintCommandPostNotifierType		= 1230,

				PreferencesCommandPreNotifierType				= 1240,
				PreferencesCommandPostNotifierType				= 1250,
				GridPreferencesCommandPreNotifierType			= 1260,
				GridPreferencesCommandPostNotifierType			= 1270,
				HyphenationPreferencesCommandPreNotifierType	= 1280,
				HyphenationPreferencesCommandPostNotifierType	= 1290,
				PluginsFolderPreferencesCommandPreNotifierType	= 1300,
				PluginsFolderPreferencesCommandPostNotifierType	= 1310,
				ColorMatchingPreferencesCommandPreNotifierType	= 1320,
				ColorMatchingPreferencesCommandPostNotifierType	= 1330,
				SeparationSetupCommandPreNotifierType			= 1340,
				SeparationSetupCommandPostNotifierType			= 1350,

				UndoCommandPreNotifierType	= 1360,
				UndoCommandPostNotifierType	= 1370,
				RedoCommandPreNotifierType	= 1380,
				RedoCommandPostNotifierType	= 1390,

				CutCommandPreNotifierType					= 1400,
				CutCommandPostNotifierType					= 1410,
				CopyCommandPreNotifierType					= 1420,
				CopyCommandPostNotifierType					= 1430,
				PasteCommandPreNotifierType					= 1440,
				PasteCommandPostNotifierType				= 1450,
				PasteInFrontCommandPreNotifierType			= 1460,
				PasteInFrontCommandPostNotifierType			= 1470,
				PasteInBackCommandPreNotifierType			= 1480,
				PasteInBackCommandPostNotifierType			= 1490,
				PasteInPlaceCommandPreNotifierType			= 1493,
				PasteInPlaceCommandPostNotifierType			= 1494,
				PasteInAllArtboardsCommandPreNotifierType	= 1497,
				PasteInAllArtboardsCommandPostNotifierType	= 1498,
				CutPictureCommandPreNotifierType			= 1500,
				CutPictureCommandPostNotifierType			= 1510,
				CopyPictureCommandPreNotifierType			= 1520,
				CopyPictureCommandPostNotifierType			= 1530,
				DragCopyCommandPostNotifierType				= 1540,

				SelectAllCommandPreNotifierType		= 1550,
				SelectAllCommandPostNotifierType	= 1560,
				DeselectAllCommandPreNotifierType	= 1570,
				DeselectAllCommandPostNotifierType	= 1580,
				ClearCommandPreNotifierType			= 1590,
				ClearCommandPostNotifierType		= 1600,
				SelectWindowCommandPreNotifierType	= 1610,
				SelectWindowCommandPostNotifierType	= 1620,
				SelectFile1CommandPreNotifierType	= 1630,
				SelectFile1CommandPostNotifierType	= 1640,
				SelectFile2CommandPreNotifierType	= 1650,
				SelectFile2CommandPostNotifierType	= 1660,
				SelectFile3CommandPreNotifierType	= 1670,
				SelectFile3CommandPostNotifierType	= 1680,
				SelectFile4CommandPreNotifierType	= 1690,
				SelectFile4CommandPostNotifierType	= 1700,

				MoveObjectsCommandPreNotifierType		= 1710,
				MoveObjectsCommandPostNotifierType		= 1720,
				TransformAgainCommandPreNotifierType	= 1730,
				TransformAgainCommandPostNotifierType	= 1740,
				TransformRotateCommandPreNotifierType	= 1750,
				TransformRotateCommandPostNotifierType	= 1760,
				TransformReflectCommandPreNotifierType	= 1770,
				TransformReflectCommandPostNotifierType	= 1780,
				TransformScaleCommandPreNotifierType	= 1790,
				TransformScaleCommandPostNotifierType	= 1800,
				TransformShearCommandPreNotifierType	= 1810,
				TransformShearCommandPostNotifierType	= 1820,

				SendToFrontCommandPreNotifierType	= 1830,
				SendToFrontCommandPostNotifierType	= 1840,
				SendForwardCommandPreNotifierType	= 1850,
				SendForwardCommandPostNotifierType	= 1860,
				SendBackwardCommandPreNotifierType	= 1870,
				SendBackwardCommandPostNotifierType	= 1880,
				SendToBackCommandPreNotifierType	= 1890,
				SendToBackCommandPostNotifierType	= 1900,

				GroupCommandPreNotifierType		= 1910,
				GroupCommandPostNotifierType	= 1920,
				UngroupCommandPreNotifierType	= 1930,
				UngroupCommandPostNotifierType	= 1940,

				LockCommandPreNotifierType			= 1950,
				LockCommandPostNotifierType			= 1960,
				UnlockAllCommandPreNotifierType		= 1970,
				UnlockAllCommandPostNotifierType	= 1980,
				HideCommandPreNotifierType			= 1990,
				HideCommandPostNotifierType			= 2000,
				UnhideAllCommandPreNotifierType		= 2010,
				UnhideAllCommandPostNotifierType	= 2020,

				AddAnchorPointsCommandPreNotifierType		= 2030,
				AddAnchorPointsCommandPostNotifierType		= 2040,
				RemoveAnchorPointsCommandPreNotifierType	= 2050,
				RemoveAnchorPointsCommandPostNotifierType	= 2060,

				TypeOutlinesCommandPreNotifierType			= 2070,
				TypeOutlinesCommandPostNotifierType			= 2080,
				JoinCommandPreNotifierType					= 2090,
				JoinCommandPostNotifierType					= 2100,
				AverageCommandPreNotifierType				= 2110,
				AverageCommandPostNotifierType				= 2120,
				MakeTextWrapCommandPreNotifierType			= 2130,
				MakeTextWrapCommandPostNotifierType			= 2140,
				ReleaseTextWrapCommandPreNotifierType		= 2150,
				ReleaseTextWrapCommandPostNotifierType		= 2160,
				MakeMaskCommandPreNotifierType				= 2170,
				MakeMaskCommandPostNotifierType				= 2180,
				ReleaseMaskCommandPreNotifierType			= 2190,
				ReleaseMaskCommandPostNotifierType			= 2200,
				MakeCompoundPathCommandPreNotifierType		= 2210,
				MakeCompoundPathCommandPostNotifierType		= 2220,
				ReleaseCompoundPathCommandPreNotifierType	= 2230,
				ReleaseCompoundPathCommandPostNotifierType	= 2240,
				MakeCropMarksCommandPreNotifierType			= 2250,
				MakeCropMarksCommandPostNotifierType		= 2260,
				ReleaseCropMarksCommandPreNotifierType		= 2270,
				ReleaseCropMarksCommandPostNotifierType		= 2280,

				TextFontCommandPreNotifierType					= 2290,
				TextFontCommandPostNotifierType					= 2300,
				TextSizeCommandPreNotifierType					= 2310,
				TextSizeCommandPostNotifierType					= 2320,
				LineSpacingCommandPreNotifierType				= 2330,
				LineSpacingCommandPostNotifierType				= 2340,
				TextAlignmentCommandPreNotifierType				= 2350,
				TextAlignmentCommandPostNotifierType			= 2360,
				TrackingKerningCommandPreNotifierType			= 2370,
				TrackingKerningCommandPostNotifierType			= 2380,
				WordSpacingCommandPreNotifierType				= 2390,
				WordSpacingCommandPostNotifierType				= 2400,
				DiscretionaryHyphenCommandPreNotifierType		= 2410,
				DiscretionaryHyphenCommandPostNotifierType		= 2420,
				LinkTextCommandPreNotifierType					= 2430,
				LinkTextCommandPostNotifierType					= 2440,
				UnlinkTextCommandPreNotifierType				= 2450,
				UnlinkTextCommandPostNotifierType				= 2460,
				TextOrientationCommandPreNotifierType			= 2470,
				TextOrientationCommandPostNotifierType			= 2480,
				GlyphSubstitutionOptionsCommandPreNotifierType	= 2490,
				GlyphSubstitutionOptionsCommandPostNotifierType	= 2500,

				SetGraphStyleCommandPreNotifierType			= 2510,
				SetGraphStyleCommandPostNotifierType		= 2520,
				EditGraphDataCommandPreNotifierType			= 2530,
				EditGraphDataCommandPostNotifierType		= 2540,
				DefineGraphDesignCommandPreNotifierType		= 2550,
				DefineGraphDesignCommandPostNotifierType	= 2560,
				SetBarDesignCommandPreNotifierType			= 2570,
				SetBarDesignCommandPostNotifierType			= 2580,
				SetIconDesignCommandPreNotifierType			= 2590,
				SetIconDesignCommandPostNotifierType		= 2600,

				ShowHideRulersCommandPreNotifierType			= 2610,
				ShowHideRulersCommandPostNotifierType			= 2620,
				ShowHideGuidesCommandPreNotifierType			= 2630,
				ShowHideGuidesCommandPostNotifierType			= 2640,
				LockUnlockGuidesCommandPreNotifierType			= 2650,
				LockUnlockGuidesCommandPostNotifierType			= 2660,
				MakeGuidesCommandPreNotifierType				= 2670,
				MakeGuidesCommandPostNotifierType				= 2680,
				ReleaseGuidesCommandPreNotifierType				= 2690,
				ReleaseGuidesCommandPostNotifierType			= 2700,
				ShowHideGridCommandPreNotifierType				= 2710,
				ShowHideGridCommandPostNotifierType				= 2720,
				EnableDisableGridSnapCommandPreNotifierType		= 2730,
				EnableDisableGridSnapCommandPostNotifierType	= 2740,
				NewViewSnapCommandPreNotifierType				= 2750,
				NewViewSnapCommandPostNotifierType				= 2760,
				EditViewSnapCommandPreNotifierType				= 2770,
				EditViewSnapCommandPostNotifierType				= 2780,

				ArtworkModeCommandPreNotifierType			= 2790,
				ArtworkModeCommandPostNotifierType			= 2800,
				PreviewModeCommandPreNotifierType			= 2810,
				PreviewModeCommandPostNotifierType			= 2820,
				FitHeadlineCommandPreNotifierType			= 2830,
				FitHeadlineCommandPostNotifierType			= 2840,
				ShowHideEdgesCommandPreNotifierType			= 2850,
				ShowHideEdgesCommandPostNotifierType		= 2860,
				ShowHidePageTilingCommandPreNotifierType	= 2870,
				ShowHidePageTilingCommandPostNotifierType	= 2880,
				ShowHideTemplateCommandPreNotifierType		= 2890,
				ShowHideTemplateCommandPostNotifierType		= 2900,

				ZoomInCommandPreNotifierType		= 2910,
				ZoomInCommandPostNotifierType		= 2920,
				ZoomOutCommandPreNotifierType		= 2930,
				ZoomOutCommandPostNotifierType		= 2940,
				ReduceToFitCommandPreNotifierType	= 2950,
				ReduceToFitCommandPostNotifierType	= 2960,
				ActualSizeCommandPreNotifierType	= 2970,
				ActualSizeCommandPostNotifierType	= 2980,

				NewViewCommandPreNotifierType					= 2990,
				NewViewCommandPostNotifierType					= 3000,
				ArrangeWindowsCascadeCommandPreNotifierType		= 3010,
				ArrangeWindowsCascadeCommandPostNotifierType	= 3020,
				ArrangeWindowsVerticalCommandPreNotifierType	= 3030,
				ArrangeWindowsVerticalCommandPostNotifierType	= 3040,
				ArrangeWindowsHorizontalCommandPreNotifierType	= 3050,
				ArrangeWindowsHorizontalCommandPostNotifierType	= 3060,
				ArrangeIconsCommandPreNotifierType				= 3070,
				ArrangeIconsCommandPostNotifierType				= 3080,

				SetNoteCommandPreNotifierType						= 3090,
				SetNoteCommandPostNotifierType						= 3100,
				NewMultipleMasterInstanceCommandPreNotifierType		= 3110,
				NewMultipleMasterInstanceCommandPostNotifierType	= 3120,

				ShowHideArtboardRulersCommandPreNotifierType	= 3130,
				ShowHideArtboardRulersCommandPostNotifierType	= 3140,

				ExpandCommandPreNotifierType		= 3150,
				ExpandCommandPostNotifierType		= 3160,
				RasterizeCommandPreNotifierType		= 3170,
				RasterizeCommandPostNotifierType	= 3180
			};
		#endif
		// HDI_CORE_AIP_MODE
	}
}



#if defined(HDI_CORE_AIP_MODE)
	hdi::core::MatchArtType operator|(const hdi::core::MatchArtType lhs_, const hdi::core::MatchArtType rhs_);
	hdi::core::MatchArtType& operator|=(hdi::core::MatchArtType& lhs__, const hdi::core::MatchArtType rhs_);
	hdi::core::MatchArtType operator^(const hdi::core::MatchArtType lhs_, const hdi::core::MatchArtType rhs_);
	hdi::core::MatchArtType& operator^=(hdi::core::MatchArtType& lhs__, const hdi::core::MatchArtType rhs_);

	hdi::core::ArtAttribute operator|(const hdi::core::ArtAttribute lhs_, const hdi::core::ArtAttribute rhs_);
	hdi::core::ArtAttribute& operator|=(hdi::core::ArtAttribute& lhs__, const hdi::core::ArtAttribute rhs_);
	hdi::core::ArtAttribute operator^(const hdi::core::ArtAttribute lhs_, const hdi::core::ArtAttribute rhs_);
	hdi::core::ArtAttribute& operator^=(hdi::core::ArtAttribute& lhs__, const hdi::core::ArtAttribute rhs_);

	hdi::core::MatchArtAttribute operator|(const hdi::core::MatchArtAttribute lhs_, const hdi::core::MatchArtAttribute rhs_);
	hdi::core::MatchArtAttribute& operator|=(hdi::core::MatchArtAttribute& lhs__, const hdi::core::MatchArtAttribute rhs_);
	hdi::core::MatchArtAttribute operator^(const hdi::core::MatchArtAttribute lhs_, const hdi::core::MatchArtAttribute rhs_);
	hdi::core::MatchArtAttribute& operator^=(hdi::core::MatchArtAttribute& lhs__, const hdi::core::MatchArtAttribute rhs_);

	hdi::core::HitRequest operator|(const hdi::core::HitRequest lhs_, const hdi::core::HitRequest rhs_);
	hdi::core::HitRequest& operator|=(hdi::core::HitRequest& lhs__, const hdi::core::HitRequest rhs_);
	hdi::core::HitRequest operator^(const hdi::core::HitRequest lhs_, const hdi::core::HitRequest rhs_);
	hdi::core::HitRequest& operator^=(hdi::core::HitRequest& lhs__, const hdi::core::HitRequest rhs_);
#endif

hdi::core::ModifierKey operator|(const hdi::core::ModifierKey lhs_, const hdi::core::ModifierKey rhs_);
hdi::core::ModifierKey& operator|=(hdi::core::ModifierKey& lhs__, const hdi::core::ModifierKey rhs_);
hdi::core::ModifierKey operator^(const hdi::core::ModifierKey lhs_, const hdi::core::ModifierKey rhs_);
hdi::core::ModifierKey& operator^=(hdi::core::ModifierKey& lhs__, const hdi::core::ModifierKey rhs_);

hdi::core::MouseButton operator|(const hdi::core::MouseButton lhs_, const hdi::core::MouseButton rhs_);
hdi::core::MouseButton& operator|=(hdi::core::MouseButton& lhs__, const hdi::core::MouseButton rhs_);
hdi::core::MouseButton operator^(const hdi::core::MouseButton lhs_, const hdi::core::MouseButton rhs_);
hdi::core::MouseButton& operator^=(hdi::core::MouseButton& lhs__, const hdi::core::MouseButton rhs_);



// Define this macro if you are including Adobe's SDK headers before the hdi_core lib headers
#if !defined(HDI_CORE_DONT_DECLARE_AI_TYPES) && !defined(HDI_CORE_DONT_DECLARE_PS_TYPES)

	#ifndef __SPTypes__
		typedef int32_t SPErr;
	#endif

	#ifndef __SPBasic__
		struct SPBasicSuite;
	#endif

	#ifndef __SPPlugins__
		typedef struct SPPlugin* SPPluginRef;
	#endif

	#ifndef __SPMessageData__
		struct SPMessageData
		{
			SPErr SPCheck;
			struct SPPlugin* self;
			void* globals;
			struct SPBasicSuite* basic;
		};
	#endif

	#if defined(HDI_CORE_AIP_MODE)
		#ifndef __AIAnnotator__
			typedef struct _t_AIAnnotatorOpaque* AIAnnotatorHandle;
		#endif

		#ifndef __AIAnnotatorDrawer__
			class AIAnnotatorDrawer;
		#endif

		#ifndef __AIEntry__
			typedef struct _AIArray* AIArrayRef;
			typedef struct _t_AIArtStyle* AIArtStyleHandle;
			typedef struct _AIDictionary* AIDictionaryRef;
			typedef struct _AIEntry* AIEntryRef;
			typedef struct _AIUID* AIUIDRef;
		#endif

		#ifndef __AIDocument__
			typedef struct _t_AIDocument* AIDocumentHandle;
		#endif

		#ifndef __AIDocumentView__
			typedef struct _t_AIDocumentViewOpaque* AIDocumentViewHandle;
		#endif

		#ifndef __AIFileFormat__
			typedef struct _t_AIFileFormatOpaque* AIFileFormatHandle;
		#endif

		#ifndef __AIFont__
			typedef struct __AIFontKey* AIFontKey;
			typedef struct _t_AITypefaceKey* AITypefaceKey;
		#endif

		#ifndef __AIMenu__
			typedef struct _t_AIMenuItemOpaque* AIMenuItemHandle;
			typedef struct _t_MenuGroupOpaque* AIMenuGroup;
		#endif

		#ifndef __AINotifier__
			typedef struct _t_AINotifierOpaque* AINotifierHandle;
		#endif

		#ifndef __AIColor__
			typedef void* AIGradientHandle;
			typedef void* AIPatternHandle;
		#endif

		#ifndef __AITimer__
			typedef struct _t_AITimerOpaque* AITimerHandle;
		#endif

		#ifndef __AITool__
			typedef struct _t_AIToolOpaque* AIToolHandle;
		#endif

		#ifndef __AILiveEffect__
			typedef struct _t_AILiveEffectOpaque* AILiveEffectHandle;
			typedef AIDictionaryRef AILiveEffectParameters;
		#endif

		#ifndef __AITypes__
			typedef struct ArtObject* AIArtHandle;
			typedef struct _t_AILayerOpaque* AILayerHandle;
			typedef struct _t_AISafeArtOpaque* AISafeArtHandle;
		#endif

		#ifndef __AICONTROLBAR_H__
			typedef struct _opaque_AIControlBar* AIControlBarRef;
		#endif

		#ifndef __AIPANEL_H__
			typedef struct _opaque_AIPanel* AIPanelRef;
		#endif

		namespace ATE
		{
			class ICharFeatures;
			class ICharInspector;
			class ICharStyle;
			class ICharStyles;
			class IGlyphRun;
			class IParagraph;
			class IParaFeatures;
			class IParaInspector;
			class IParaStyle;
			class IParaStyles;
			class IStory;
			class IStories;
			class ITabStop;
			class ITabStops;
			class ITextFrame;
			class ITextLine;
			class ITextRange;
			class ITextRanges;
			class IWordsIterator;
		}
	#endif
	// HDI_CORE_AIP_MODE

	#if defined(HDI_CORE_PSP_MODE)
		#ifndef __PIGeneral_h__
			typedef struct _PIChannelPortDesc* PIChannelPort;

			struct PixelMemoryDesc
			{
				/**
					\brief	Coordinate of the first byte of the first pixel
				*/
				void* data;
				
				/**
					\brief	Number of bits per row. Should be multiple of the depth member, and is generally a multiple
							of 8, since this reflects bits rather than bytes.
				*/
				int32_t rowBits;
				
				/**
					\brief	Number of bits per column. Should be a multiple of the depth member. If depth=1, then set
							colBits=1.
				*/
				int32_t colBits;
				
				/**
					\brief	Bit offset from the data pointer value
				*/
				int32_t bitOffset;
				
				/**
					\brief	Pixel depth
				*/
				int32_t depth;
			};
		#endif

		#ifndef __PIActions__
			typedef struct _ADsc* PIActionDescriptor;
			typedef struct _ALst* PIActionList;
			typedef struct _ARef* PIActionReference;
			typedef uint32_t DescriptorClassID;
			typedef uint32_t DescriptorEventID;
			typedef uint32_t DescriptorKeyID;
		#endif
		
		#ifndef __PIAcquire__
			typedef struct AcquireRecord* AcquireRecordPtr;
		#endif

		#ifndef __PIExport__
			typedef struct ExportRecord* ExportRecordPtr;
		#endif

		#ifndef __PIFilter__
			typedef struct FilterRecord* FilterRecordPtr;
		#endif

		#ifndef __PIFormat__
			typedef struct FormatRecord* FormatRecordPtr;
		#endif

		#ifndef __PIMeasurement__
			typedef struct MeasurementBaseRecord* MeasurementBaseRecordPtr;
			typedef struct MeasurementRegisterRecord* MeasurementRegisterRecordPtr;
			typedef struct MeasurementPrepareRecord* MeasurementPrepareRecordPtr;
			typedef struct MeasurementRecordRecord* MeasurementRecordRecordPtr;
			typedef struct MeasurementExportRecord* MeasurementExportRecordPtr;
		#endif

		#ifndef __PIPicker__
			typedef struct PIPickerParams* PickerRecordPtr;
		#endif

		#ifndef __PISelection__
			typedef struct PISelectionParams* SelectionRecordPtr;
		#endif

		#ifndef __XMP_hpp__
			template<class tStringObj> class TXMPMeta;
			typedef class TXMPMeta<std::string> SXMPMeta;
		#endif
	#endif
	// HDI_CORE_PSP_MODE

#endif
// !HDI_CORE_DONT_DECLARE_AI_TYPES && !HDI_CORE_DONT_DECLARE_PS_TYPES

typedef int32_t __MainError;

#endif
// __HDI_CORE_TYPES__
