/**
	\file
	\brief		Header file for Illustrator art UID manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_UID__
#define __HDI_CORE_UID__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"
	
namespace hdi
{
	namespace aip
	{
		class UID;
	}

	namespace core
	{
		class Art;

		/**
			\brief	Allows for easy UID acquisition, art access, etc.
			
			\note	Be aware that even though Adobe advertises UIDs as uniquely identifying a piece of art, they can
					change and become invalid at times [e.g. if the user moves a piece of art])
		*/
		class UID
		{
			public:
				/**
					\brief		Normalizes the proposed name into Adobe's special UID naming format
					\author		GW
					\date		05/2024
					
					\param		name_	The name string to normalize
					\returns	The given string but normalized for UID usage
					
					\note		To losslessly undo the normalization, call denormalizeID() on the normalized form
				*/
				static std::string normalizeName(const std::string& name_);
				
				/**
					\brief		Denormalizes the UID name/ID back into a regular/user-readable name
					\author		GW
					\date		05/2024
					
					\param		id_		The UID name/ID string to convert
					\returns	The given string but denormalized for user readability
				*/
				static std::string denormalizeID(const std::string& id_);
				
				/**
					\brief	Creates an empty UID object
					\author	GW
					\date	08/2013
					
					\note	To test if a UID object is empty, call isEmpty() on it
					\note	Empty UID objects do not relate to any art on the Illustrator artboard; they are designed to
							be "receivers" of some other UID object via the overloaded assignment operator. Empty UID
							objects are useless until such time (though it is safe to call any of their methods)
				*/
				UID();
			
				/**
					\brief	UID copy constructor
					\author	GW
					\date	08/2013
					
					\param	u_	UID object to copy values from
				*/
				UID(const UID& u_);
			
				/**
					\brief	Constructs a new UID object from a UID name
					\author	GW
					\date	09/2016
					
					\param	name_	Unique UID name (see name() method in this class)
				*/
				UID(const std::string& name_);
				
				/**
					\brief	Constructs a new UID object from an AIUIDRef
					\author	GW
					\date	08/2015
					
					\param	aiUID_	Raw Illustrator AIUIDRef
				*/
				UID(const AIUIDRef aiUID_);

				/**
					\brief	Destructs a UID object
					\author	GW
					\date	08/2013
				*/
				virtual ~UID();
			
				/**
					\brief		Overloaded assignment operator for UID objects
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the assignment operator; UID object to copy values from
					\returns	The lefthand side of the assignment operator, but with its values updated
				*/
				virtual UID& operator=(const UID& rhs_);
				
				/**
					\brief		Tests whether a given UID object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	UID to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same UID, false otherwise
				*/
				virtual bool operator==(const UID& rhs_) const;
				
				/**
					\brief		Tests whether a given UID object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	UID to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different UID, false otherwise
				*/
				virtual bool operator!=(const UID& rhs_) const;

				/**
					\brief		Gets the UID ref around which the target object is wrapped
					\author		GW
					\date		11/2013
					
					\returns	AIUIDRef for the target object

					\note		The caller should not release the returned AIUIDRef (its reference count is not
								increased before being returned).
					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual AIUIDRef aiUIDRef() const;

				/**
					\brief		Gets whether the target UID object is empty (constructed with the default ctor)
					\author		GW
					\date		08/2013
					
					\returns	true if the target UID object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Gets an Art object for the target UID
					\author		GW
					\date		08/2013

					\returns	The Art object for the target UID object, or NULL for none
				*/
				virtual std::unique_ptr<Art> art() const;

				/**
					\brief		Gets the unique name of the target UID object
					\author		GW
					\date		08/2013
					
					\returns	A string for the name/identifier of the UID object, as UTF-8
				*/
				virtual std::string name() const;
				
				/**
					\brief	Sets the name of the target UID object
					\author	GW
					\date	01/2014
					
					\param	name_	Desired name for the UID object, as UTF-8
					
					\note	You can pass in any name (aside from "") because the internal UID suite will make it unique
							for you. For example, if a piece of art is already named "Hello", then trying to name
							another piece of art "Hello" will simply cause it to be named something similar (e.g.
							"Hello_2_"). If the user has chosen to show art names as their XML ID value (configurable in
							the Illustrator preferences), then the first piece of art will show "Hello" and the second
							will show "Hello_2_". However, if the user changes the setting to show art display names,
							then they will both show "Hello" (but the second is still called "Hello_2_" internally by
							Illustrator). If you are curious how your name_ parameter has been transformed, then just
							call the name() method afterwards.
				*/
				void setName(const std::string& name_);


			private:
				friend aip::UID* __accessImpl(const UID&);
				friend UID __accessCtor(const aip::UID&);

				/**
					\brief	Private implementation object
				*/
				aip::UID* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				UID(const aip::UID&);
		};
		
		typedef std::unique_ptr<UID> UIDUP;
		typedef std::shared_ptr<UID> UIDSP;
		typedef std::weak_ptr<UID> UIDWP;
		
		extern aip::UID* __accessImpl(const UID&);
		extern UID __accessCtor(const aip::UID&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_UID__
