/**
	\file
	\brief		Header file for plugin user interface web views
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_WEBVIEW__
#define __HDI_CORE_WEBVIEW__

#include <vector>

#include "hdicoreMacros.h"

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		class Callback;

		/**
			\brief	WebView widget, for loading a website within a UI window
		*/
		class WebView : public Widget
		{
			public:
				typedef std::vector<std::string> StringVector;
			
				/**
					\brief	Constructs an empty WebView object
					\author	GW
					\date	09/2015
					
					\note	To test if a WebView object is empty, call isEmpty() on it
					\note	Empty WebView objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other WebView object via the overloaded assignment operator. Empty
							WebView objects are useless until such time (though it is safe to call any of their methods).
				*/
				WebView();

				/**
					\brief	Constructs a new WebView object from an existing WebView object (copy constructor)
					\author	GW
					\date	09/2015

					\param	wv_		Existing WebView object
				*/
				WebView(const WebView& wv_);

				/**
					\brief	Constructs a web view with a given frame
					\author	GW
					\date	09/2015

					\param	frame_	Frame of the web view, in 1x resolution coordinates
				*/
				WebView(const Rect& frame_);

				/**
					\brief	WebView destructor
					\author	GW
					\date	09/2015
				*/
				virtual ~WebView();

				/**
					\brief		Allows one WebView object to be assigned from another
					\author		GW
					\date		09/2015

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target WebView object, but with its value updated to match that of rhs_
				*/
				virtual WebView& operator=(const WebView& rhs_);
				
				/**
					\brief		Convenience method to clone a WebView object on the heap
					\author		GW
					\date		09/2015

					\returns	A pointer to the new WebView object
					
					\note		If you subclass WebView, you MUST overload this method yourself! If you don't and/or your
								clone() method does not return an instance of your WebView subclass, you will experience
								"object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned WebView object.
				*/
				virtual WebView* clone() const;
				
				/**
					\brief		Convenience method to duplicate a WebView object, creating a new and identical UI element
								to the target (but not belonging to the same containing window)
					\author		GW
					\date		09/2015
					
					\returns	A pointer to the new WebView object (and new UI element)
					
					\warning	The caller must manage the memory for the returned WebView object.
				*/
				virtual WebView* duplicate() const;

				/**
					\brief		Does nothing (web views do not have a text value)
					\author		GW
					\date		09/2015
					
					\returns	""
				*/
				virtual std::string text() const;

				/**
					\brief	Does nothing (web views do not have a text value)
					\author	GW
					\date	09/2015

					\param	text_	Ignored
				*/
				virtual void setText(const std::string& text_);

				/**
					\brief		Does nothing (web views do not have a tooltip)
					\author		GW
					\date		09/2015
					
					\returns	""
				*/
				virtual std::string tooltip() const;

				/**
					\brief	Does nothing (web views do not have a tooltip)
					\author	GW
					\date	09/2015
					
					\param	tip_	Ignored
				*/
				virtual void setTooltip(const std::string& tip_);

				/**
					\brief		Gets the currently loaded URL
					\author		GW
					\date		11/2016

					\returns	The current page URL, or empty string for none
				*/
				virtual std::string currentURL() const;
				
				/**
					\brief	Loads the given HTTP(S) URL
					\author	GW
					\date	09/2015
					
					\param	url_	Target URL to load
				*/
				virtual void loadURL(const std::string& url_);
				
				/**
					\brief	Stops any page loading that might be occurring
					\author	GW
					\date	09/2015
				*/
				virtual void stopLoading();
				
				/**
					\brief	Reloads the currently loaded page, if any
					\author	GW
					\date	09/2015
				*/
				virtual void reload();
				
				/**
					\brief		Goes back one step in the browsing history
					\author		GW
					\date		09/2015
					
					\returns	true if a previous page was available, and whose loading was begun
				*/
				virtual bool goBack();
				
				/**
					\brief		Goes forward one step in the browsing history
					\author		GW
					\date		09/2015
					
					\returns	true if a subsequent page was available, and whose loading was begun
				*/
				virtual bool goForward();

				/**
					\brief		Gets whether the target web view is currently loading a page
					\author		GW
					\date		09/2015
					
					\returns	true if still loading, false otherwise
				*/
				virtual bool isLoading() const;
				
				/**
					\brief		Calls the provided JavaScript function, calling the callback when it has finished
					\author		GW
					\date		05/2018

					\param		func_		Function to call (i.e. "myFunc()")
					\param		args_		Arguments to provide to func_
					\param		callback_	Callback to execute when the JS call is finished
				 
					\note		To acquire any return value from calling the given JS function, call the
								javaScriptFunctionResult() method from within your callback.
					\note		To acquire any error state from calling the given JS function, call the
								javaScriptFunctionError() method from within your callback.
				*/
				virtual void callJavaScriptFunction(
					const std::string& func_,
					const StringVector& args_,
					const Callback& callback_
				);

				/**
					\brief		Returns the result from a prior JavaScript function call via the callJavaScriptFunction()
								method
					\author		GW
					\date		11/2020

					\returns	The result as a string, or "" for error
				 
					\warning	The result of this function is only valid when called from within the callback provided
								to the callJavaScriptFunction() method.
				*/
				virtual std::string javaScriptFunctionResult();

				/**
					\brief		Returns the error state, if any, from a prior JavaScript function call via the
								callJavaScriptFunction() method
					\author		GW
					\date		11/2020

					\param		error_	A return-by-reference for the error string, if any
					\returns	true if an error occurred, false otherwise
				 
					\warning	The result of this function is only valid when called from within the callback provided
								to the callJavaScriptFunction() method.
				*/
				virtual bool javaScriptFunctionError(std::string& error_);
				
				#if defined(MAC_ENV)
					/**
						\brief		Gets an estimate of the current loading progress
						\author		GW
						\date		09/2015
					
						\returns	The loading progress, in range [0.0, 1.0]
					*/
					virtual double estimatedProgress() const;
				#endif
		};
		
		typedef std::unique_ptr<WebView> WebViewUP;
		typedef std::shared_ptr<WebView> WebViewSP;
		typedef std::weak_ptr<WebView> WebViewWP;
	}
}

#endif
// __HDI_CORE_WEBVIEW__
