/**
	\file
	\brief		Header file for plugin user interface widgets
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_WIDGET__
#define __HDI_CORE_WIDGET__

#include "hdicoreImageIDs.h"
#include "hdicoreRect.h"
#include "hdicoreSVGIDs.h"

namespace hdi
{
	namespace pui
	{
		class Widget;
	}
	
	namespace core
	{
		class Callback;
		class Window;

		/**
			\brief	Base class for all interface widgets; handles many common needs, e.g. text, position, size, etc.
		*/
		class Widget
		{
			public:
				/**
					\brief	Supported widget types
				*/
				enum Type
				{
					UnknownType					= 0,
					BoxType						= 10,	// Box class
					ButtonType					= 20,	// Button class
					CheckboxType				= 30,	// Checkbox class
					ColumnViewType				= 40,	// ColumnView class
					ComboBoxType				= 50,	// ComboBox class
					ColorWellType				= 60,	// ColorWell class
					CustomType					= 70,	// CustomWidget class
					DividerType					= 80,	// Divider class
					HierarchicalColumnViewType	= 85,	// HierarchicalColumnView class
					HierarchicalViewType		= 90,	// HierarchicalView class
					LabelType					= 100,	// Label class
					ListViewType				= 110,	// ListView class
					PopupMenuType				= 120,	// PopupMenu class
					RadioButtonType				= 130,	// RadioButton class
					SliderType					= 140,	// Slider class
					StepperType					= 150,	// Stepper class
					TextFieldType				= 160,	// TextField class
					TextViewType				= 170,	// TextView class
					WebViewType					= 180	// WebView class
				};

				/**
					\brief	Destructs a Widget
					\author	GW
					\date	09/2013
				*/
				virtual ~Widget();

				/**
					\brief		Gets whether the given platform widget is a hdi_core widget
					\author		GW
					\date		01/2017
					
					\param		widget_		Platform widget pointer to inspect
					\returns	true if the widget is one of "ours" (i.e. is encapsulated by the hdi_core UI lib), false
								otherwise
				*/
				static bool isCoreWidget(const PlatformWidgetPtr widget_);
				
				/**
					\brief		Convenience method to clone a Widget object in memory, taking its subclass type into
								account
					\author		GW
					\date		10/2013

					\returns	A pointer to the new Widget object
					
					\warning	The caller must manage the memory for the returned Widget object.
				*/
				virtual Widget* clone() const = 0;
				
				/**
					\brief		Convenience method to duplicate a Widget object, creating a new and identical UI element
								to the target (but not belonging to the same containing window)
					\author		GW
					\date		11/2013
					
					\returns	A pointer to the new Widget object (and new UI element)
					
					\warning	The caller must manage the memory for the returned Widget object.
				*/
				virtual Widget* duplicate() const = 0;

				/**
					\brief		Gets the platform-specific widget, around which the target object is wrapped
					\author		GW
					\date		11/2013

					\returns	Pointer to platform-specific widget. See PlatformWidgetPtr typedef.

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of UI functionality is not handled by this class (or related
								classes), then it should probably be added to the hdi_core library.
				*/
				virtual PlatformWidgetPtr platformWidget() const;

				/**
					\brief		Gets whether the target Widget object is empty (constructed with the default ctor)
					\author		GW
					\date		09/2013
					
					\returns	true if the target Widget object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Tests whether a given Widget object is the same as another
					\author		GW
					\date		09/2013

					\param		rhs_	Widget to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same UI widget, false otherwise
				*/
				virtual bool operator==(const Widget& rhs_) const;

				/**
					\brief		Tests whether a given Widget object is not the same as another
					\author		GW
					\date		09/2013

					\param		rhs_	Widget to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different UI widgets, false otherwise
				*/
				virtual bool operator!=(const Widget& rhs_) const;

				/**
					\brief		Gets the type of the widget
					\author		GW
					\date		09/2013
					
					\returns	The widget type
				*/
				virtual Type type() const;

				/**
					\brief		Gets the parent window of the widget
					\author		GW
					\date		09/2013

					\returns	Parent window object pointer (NULL if the widget hasn't been added to one yet)
				*/
				virtual Window* const window() const;

				/**
					\brief		Gets the widget frame
					\author		GW
					\date		09/2013

					\returns	Rect describing the top-left (origin) point of the widget and its width & height, in 1x
								resolution coordinates
				*/
				virtual Rect frame() const;

				/**
					\brief	Sets the widget frame
					\author	GW
					\date	09/2013

					\param	frame_	New frame for the widget, in 1x resolution coordinates
				*/
				virtual void setFrame(const Rect& frame_);
				
				/**
					\brief	Offsets the widget from its current location by the given amounts, in 1x resolution coordinates
					\author	GW
					\date	09/2013
					
					\param	x_	Distance to move the widget along the x-axis
					\param	y_	Distance to move the widget along the y-axis
				*/
				virtual void offset(const double x_, const double y_);

				/**
					\brief		Gets the current text value of the widget
					\author		GW
					\date		09/2013
					
					\returns	The current text value as UTF-8, or "" if not supported (e.g. sliders, image labels, etc.)
				*/
				virtual std::string text() const;

				/**
					\brief	Sets the current text value of the widget
					\author	GW
					\date	09/2013

					\param	text_	New text value, as UTF-8 (ignored if not supported)

					\note	If the widget does not support text (e.g. sliders, picture labels, etc.), this method will
							bail early
				*/
				virtual void setText(const std::string& text_);

				/**
					\brief		Gets whether the widget is enabled
					\author		GW
					\date		09/2013

					\returns	true if enabled, false otherwise
				*/
				virtual bool enabled();

				/**
					\brief	Enables or disables the widget
					\author	GW
					\date	09/2013

					\param	enable_		true to enable, false to disable
				*/
				virtual void setEnabled(const bool enable_);

				/**
					\brief		Gets whether the widget is visible
					\author		GW
					\date		08/2014

					\returns	true if visible, false otherwise
				*/
				virtual bool visible();

				/**
					\brief	Shows or hides the widget
					\author	GW
					\date	08/2014

					\param	visible_	true to show, false to hide
				*/
				virtual void setVisible(const bool visible_);

				/**
					\brief	Convenience method to set just the origin of the target widget's frame
					\author	GW
					\date	09/2013

					\param	p_	The new location of the widget, in 1x resolution coordinates
				*/
				virtual void setOrigin(const Point& p_);

				/**
					\brief	Convenience method to set just the size of the target widget's frame
					\author	GW
					\date	09/2013

					\param	s_	The new size of the widget, in 1x resolution coordinates
				*/
				virtual void setSize(const Size& s_);

				/**
					\brief	Convenience method to set just the width of the target widget's frame
					\author	GW
					\date	09/2013

					\param	w_	The width of the widget, at 1x resolution
				*/
				virtual void setWidth(const double w_);

				/**
					\brief	Convenience method to set just the height of the target widget's frame
					\author	GW
					\date	09/2013

					\param	h_	The height of the widget, at 1x resolution 
				*/
				virtual void setHeight(const double h_);

				/**
					\brief		Gets whether the widget currently has focus
					\author		GW
					\date		09/2013
					
					\returns	true for the widget being focused, false otherwise
				*/
				virtual bool hasFocus() const;

				/**
					\brief	Forces the widget to have focus, taking it from any other focused widget of the same window
					\author	GW
					\date	09/2013
				*/
				virtual void focus() const;

				/**
					\brief		Gets the current focus callback for the widget
					\author		GW
					\date		05/2023

					\returns	A pointer to the currently registered focus callback
				*/
				virtual Callback* const focusCallback() const;
				
				/**
					\brief	Sets the focus callback for the widget
					\author	GW
					\date	05/2023

					\param	callback_	New callback for when the widget is focused
					
					\note	The following widgets do not support focus callbacks: Box, Divider, Label, and WebView.
				*/
				virtual void setFocusCallback(const Callback& callback_);

				/**
					\brief	Removes focus from the widget
					\author	GW
					\date	09/2013
				*/
				virtual void blur() const;

				/**
					\brief		Gets the current blur callback for the widget
					\author		GW
					\date		05/2023

					\returns	A pointer to the currently registered blur callback
				*/
				virtual Callback* const blurCallback() const;
				
				/**
					\brief	Sets the blur callback for the widget
					\author	GW
					\date	05/2023

					\param	callback_	New callback for when the widget is blured
					
					\note	The following widgets do not support blur callbacks: Box, Divider, Label, and WebView.
				*/
				virtual void setBlurCallback(const Callback& callback_);

				/**
					\brief		Gets the tool tip string for the widget
					\author		GW
					\date		12/2013
					
					\returns	The tool tip string as UTF-8, or "" if none
				*/
				virtual std::string tooltip() const;

				/**
					\brief	Sets the tool tip string for the widget
					\author	GW
					\date	12/2013
					
					\param	tip_	New widget tool tip string, as UTF-8
				*/
				virtual void setTooltip(const std::string& tip_);
				
				/**
					\brief	Force the widget to update (i.e. cause it to redraw)
					\author	GW
					\date	09/2013
				*/
				virtual void update();

				/**
					\brief		Gets the last widget that received a click event (either by the user or programmatically)
					\author		GW
					\date		09/2013

					\returns	Pointer to last widget clicked
				*/
				static std::unique_ptr<Widget> lastClickedWidget();

				/**
					\brief		Gets the last widget that received a value changing event (either by the user or
								programmatically)
					\author		GW
					\date		09/2013

					\returns	Pointer to last widget whose value was changing
				*/
				static std::unique_ptr<Widget> lastValueChangingWidget();

				/**
					\brief		Gets the last widget that received a value changed event (either by the user or
								programmatically)
					\author		GW
					\date		09/2013

					\returns	Pointer to last widget whose value changed
				*/
				static std::unique_ptr<Widget> lastValueChangedWidget();

				/**
					\brief		Gets the last widget that received a drag-and-drop drop event (either by the user or
								programmatically)
					\author		GW
					\date		09/2013

					\returns	Pointer to last widget whose value changed
				*/
				static std::unique_ptr<Widget> lastDropWidget();

				/**
					\brief		Gets the last widget that received a potential-reorder event
					\author		GW
					\date		12/2022

					\returns	Pointer to last widget to receive a potential-reorder
				*/
				static std::unique_ptr<Widget> lastPotentialReorderWidget();

				/**
					\brief		Gets the last widget that received a reorder event
					\author		GW
					\date		12/2022

					\returns	Pointer to last widget to receive a reorder
				*/
				static std::unique_ptr<Widget> lastReorderWidget();
			
				#if defined(WIN_ENV)
					/**
						\brief		Gets whether widgets should use an ideographically appropriate font size for widget
									text on Windows (whether or not the app locale dictates one should be used)
						\author		GW
						\date		05/2014

						\param		larger_		Return-by-reference for whether the larger font size should be used
												(true for larger, false for smaller)
						\returns	true if the default behavior of automatically choosing the best font size, based on
									locale, should be overridden

						\note		Unavailable on Mac because this sort of thing is handled automatically in Cocoa.
						\note		Widgets will use a larger font size appropriate for ideographic text on their own
									if the app's locale is Locale_JPN_JP, Locale_ZHO_CN, or Locale_ZHO_TW. However,
									this function and the related setForceIdeographicFontSize() merely allow one to
									override this functionality and dictate the size.
					*/
					static bool forceIdeographicFontSize(bool& larger__);
					
					/**
						\brief	Sets whether widgets should use an ideographically appropriate font size for widget
								text on Windows (whether or not the app locale dictates one should be used)
						\author	GW
						\date	05/2014

						\param	force_		true if the default behavior of automatically choosing the best font size,
											based on locale, should be overridden
						\param	larger_		true if a larger font size should be used, false for smaller
						
						\note	Unavailable on Mac because this sort of thing is handled automatically in Cocoa.
						\note	Widgets will use a larger font size appropriate for ideographic text on their own if
								the app's locale is Locale_JPN_JP, Locale_ZHO_CN, or Locale_ZHO_TW. However, this
								function and the related forceIdeographicFontSize() merely allow one to override this
								functionality and dictate the size.
						\note	As this affects the construction of widget controls, it is highly recommended that this
								function be called during plugin startup if you wish to force a larger font size for
								all widgets.
					*/
					static void setForceIdeographicFontSize(const bool force_, const bool larger_);
				#endif


			protected:
				friend class ModalDialog;
				friend class Window;
				friend pui::Widget* __accessImpl(const Widget&);
				
				/**
					\brief	Private implementation data
				*/
				void* _data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	09/2013
				*/
				void* _impl() const;
			
				/**
					\brief	Constructs an empty Widget object
					\author	GW
					\date	09/2013
					
					\note	To test if a Widget object is empty, call isEmpty() on it
					\note	Empty Widget objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other Widget object via the overloaded assignment operator. Empty Widget
							objects are useless until such time (though it is safe to call any of their methods).
				*/
				Widget();
			
				/**
					\brief	Constructs a new Widget object from an existing Widget object (copy constructor)
					\author	GW
					\date	09/2013

					\param	w_	Existing Widget object
					
					\note	This constructor should not be called directly; subclasses should "overload" it and call
							another ctor.
				*/
				Widget(const Widget& w_);

				/**
					\brief	Internal use only
					\author	GW
					\date	09/2013
				*/
				Widget(pui::Widget*&);
				
				/**
					\brief		Internal use only
					\author		GW
					\date		12/2014

					\returns	The target Widget object, but with its values updated to match that of the param
				*/
				virtual Widget& operator=(const Widget&);
		};
		
		typedef std::unique_ptr<Widget> WidgetUP;
		typedef std::shared_ptr<Widget> WidgetSP;
		typedef std::weak_ptr<Widget> WidgetWP;
		
		extern pui::Widget* __accessImpl(const Widget&);
	}
}

#endif
// __HDI_CORE_WIDGET__
