/**
	\file
	\brief		Header file for grouping plugin widgets, such that the geometry of all contained widgets can be modified
				more easily
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_WIDGET_GROUP__
#define __HDI_CORE_WIDGET_GROUP__

#include <vector>

#include "hdicorePoint.h"

namespace hdi
{
	namespace core
	{
		class Widget;

		/**
			\brief	Allows for grouping widgets together, such that all can be added to a window in one call, removed
					from a window in one call, have their geometry adjusted collectively in one call, etc.
		*/
		class WidgetGroup
		{
			public:
				typedef std::vector< std::shared_ptr<Widget> > WidgetVector;
			
				/**
					\brief	Constructs a new widget group
					\author	GW
					\date	02/2015
				*/
				WidgetGroup();
				
				/**
					\brief	Constructs a new WidgetGroup object from an existing WidgetGroup object (copy constructor)
					\author	GW
					\date	02/2015

					\param	group_	Existing WidgetGroup object
				*/
				WidgetGroup(const WidgetGroup& group_);
				
				/**
					\brief	WidgetGroup destructor
					\author	GW
					\date	02/2015
				*/
				virtual ~WidgetGroup();

				/**
					\brief		Allows one WidgetGroup object to be assigned from another
					\author		GW
					\date		02/2015

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target WidgetGroup object, but with its value updated to match that of rhs_
				*/
				virtual WidgetGroup& operator=(const WidgetGroup& rhs_);

				/**
					\brief		Tests whether a given WidgetGroup object is the same as another
					\author		GW
					\date		02/2015

					\param		rhs_	WidgetGroup to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same UI widget group, false otherwise
				*/
				virtual bool operator==(const WidgetGroup& rhs_) const;

				/**
					\brief		Tests whether a given WidgetGroup object is not the same as another
					\author		GW
					\date		02/2015

					\param		rhs_	WidgetGroup to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different UI widget groups, false otherwise
				*/
				virtual bool operator!=(const WidgetGroup& rhs_) const;
				
				/**
					\brief	Adds a widget to the group
					\author	GW
					\date	02/2015

					\param	w_	Widget to add

					\note	If the widget is already in a group, this method will bail early.
					\note	If the target WidgetGroup has already been added to a Window object, then the widget will
							automatically be added to the same Window object.
				*/
				virtual void add(const Widget& w_);
				
				/**
					\brief	Removes a widget from the group
					\author	GW
					\date	02/2015

					\param	w_	Widget to remove

					\note	If the widget isn't in the group, this method will bail early.
					\note	If the target WidgetGroup has already been added to a Window object, then the widget will
							automatically be removed from the same Window object.
				*/
				virtual void remove(const Widget& w_);
			
				/**
					\brief	Removes all widgets currently contained by the group
					\author	GW
					\date	12/2015
				
					\note	If the target WidgetGroup has already been added to a Window object, then the widgets will
							automatically be removed from the same Window object.
				*/
				virtual void clear();

				/**
					\brief		Checks for the existence of a widget in the group
					\author		GW
					\date		02/2015

					\param		w_	Widget in question
					\returns	true if the widget is contained in the group, false otherwise
				*/
				virtual bool hasWidget(const Widget& w_);
				
				/**
					\brief		Gets a vector of the contained widgets
					\author		GW
					\date		02/2015

					\returns	A vector containing all Widget objects in the target group
				*/
				virtual const WidgetVector widgets() const;
				
				/**
					\brief		Gets the top-left-most point for the contained widgets
					\author		GW
					\date		02/2015
					
					\returns	The top-left of all the contained widgets, or (0,0) if none are in the group
				*/
				virtual Point origin() const;
				
				/**
					\brief	Sets the top-left-most point for the contained widgets
					\author	GW
					\date	02/2015
					
					\param	pt_		New top-left point for the top-left-most contained widgets; all other contained
									widgets will have their origins adjusted by the same delta
				*/
				virtual void setOrigin(const Point& pt_);
				
				/**
					\brief		Gets the total width of all the contained widgets
					\author		GW
					\date		02/2015
					
					\returns	The width from the leftmost widget to the rightmost widget
				*/
				virtual double width() const;
				
				/**
					\brief		Gets the total height of all the contained widgets
					\author		GW
					\date		02/2015
					
					\returns	The height from the topmost widget to the bottommost widget
				*/
				virtual double height() const;

				/**
					\brief	Enables or disables all the contained widgets
					\author	GW
					\date	05/2017

					\param	enable_		true to enable, false to disable
				*/
				virtual void setEnabled(const bool enable_);

				/**
					\brief	Shows or hides all the contained widgets
					\author	GW
					\date	05/2017

					\param	visible_	true to show, false to hide
				*/
				virtual void setVisible(const bool visible_);
				
				/**
					\brief	Force all the contained widgets to update (i.e. cause them to redraw)
					\author	GW
					\date	05/2017
				*/
				virtual void update();


			private:
				/**
					\brief	Private implementation data
				*/
				void* __data;
		};
		
		typedef std::unique_ptr<WidgetGroup> WidgetGroupUP;
		typedef std::shared_ptr<WidgetGroup> WidgetGroupSP;
		typedef std::weak_ptr<WidgetGroup> WidgetGroupWP;
	}
}

#endif
// __HDI_CORE_WIDGET_GROUP__
