#include "toolPlugin.h"

#include <algorithm>
#include <math.h>

#include "hdicoreAlerts.h"
#include "hdicoreAngle.h"
#include "hdicoreArt.h"
#include "hdicoreArtboardRect.h"
#include "hdicoreArtboardSegPoint.h"
#include "hdicoreCallback.h"
#include "hdicoreCurrentDocument.h"
#include "hdicoreDraw.h"
#include "hdicoreHitData.h"
#include "hdicoreIllustrator.h"
#include "hdicoreLabel.h"
#include "hdicoreLayer.h"
#include "hdicoreModalDialog.h"
#include "hdicorePathArt.h"
#include "hdicorePlugin.h"
#include "hdicoreTabOrder.h"

#include "toolFormatting.h"
#include "toolResourceIDs.h"



// MARK: Types

namespace tool
{
	typedef HDI_CORE_CALLBACK_T(tool::Plugin) PluginCB;
}

// MARK: - Tool Sample plugin lifecycle

tool::Plugin* tool::Plugin::__instance = NULL;

// static
tool::Plugin* const tool::Plugin::create(hdi::core::Plugin* const corePlug_)
{
	return (tool::Plugin::__instance = new tool::Plugin(corePlug_));
}

// static
tool::Plugin* const tool::Plugin::instance()
{
	return tool::Plugin::__instance;
}

// static
void tool::Plugin::destroy()
{
	delete tool::Plugin::__instance;
	tool::Plugin::__instance = NULL;
}

tool::Plugin::Plugin(hdi::core::Plugin* const corePlug_) :
	__corePlug(corePlug_),
	__crosshairsCrsr(
		TS_CROSSHAIRS_PNGI_RSRC_ID,
		hdi::core::Cursor::noneImageID,
		hdi::core::Cursor::noneImageID,
		hdi::core::Point(9.0, 9.0)
	)
{
	this->__corePlug->setName("Hot Door Tool Sample Plugin");
}

tool::Plugin::~Plugin()
{
	// This doesn't belong to us, so don't delete it
	this->__corePlug = NULL;
}

void tool::Plugin::startup()
{
	this->__gearTool = hdi::core::Tool(
		hdi::core::Tool(),	// Not in same set as any other tool
		NULL,				// Not in same group as any other tool
		"Sample Gear Tool",
		"Sample Gear Tool - Used to generate simple gears",
		hdi::core::IconResourceTypePNGI,
		TS_GEAR_PNGI_RSRC_ID,
		TS_GEAR_DARK_PNGI_RSRC_ID
	);
	this->__gearTool.setTrackCallback(HDI_CORE_CALLBACK(hdi::core::Cursor, &this->__crosshairsCrsr, enable));
	this->__gearTool.setDragCallback(HDI_CORE_CALLBACK(tool::Plugin, this, __gearToolDragCB));
	this->__gearTool.setMouseUpCallback(HDI_CORE_CALLBACK(tool::Plugin, this, __gearToolMouseUpCB));
}

void tool::Plugin::postStartup()
{
	// Dimensions for the modal
	const double modalWidth = 224.0;
	const double modalHeight = 180.0;
	
	// Margin (inset) for the widgets of the modal
	const double marginWidth = 14.0;
	const double marginHeight = 14.0;
	
	// Vertical spacing between labels
	const double labelSpacingY = 8.0;
	
	// Common text field width
	const double textFieldWidth = 70.0;
	
	// Common callback for most text fields
	const tool::PluginCB updateCB(this, &tool::Plugin::__gearToolUpdateModalWidgetsCB);

	// Create modal dialog
	hdi::core::ModalDialog gearInputMD(
		"Gear Generator",
		hdi::core::Size(modalWidth, modalHeight),
		hdi::core::ModalDialog::ButtonOptions(	// OK button options
			true,	// Enabled
			HDI_CORE_CALLBACK(tool::Plugin, this, __gearToolDrawInputCB)
		),
		hdi::core::ModalDialog::ButtonOptions(	// Cancel button options
			true,	// Enabled
			HDI_CORE_CALLBACK_NONE
		)
	);
	this->__gearTool.setInputDialog(gearInputMD);
	
	// Add gear diameter label and field
	hdi::core::Label gearDiamLabel(
		hdi::core::Point(marginWidth, marginHeight),
		"Gear diameter:",
		hdi::core::ModalDialogWindowType
	);
	gearInputMD.addWidget(gearDiamLabel);

	this->__gearToolGearDiamField = hdi::core::TextField(
		hdi::core::Point(modalWidth - textFieldWidth - marginWidth, gearDiamLabel.frame().topRight().y),
		"",
		textFieldWidth
	);
	this->__gearToolGearDiamField.setEnabled(false);
	gearInputMD.addWidget(this->__gearToolGearDiamField);

	// Add number of gear teeth label and field
	hdi::core::Label gearNumTeethLabel(
		hdi::core::Point(marginWidth, gearDiamLabel.frame().bottomLeft().y + labelSpacingY),
		"Number of teeth:",
		hdi::core::ModalDialogWindowType
	);
	gearInputMD.addWidget(gearNumTeethLabel);
	
	this->__gearToolNumTeethField = hdi::core::TextField(
		hdi::core::Point(modalWidth - textFieldWidth - marginWidth, gearNumTeethLabel.frame().topRight().y),
		"",
		textFieldWidth
	);
	this->__gearToolNumTeethField.setValueChangedCallback(updateCB);
	gearInputMD.addWidget(this->__gearToolNumTeethField);

	// Add gear tooth length label and field
	hdi::core::Label gearToothLengthLabel(
		hdi::core::Point(marginWidth, gearNumTeethLabel.frame().bottomLeft().y + labelSpacingY),
		"Tooth length:",
		hdi::core::ModalDialogWindowType
	);
	gearInputMD.addWidget(gearToothLengthLabel);

	this->__gearToolToothLengthField = hdi::core::TextField(
		hdi::core::Point(modalWidth - textFieldWidth - marginWidth, gearToothLengthLabel.frame().topRight().y),
		"",
		textFieldWidth
	);
	this->__gearToolToothLengthField.setValueChangedCallback(updateCB);
	gearInputMD.addWidget(this->__gearToolToothLengthField);

	// Add gear rim diameter label and field
	hdi::core::Label gearRimDiamLabel(
		hdi::core::Point(marginWidth, gearToothLengthLabel.frame().bottomLeft().y + labelSpacingY),
		"Rim diameter:",
		hdi::core::ModalDialogWindowType
	);
	gearInputMD.addWidget(gearRimDiamLabel);

	this->__gearToolRimDiamField = hdi::core::TextField(
		hdi::core::Point(modalWidth - textFieldWidth - marginWidth, gearRimDiamLabel.frame().topRight().y),
		"",
		textFieldWidth
	);
	this->__gearToolRimDiamField.setValueChangedCallback(updateCB);
	gearInputMD.addWidget(this->__gearToolRimDiamField);

	// Add gear hub diameter label and field
	hdi::core::Label gearHubDiamLabel(
		hdi::core::Point(marginWidth, gearRimDiamLabel.frame().bottomLeft().y + labelSpacingY),
		"Hub diameter:",
		hdi::core::ModalDialogWindowType
	);
	gearInputMD.addWidget(gearHubDiamLabel);

	this->__gearToolHubDiamField = hdi::core::TextField(
		hdi::core::Point(modalWidth - textFieldWidth - marginWidth, gearHubDiamLabel.frame().topRight().y),
		"",
		textFieldWidth
	);
	this->__gearToolHubDiamField.setValueChangedCallback(updateCB);
	gearInputMD.addWidget(this->__gearToolHubDiamField);

	// Add gear bore diameter label and field
	hdi::core::Label gearBoreDiamLabel(
		hdi::core::Point(marginWidth, gearHubDiamLabel.frame().bottomLeft().y + labelSpacingY),
		"Bore diameter:",
		hdi::core::ModalDialogWindowType
	);
	gearInputMD.addWidget(gearBoreDiamLabel);

	this->__gearToolBoreDiamField = hdi::core::TextField(
		hdi::core::Point(modalWidth - textFieldWidth - marginWidth, gearBoreDiamLabel.frame().topRight().y),
		"",
		textFieldWidth
	);
	this->__gearToolBoreDiamField.setValueChangedCallback(updateCB);
	gearInputMD.addWidget(this->__gearToolBoreDiamField);
	
	gearInputMD.tabOrder()->pushBack(this->__gearToolNumTeethField);
	gearInputMD.tabOrder()->pushBack(this->__gearToolToothLengthField);
	gearInputMD.tabOrder()->pushBack(this->__gearToolRimDiamField);
	gearInputMD.tabOrder()->pushBack(this->__gearToolHubDiamField);
	gearInputMD.tabOrder()->pushBack(this->__gearToolBoreDiamField);
}

void tool::Plugin::preShutdown()
{
	// Nothing to do here
}

void tool::Plugin::shutdown()
{
	this->__gearTool.destroy();
}

// MARK: - Gear tool callbacks

void tool::Plugin::__gearToolUpdateModalWidgetsCB()
{
	double numTeeth = 0.0, toothLength = 0.0, rimDiam = 0.0, hubDiam = 0.0, boreDiam = 0.0;
	if(!this->__gearToolVerifyWidgets(true, numTeeth, toothLength, rimDiam, hubDiam, boreDiam))
		return;

	this->__gearToolGearDiamField.setText(tool::formatPoints(rimDiam + (2.0 * toothLength)));
	this->__gearToolToothLengthField.setText(tool::formatPoints(toothLength));
	this->__gearToolRimDiamField.setText(tool::formatPoints(rimDiam));
	this->__gearToolHubDiamField.setText(tool::formatPoints(hubDiam));
	this->__gearToolBoreDiamField.setText(tool::formatPoints(boreDiam));
}

void tool::Plugin::__gearToolDragCB()
{
	std::unique_ptr<hdi::core::HitData> startHit = this->__gearTool.mouseDownHitData(0, hdi::core::AnyHitRequestNoGuides);
	if( !startHit.get() )
		return;

	std::unique_ptr<hdi::core::HitData> endHit = this->__gearTool.cursorHitData(hdi::core::AnyHitRequestNoGuides);
	if( !endHit.get() )
		return;

	double width = endHit->point().x - startHit->point().x, height = endHit->point().y - startHit->point().y;
	width = height = std::max(fabs(width), fabs(height));
	width = (endHit->point().x > startHit->point().x ? width : -width);
	height = (endHit->point().y > startHit->point().y ? height : -height);
	
	this->__gearToolGearDiamField.setText(tool::formatPoints(fabs(width)));
	this->__gearToolRimDiamField.setText(tool::formatPoints(fabs(width)));

	hdi::core::draw::ellipse(startHit->point(), width, height, false /* Not centered */);
}

void tool::Plugin::__gearToolMouseUpCB()
{
	this->__gearToolUpdateModalWidgetsCB();
	
	this->__gearTool.inputDialog()->show();
	
	this->__gearTool.resetCursorLocs();
}

void tool::Plugin::__gearToolDrawInputCB()
{
	double numTeeth = 0.0, toothLength = 0.0, rimDiam = 0.0, hubDiam = 0.0, boreDiam = 0.0;
	if(!this->__gearToolVerifyWidgets(false, numTeeth, toothLength, rimDiam, hubDiam, boreDiam))
		return;

	// Get the hit data for the first click
	std::unique_ptr<hdi::core::HitData> startHit = this->__gearTool.mouseDownHitData(0, hdi::core::AnyHitRequestNoGuides);
	if( !startHit.get() )
		return;

	// Get the snapped point for the first click
	hdi::core::ArtboardPoint startPoint = startHit->point();
	if(this->__gearTool.wasDragged())
	{
		// If the tool was dragged, the user may have gone in any direction (instead of just dragging at an angle in
		// quadrant 4). Adjust the start point accordingly.
		std::unique_ptr<hdi::core::HitData> endHit = this->__gearTool.cursorHitData(hdi::core::AnyHitRequestNoGuides);
		if( !endHit.get() )
			return;

		startPoint = hdi::core::ArtboardPoint(
			std::min(startHit->point().x, endHit->point().x) - toothLength,
			std::min(startHit->point().y, endHit->point().y) - toothLength
		);
	}

	// Construct a bounds rect that will contain the entire gear (for convenience)
	const double gearDiam = rimDiam + (2.0 * toothLength);
	const double rimRadius = rimDiam / 2.0;
	hdi::core::ArtboardRect bounds(
		startPoint,
		hdi::core::ArtboardPoint(startPoint.x + gearDiam, startPoint.y + gearDiam)
	);
	const hdi::core::ArtboardPoint gearCenter = bounds.midCenter();
	
	// Create a group to contain the gear art
	std::unique_ptr<hdi::core::Art> currentLayerGroup = HDI_CORE_ILLUSTRATOR->currentDocument()->currentLayer()->group();
	hdi::core::Art gearGroup(hdi::core::ArtTypeGroup, hdi::core::PlaceInsideOnTop, currentLayerGroup.get());

	// Create a path to contain the gear teeth and rim
	hdi::core::Art gearRimTeeth(hdi::core::ArtTypePath, hdi::core::PlaceInsideOnTop, &gearGroup);
	gearRimTeeth.path()->setClosed(true);
	
	// We'll need these constants several times
	const hdi::core::Angle twoPi = hdi::core::Angle::TwoPi();
	const hdi::core::Angle piOver2 = hdi::core::Angle::Pi() / 2.0;
	
	// Loop over each tooth that should be drawn, adding new segment points to the path iteratively such that a complete
	// gear with teeth will be drawn at the end
	bool drawOutward = true;
	const hdi::core::Angle incAngle = twoPi / (2.0 * numTeeth);
	const double rimKappa = incAngle.asRadians() / piOver2.asRadians() * hdi::core::bezierKappa;
	for(hdi::core::Angle currAngle(0.0); currAngle < twoPi - (incAngle / 2.0); currAngle += incAngle)
	{
		hdi::core::ArtboardPoint rimPt = gearCenter.move(rimRadius, currAngle);

		std::vector<hdi::core::ArtboardSegPoint> newSegPts;
		if(drawOutward)
		{
			hdi::core::Angle apexAngle = currAngle + (incAngle / 2.0);
			hdi::core::ArtboardPoint apexPt = gearCenter.move(rimRadius + toothLength, apexAngle);
		
			// This point is at the rim, before going "up" the side of the tooth
			newSegPts.push_back(hdi::core::ArtboardSegPoint(
				rimPt,	// "p" point
				rimPt.move(rimKappa * rimRadius, currAngle - piOver2),	// "in" point
				rimPt,	// "out" point
				true	// Corner
			));
			
			// This point is half way "up" the side of the tooth, before it starts to curve
			newSegPts.push_back(hdi::core::ArtboardSegPoint(
				gearCenter.move(rimRadius + toothLength / 2.0, currAngle),
				true	// Corner
			));
			
			// This point is the apex of the tooth curve
			newSegPts.push_back(hdi::core::ArtboardSegPoint(
				apexPt,	// "p" point
				apexPt.move(rimKappa * rimRadius, apexAngle - piOver2),	// "in" point
				apexPt.move(rimKappa * rimRadius, apexAngle + piOver2),	// "out" point
				true	// Corner
			));
		}
		else	// Draw inward
		{
			// This point is half way "down" the side of the tooth, after it has curved
			newSegPts.push_back(hdi::core::ArtboardSegPoint(
				gearCenter.move(rimRadius + toothLength / 2.0, currAngle),
				true	// Corner
			));
			
			// This point is at the rim, after going "down" the side of the tooth
			newSegPts.push_back(hdi::core::ArtboardSegPoint(
				rimPt,	// "p" point
				rimPt,	// "in" point
				rimPt.move(rimKappa * rimRadius, currAngle + piOver2),	// "out" point
				true	// Corner
			));
		}
		
		// Always flip this bit so we alternate how we draw
		drawOutward = !drawOutward;

		// Actually set the segment points (at the end of the path)
		gearRimTeeth.path()->setSegPoints(gearRimTeeth.path()->segPointCount(), newSegPts);
	}
	
	// We start drawing at angle 0.0, so rotate the gear a bit so the first tooth peaks at 0.0
	gearRimTeeth.rotate(incAngle / 2.0);

	// The hub and bore are simple circles around the gear's center point
	hdi::core::draw::ellipse(gearCenter, hubDiam, hubDiam, true /* Centered */, &gearGroup);
	hdi::core::draw::ellipse(gearCenter, boreDiam, boreDiam, true /* Centered */, &gearGroup);
}

// MARK: - Helpers

bool tool::Plugin::__gearToolVerifyWidgets(
	const bool emptyAllowed_,
	double& numTeeth__,
	double& toothLength__,
	double& rimDiam__,
	double& hubDiam__,
	double& boreDiam__
)
{
	if(!tool::scanNumber(this->__gearToolNumTeethField.text(), numTeeth__) || numTeeth__ < 0.0)
	{
		if(this->__gearToolNumTeethField.text() != "" || !emptyAllowed_)
		{
			hdi::core::alerts::error("You must enter a valid number of teeth for the gear.");
			return false;
		}
	}

	if(!tool::scanPoints(this->__gearToolToothLengthField.text(), toothLength__) || toothLength__ < 0.0)
	{
		if(this->__gearToolToothLengthField.text() != "" || !emptyAllowed_)
		{
			hdi::core::alerts::error("You must enter a valid tooth length for the gear.");
			return false;
		}
	}

	if(!tool::scanPoints(this->__gearToolRimDiamField.text(), rimDiam__) || rimDiam__ < 0.0)
	{
		if(this->__gearToolRimDiamField.text() != "" || !emptyAllowed_)
		{
			hdi::core::alerts::error("You must enter a valid gear rim diameter.");
			return false;
		}
	}

	if(!tool::scanPoints(this->__gearToolHubDiamField.text(), hubDiam__) || hubDiam__ < 0.0 || hubDiam__ > rimDiam__)
	{
		if(this->__gearToolHubDiamField.text() != "" || !emptyAllowed_)
		{
			hdi::core::alerts::error("You must enter a valid gear hub diameter.");
			return false;
		}
	}

	if(!tool::scanPoints(this->__gearToolBoreDiamField.text(), boreDiam__) || boreDiam__ < 0.0 || boreDiam__ > hubDiam__)
	{
		if(this->__gearToolBoreDiamField.text() != "" || !emptyAllowed_)
		{
			hdi::core::alerts::error("You must enter a valid gear bore diameter.");
			return false;
		}
	}
	
	return true;
}
